/*
 * MMA7660FC Three-Axis Digital Accelerometer
 *
 * Copyright (C) 2009 ?????????????????????????????
 * Licensed under the GPL-2 or later.
 */

#include <linux/device.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/i2c.h>
#include <linux/irq.h>
#include <linux/slab.h>
#include <linux/workqueue.h>

/* MMA7660 Register Map */
#define XOUT_REG	0x00	/* R   6-bit output value X */
#define YOUT_REG	0x01	/* R   6-bit output value Y */
#define ZOUT_REG	0x02	/* R   6-bit output value Z */
#define TILT_REG	0x03	/* R   tilt status */
#define SRST_REG	0x04	/* R   sample rate status */
#define SPCNT_REG	0x05	/* R/W sleep count */
#define INTSU_REG	0x06	/* R/W interrupt setup */
#define MODE_REG	0x07	/* R/W mode */
#define SR_REG		0x08	/* R/W sample rate */
#define PDET_REG	0x09	/* R/W tap/pulse detection */
#define PD_REG		0x0A	/* R/W tap/pulse debounce count */

/* XOUT, YOUT, ZOUT register definitions */
#define XYZ_ALERT	(1 << 6)
#define XYZ_SHIFT	2

/* TILT register bit / bit field definitions */
#define TILT_BAFRO(x)	(x & 0x03)
#define BAFRO_FRONT	0x01
#define BAFRO_BACK	0x02
#define TILT_POLA(x)	((x >> 2) & 0x07)
#if 0
#define POLA_LEFT	0x01
#define POLA_RIGHT	0x02
#define POLA_DOWN	0x05
#define POLA_UP		0x06
#endif
#define POLA_LEFT	0x06	/* Mantis orientation */
#define POLA_RIGHT	0x05	/* Mantis orientation */
#define POLA_DOWN	0x02	/* Mantis orientation */
#define POLA_UP		0x01	/* Mantis orientation */
#define TILT_TAP	(1 << 5)
#define TILT_ALERT	(1 << 6)
#define TILT_SHAKE	(1 << 7)

/* SRST register bit definitions */
#define SRST_AMSRS	(1 << 0)
#define SRST_AWSRS	(1 << 1)

/* INTSU register bit definitions */
#define INTSU_FBINT	(1 << 0)
#define INTSU_PLINT	(1 << 1)
#define INTSU_PDINT	(1 << 2)
#define INTSU_ASINT	(1 << 3)
#define INTSU_GINT	(1 << 4)
#define INTSU_SHINTZ	(1 << 5)
#define INTSU_SHINTY	(1 << 6)
#define INTSU_SHINTX	(1 << 7)

/* MODE register bit definitions */
#define MODE_MODE	(1 << 0)
#define MODE_TON	(1 << 2)
#define MODE_AWE	(1 << 3)
#define MODE_ASE	(1 << 4)
#define MODE_SCPS	(1 << 5)
#define MODE_IPP	(1 << 6)
#define MODE_IAH	(1 << 7)

/* SR register bit / bit field definitions */
#define SR_AMSR(x)	(x & 0x07)
#define AMSR_AMPD	0
#define AMSR_AM64	1
#define AMSR_AM32	2
#define AMSR_AM16	3
#define AMSR_AM8	4
#define AMSR_AM4	5
#define AMSR_AM2	6
#define AMSR_AM1	7
#define SR_AWSR(x)	((x & 0x03) << 3)
#define AWSR_AW32	0
#define AWSR_AW16	1
#define AWSR_AW8	2
#define AWSR_AW1	3
#define SR_FILT(x)	((x & 0x07) << 5)
#define FILT_NODB	0
#define FILT_DB2	1
#define FILT_DB3	2
#define FILT_DB4	3
#define FILT_DB5	4
#define FILT_DB6	5
#define FILT_DB7	6
#define FILT_DB8	7

struct cicada_mma7660 {
	struct i2c_client	*client;
	struct input_dev	*dev;
	struct workqueue_struct *workq;
	struct work_struct	event_work;
	int			irq;
	int			irq_trigger;
	int			opened;
	unsigned int		debug_level;
	int			last_key;
};

struct cicada_mma7660_data {
	s8 x;
	s8 y;
	s8 z;
	u8 tilt;
};

struct i2c_client *cicada_mma7660_client;

/* touch screen irq handler */
static irqreturn_t cicada_mma7660_irq(int irq, void *v)
{
	struct cicada_mma7660 *pdata = v;

	disable_irq_nosync(pdata->irq);
	queue_work(pdata->workq, &pdata->event_work);

	return IRQ_HANDLED;
}


static int cicada_mma7660_read_reg(struct i2c_client *client,
				   u8 command, bool alert)
{
	int ret;
	int i;

	for (i = 0; i < 3; ++i) {
		ret = i2c_smbus_read_byte_data(client, command);
		if ((ret != -1) && (!alert || !(ret & XYZ_ALERT)))
			break;
		else
			dev_dbg(&client->dev, "%s, retry %d\n", __func__, i+1);

		udelay(10);
	}
	if ((ret < 0) || (alert && (ret & XYZ_ALERT))) {
		ret = -1;
		dev_dbg(&client->dev, "%s, failed\n", __func__);
	}

	return ret;
}

static int cicada_mma7660_write_reg(struct i2c_client *client,
				    u8 command, u8 value)
{
	int ret;
	int i;

	for (i = 0; i < 3; ++i) {
		ret = i2c_smbus_write_byte_data(client, command, value);
		if (ret == 0)
			break;
		else
			dev_dbg(&client->dev, "%s, retry %d\n", __func__, i+1);

		udelay(10);
	}
	if (ret < 0)
		dev_dbg(&client->dev, "%s, failed\n", __func__);

	return ret;
}

static int cicada_mma7660_get_data(struct i2c_client *client,
				   struct cicada_mma7660_data *data)
{
	int xout, yout, zout, tilt;

	xout = cicada_mma7660_read_reg(client, XOUT_REG, true);
	yout = cicada_mma7660_read_reg(client, YOUT_REG, true);
	zout = cicada_mma7660_read_reg(client, ZOUT_REG, true);
	tilt = cicada_mma7660_read_reg(client, TILT_REG, true);

	if ((xout < 0) || (yout < 0) || (zout < 0) || (tilt < 0)) {
		dev_err(&client->dev, "%s: read error\n", __func__);
		return -EIO;
	}

	if (data) {
		data->x = (((s8)(xout << XYZ_SHIFT)) >> XYZ_SHIFT);
		data->y = (((s8)(yout << XYZ_SHIFT)) >> XYZ_SHIFT);
		data->z = (((s8)(zout << XYZ_SHIFT)) >> XYZ_SHIFT);
		data->tilt = tilt;
	}

	return 0;
}

static void cicada_mma7660_key_single(struct input_dev *input, int key)
{
	input_report_key(input, key, true);
	input_sync(input);
	input_report_key(input, key, false);
	input_sync(input);
}

static void cicada_mma7660_worker(struct work_struct *work)
{
	struct cicada_mma7660 *pdata =
		container_of(work, struct cicada_mma7660, event_work);
	struct cicada_mma7660_data acdata;
	int key = 0;

	if (cicada_mma7660_get_data(pdata->client, &acdata)) {
		dev_dbg(&pdata->client->dev,
			"%s, cicada_mma7660_get_data failed\n", __func__);
		goto exit;
	}

	if (pdata->debug_level > 0) {
		char buffer[100];
		int i = 0;
		buffer[0] = 0;
		switch TILT_BAFRO(acdata.tilt)
		{
			 case BAFRO_FRONT:
				 i += sprintf(&buffer[i], "front ");
				 break;
			 case BAFRO_BACK:
				 i += sprintf(&buffer[i], "back ");
				 break;
			 default:
				 i += sprintf(&buffer[i], "unknown ");
				 break;
		}

		switch TILT_POLA(acdata.tilt)
		{
			case POLA_LEFT:
				i += sprintf(&buffer[i], "left ");
				break;
			case POLA_RIGHT:
				i += sprintf(&buffer[i], "right ");
				break;
			case POLA_UP:
				i += sprintf(&buffer[i], "up ");
				break;
			case POLA_DOWN:
				i += sprintf(&buffer[i], "down ");
				break;
			default:
				i += sprintf(&buffer[i], "unknown ");
				break;
		}

		if (acdata.tilt & TILT_SHAKE)
			i += sprintf(&buffer[i], "shake ");

		printk(KERN_INFO "%u, (%3d, %3d, %3d) tilt %02x, %s\n",
		       jiffies_to_msecs(jiffies),
		       acdata.x,
		       acdata.y,
		       acdata.z,
		       acdata.tilt,
		       buffer);

	}

	switch TILT_POLA(acdata.tilt)
	{
		case POLA_LEFT:
			key = KEY_LEFT;
			break;
		case POLA_RIGHT:
			key = KEY_RIGHT;
			break;
		case POLA_UP:
			key = KEY_UP;
			break;
		case POLA_DOWN:
			key = KEY_DOWN;
			break;
		default:
			break;
	}

	if (key != pdata->last_key) {
		cicada_mma7660_key_single(pdata->dev, key);
		pdata->last_key = key;
	}

exit:
	enable_irq(pdata->irq);
}

static ssize_t cicada_mma7660_debug_store(struct device *dev,
					  struct device_attribute *attr,
					  const char *buf, size_t n)
{
	unsigned int value;
	struct input_dev *in_dev = container_of(dev, struct input_dev, dev);
	struct cicada_mma7660 *pdata = input_get_drvdata(in_dev);

	if (!pdata)
		return 0;

	if (kstrtouint(buf, 10, &value) != 0) {
		dev_err(dev, "%s, failed to convert value\n", __func__);
		return -EINVAL;
	}

	pdata->debug_level = value;

	return n;
}

DEVICE_ATTR(ca_dbg_en, 0200, NULL, cicada_mma7660_debug_store);

static int cicada_mma7660_dev_open(struct input_dev *dev)
{
	struct cicada_mma7660 *pdata = input_get_drvdata(dev);

/*	dev_dbg(&pdata->client->dev, "%s\n", __func__); */

	if (pdata->opened > 0) {
		pdata->opened++;
		return 0;
	}

	pdata->opened++;

	return 0;
}

static void cicada_mma7660_dev_close(struct input_dev *dev)
{
	struct cicada_mma7660 *pdata = input_get_drvdata(dev);

/*	dev_dbg(&pdata->client->dev, "%s\n", __func__); */

	if (pdata->opened == 0) {
		dev_info(&pdata->client->dev, "unbalanced close\n");
		return;
	}

	pdata->opened--;
}

static int cicada_mma7660_device_register(struct cicada_mma7660 *pdata)
{
	struct input_dev *dev;
	int ret = 0;

/*	dev_dbg(&pdata->client->dev, "%s\n", __func__); */

	dev = input_allocate_device();
	if (dev == NULL)
		return -ENOMEM;

	pdata->dev = dev;
	input_set_drvdata(dev, pdata);

	dev->name = "cicada-mma7660 accelerometer";
	dev->phys = "cicada-mma7660";
	dev->open = cicada_mma7660_dev_open;
	dev->close = cicada_mma7660_dev_close;

	__set_bit(EV_KEY, dev->evbit);
	__set_bit(KEY_UP, dev->keybit);
	__set_bit(KEY_DOWN, dev->keybit);
	__set_bit(KEY_LEFT, dev->keybit);
	__set_bit(KEY_RIGHT, dev->keybit);

	pdata->workq = create_singlethread_workqueue("cicada_mma7660");
	if (pdata->workq == NULL) {
		dev_err(&dev->dev, "Failed to create workqueue\n");
		ret = -ENOMEM;
		goto error_1;
	}

	INIT_WORK(&pdata->event_work, cicada_mma7660_worker);

	ret = input_register_device(dev);
	if (ret) {
		dev_err(&dev->dev, "input_register_device failed\n");
		goto error_1;
	}

	ret = request_irq(pdata->irq, cicada_mma7660_irq,
			  pdata->irq_trigger, "cicada-mma7660", pdata);
	if (ret) {
		dev_err(&dev->dev, "request_irq %d failed\n", pdata->irq);
		goto error_2;
	}

	dev_info(&dev->dev, "Device Registered\n");
	return 0;

error_2:
	input_unregister_device(dev);
error_1:
	input_free_device(dev);

	return ret;
}

static int cicada_mma7660_detect(struct i2c_client *client, int kind,
				 struct i2c_board_info *info)
{
	struct i2c_adapter *adapter = client->adapter;
	int i = 0;

/*	dev_dbg(&client->dev, "%s\n", __func__); */

	if (!i2c_check_functionality(adapter, I2C_FUNC_I2C))
		goto error;

	while (1) {
		if (!cicada_mma7660_get_data(client, NULL))
			break;
		if (++i >= 3)
			goto error;
	}

	if (info)
		strlcpy(info->type, "cicada-mma7660", I2C_NAME_SIZE);

	return 0;

error:
	dev_dbg(&client->dev, "%s, failed\n", __func__);
	return -ENODEV;
}

static int cicada_mma7660_probe(struct i2c_client *client,
				const struct i2c_device_id *id)
{
	struct cicada_mma7660 *pdata;
	struct device *dev = &client->dev;
	struct device_node *np = dev->of_node;
	enum of_gpio_flags flags;
	int err = 0;

	dev_dbg(dev, "%s\n", __func__);

	if (cicada_mma7660_client) {
		dev_err(dev, "%s, already allocated\n", __func__);
		return -ENODEV;
	}

	err = cicada_mma7660_detect(client, 0, NULL);
	if (err)
		goto error;

	pdata = kzalloc(sizeof(struct cicada_mma7660), GFP_KERNEL);
	if (!pdata)
		return -ENOMEM;

	pdata->irq = of_get_named_gpio_flags(np, "irq-gpios", 0, &flags);
	pdata->irq_trigger = (flags & OF_GPIO_ACTIVE_LOW) ? IRQF_TRIGGER_LOW : IRQF_TRIGGER_HIGH;
	if (!gpio_is_valid(pdata->irq)) {
		dev_err(dev, "failed to get irq\n");
		return -EINVAL;
	}
	retval = devm_gpio_request_one(dev, pdata->irq,
			GPIOF_IN, "irq-gpios");
	if (retval) {
		dev_err(dev, "failed to request gpio irq-gpios\n");
		return retval;
	}

	pdata->client = client;
	pdata->last_key = 999999;
	pdata->irq = gpio_to_irq(pdata->gpio_irq);

	i2c_set_clientdata(client, pdata);

	cicada_mma7660_client = client;

	cicada_mma7660_write_reg(client, MODE_REG, 0);
	cicada_mma7660_write_reg(client, SR_REG,
				 SR_FILT(FILT_DB4) | SR_AWSR(AWSR_AW8) |
				 SR_AMSR(AMSR_AM8));
	cicada_mma7660_write_reg(client, INTSU_REG,
				 INTSU_FBINT | INTSU_PLINT |
				 INTSU_SHINTX | INTSU_SHINTY | INTSU_SHINTZ);
	cicada_mma7660_write_reg(client, MODE_REG,
				 MODE_MODE | MODE_AWE | MODE_ASE);

	err = device_create_file(dev, &dev_attr_ca_dbg_en);
	if (err)
		goto error_4;

	err = cicada_mma7660_device_register(pdata);
	if (err < 0)
		goto error_5;

	dev_info(dev, "probed, chip @ 0x%x (%s)\n",
		 client->addr, client->adapter->name);

	return 0;

error_5:
	device_remove_file(dev, &dev_attr_ca_dbg_en);
error_4:
	kfree(pdata);
error:
	dev_err(dev, "probe of chip @ 0x%x (%s) failed\n",
		client->addr, client->adapter->name);

	return err;
}

static int cicada_mma7660_remove(struct i2c_client *client)
{
	struct cicada_mma7660 *pdata = i2c_get_clientdata(client);

/*	dev_dbg(&client->dev, "%s:\n", __func__); */

	free_irq(pdata->irq, pdata);
	destroy_workqueue(pdata->workq);
	device_remove_file(&client->dev, &dev_attr_ca_dbg_en);
	input_unregister_device(pdata->dev);
	kfree(pdata->dev);
	kfree(pdata);

	return 0;
}

static const struct i2c_device_id cicada_i2c_device_ids[] = {
	{ "cicada-mma7660", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, cicada_i2c_device_ids);

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-mma7660" },
	{},
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

struct i2c_driver cicada_mma7660_driver = {
	.driver = {
		.name = "cicada-mma7660",
		.owner = THIS_MODULE,
		.of_match_table = cicada_device_ids,
	},
	.probe = cicada_mma7660_probe,
	.remove = cicada_mma7660_remove,
	.id_table = cicada_i2c_device_ids,
};

module_i2c_driver(cicada_ti_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("Cicada - MMA7660 accelerometer driver");

/* Use 8 spaces for indentation and tabs.
   -*- mode: linux-c; c-basic-offset: 8; indent-tabs-mode: true -*-
*/
