// Program for abuse testing of the cicada_meas driver.
//
// Parts of the meascop program and support utilities are duplicated
// here.  I wanted this program to be minimal and self contained.  It
// is very difficult to do this type of testing with a full blown
// measurement system.

#include <string>
#include <list>
using namespace std;

#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <sys/time.h>


//===========================================================================
//==== Configuration Switches ===============================================
//===========================================================================

//#define TRFLOW_ENABLE
#ifdef  TRFLOW_ENABLE
#define	TRFLOW(fmt, args...) printf(fmt, ## args)
#else
#define	TRFLOW(fmt, args...) ;
#endif


//#define NO_IO
#ifdef  NO_IO
#warning "NO_IO enabled (I/O to dev nodes disabled)"
#endif

//#define DUMP_OUTDATA
//#define DUMP_INDATA

const char* cmdDev = "/dev/cicada_meas_cmd0";
const char* evtDev = "/dev/cicada_meas_evt0";

static const int COUNT_UPDATE_ITERATIONS = 1000;

// If this is true, it outputs slabinfo stats every iteration
static const bool SLABINFO_ALWAYS = true;

// if SLABINFO_ALWAYS is false, it generates output every this many
// iterations
static const int SLABINFO_ITERATIONS = 1000;


#define SLABINFO_FIELD 0
//#define SLABINFO_FIELD "cred_jar"


#define OUTPUT_SLABINFO_TOTALS

static const bool ABS_COMPARISONS = true;


//===========================================================================
//==== Utilities ============================================================
//===========================================================================

#include <malloc.h>
#include <stdint.h>

// handy function returns the number of malloced bytes.
//
// Note: mallinfo() walks through internal lists to generate uordblks.
//       It sums the bin and fastbin usage.  I believe that it should
//       be around 200 steps which on a modern CPU should be pretty
//       fast.  However, you may not want to call this excessively.

inline uint32_t malloced()
{
    struct mallinfo mi = mallinfo();
    return mi.hblkhd + mi.uordblks;
}

//===========================================================================
//==== Module USB Cmd/Resp Structs ==========================================
//===========================================================================

#define	CICADA_MSG_BASE_DRVDEFS		   (0x0) /* cmds/responses also used by */
#define CICADA_MSG_BASE_SHARED		(0x0100) /* shared cmd/response base */
#define CICADA_MSG_BASE_COPPER		(0x0200) /* copper cmd/response base */


/**** Driver Level Definitions  ****/

#define DEV_RESP_STD		(CICADA_MSG_BASE_DRVDEFS + 0x80)


/**** Shared Definitions  ****/

#define CMD_CODE_LOADMEM          (CICADA_MSG_BASE_SHARED +    1)
#define CMD_CODE_READMEM          (CICADA_MSG_BASE_SHARED +    2)

#define DEV_RESP_OFFSET           (0x80)
#define DEV_RESP_BASE_SHARED      (CICADA_MSG_BASE_SHARED + DEV_RESP_OFFSET)
#define DEV_RESP_READMEM          (DEV_RESP_BASE_SHARED   +    1)

#define CMD_STAT_OK		    (0)	/* Command completed ok */
#define CMD_STAT_BADCMD		(1)	/* Illegal command */
#define CMD_STAT_BADPARAM	(2)	/* Command parameter error */
#define CMD_STAT_ABORT		(3)	/* Command was aborted */
#define CMD_STAT_EXEC		(4)	/* General execution error */


/***** Copper Specific *****/

#define COP_CMD_BASE    (CICADA_MSG_BASE_COPPER + 0x00)
#define COP_RESP_BASE   (CICADA_MSG_BASE_COPPER + 0x80)
#define COP_INT_BASE    (CICADA_MSG_BASE_COPPER + 0xe0)

#define CMD_CODE_SWITCHPWR      (COP_CMD_BASE + 0x05) // On/Off Analog Main Power
#define CMD_CODE_MEASPUMP       (COP_CMD_BASE + 0x09) // Phake Measure Gum Command


#define CUID_BASE       (CICADA_MSG_BASE_COPPER)
#define CUMEM_BASE      (CICADA_MSG_BASE_COPPER)
#define CM_MEM_MEASBLK  (CUMEM_BASE +   6)   // Measurement Data Block


#define INT_CODE_GUM_VEC    (COP_INT_BASE + 0x01)
#define INT_CODE_GUM_LOG    (COP_INT_BASE + 0x02)
#define INT_CODE_CMD_COMP   (COP_INT_BASE + 0x03)


//===========================================================================
//==== Serialization and Byte-Ordering Support ==============================
//===========================================================================

class Xfer {
public:
    static int read(uint8_t*  pV, uint8_t** ppBuf, int* pnRemaining);
    static int read(uint16_t* pV, uint8_t** ppBuf, int* pnRemaining);
    static int read(uint32_t* pV, uint8_t** ppBuf, int* pnRemaining);

    static int write(uint8_t  v, uint8_t** ppBuf, int* pnRemaining);
    static int write(uint16_t v, uint8_t** ppBuf, int* pnRemaining);
    static int write(uint32_t v, uint8_t** ppBuf, int* pnRemaining);
};

int Xfer::read(uint8_t*  pV, uint8_t** ppBuf, int* pnRemaining)
{
    uint8_t* pBuf = *ppBuf;
    int nBytes = sizeof(uint8_t);
    if (*pnRemaining >= nBytes) {
        *pV = *pBuf;
        *ppBuf += nBytes;
        *pnRemaining -= nBytes;
        return 1;
    }
    return 0;
}

int Xfer::read(uint16_t* pV, uint8_t** ppBuf, int* pnRemaining)
{
    uint8_t* pBuf = *ppBuf;
    int nBytes = sizeof(uint16_t);
    if (*pnRemaining >= nBytes) {
        uint16_t v = (uint16_t)  *pBuf++;
        v |= (uint16_t) (*pBuf << 8);
        *pV = v;
        *ppBuf += nBytes;
        *pnRemaining -= nBytes;
        return nBytes;
    }
    return 0;
}

int Xfer::read(uint32_t* pV, uint8_t** ppBuf, int* pnRemaining)
{
    uint8_t* pBuf = *ppBuf;
    int nBytes = sizeof(uint32_t);
    if (*pnRemaining >= nBytes) {
        uint32_t v = 0;
        v |= *pBuf++;
        v |= *pBuf++ <<  8;
        v |= *pBuf++ << 16;
        v |= *pBuf   << 24;
        *pV = v;
        *ppBuf += nBytes;
        *pnRemaining -= nBytes;
        return nBytes;
    }
    return 0;
}

int  Xfer::write(uint8_t  v, uint8_t** ppBuf, int* pnRemaining)
{
    uint8_t* pBuf = *ppBuf;
    int nBytes = sizeof(uint8_t);
    if (*pnRemaining >= nBytes) {
        *pBuf++ = v & 0xff;
        *ppBuf += nBytes;
        *pnRemaining -= nBytes;
        return nBytes;
    }
    return 0;
}

int Xfer::write(uint16_t v, uint8_t** ppBuf, int* pnRemaining)
{
    uint8_t* pBuf = *ppBuf;
    int nBytes = sizeof(uint16_t);
    if (*pnRemaining >= nBytes) {
        *pBuf++ = (uint8_t)( v        & 0xff);
        *pBuf++ = (uint8_t)((v >>  8) & 0xff);
        *ppBuf += nBytes;
        *pnRemaining -= nBytes;
        return nBytes;
    }
    return 0;
}
int Xfer::write(uint32_t v, uint8_t** ppBuf, int* pnRemaining)
{
    uint8_t* pBuf = *ppBuf;
    int nBytes = sizeof(uint32_t);
    if (*pnRemaining >= nBytes) {
        *pBuf++ = (uint8_t)( v        & 0xff);
        *pBuf++ = (uint8_t)((v >>  8) & 0xff);
        *pBuf++ = (uint8_t)((v >> 16) & 0xff);
        *pBuf++ = (uint8_t)((v >> 24) & 0xff);
        *ppBuf += nBytes;
        *pnRemaining -= nBytes;
        return nBytes;
    }
    return 0;
}



//===========================================================================
//==== Cicada/Mantis USB Transaction Header =================================
//===========================================================================

struct USB_HDR
{
    uint8_t	MagicNum;	/* fixed value for a standard header */
    uint8_t	SeqNum;		/* Incrementing sequence number */
    uint8_t	SeqInv;		/* 1's complement of SeqNum */
    uint8_t	RouteID;	/* Routing code for int endpoint packets */
    uint32_t	TransferSize;	/* Total bytes including this header */
    uint16_t	Reserved;	/* reserved */
    uint16_t	MsgID;		/* message ID (cmd, response or int msg) */

    USB_HDR();
    static int  nBytes();

    int         serialize(uint8_t* pBuf, int bufMax);
    int         deserialize(uint8_t* pBuf, int bufLen);
    void        dump();
};


#define ROUTE_EVENT	(0x01)	  /* Event Interrupt */
#define ROUTE_NET	(0x02)	  /* Communications related */
#define ROUTE_CNTL	(0x03)	  /* Measurement related */

USB_HDR::USB_HDR()
{
    MagicNum     = 0xa5;
    SeqNum       = 0;           // cheat, fixed seqnum & seqinv
    SeqInv       = 0xff;
    RouteID      = ROUTE_CNTL;
    TransferSize = 0;
    Reserved     = 0;
    MsgID        = 0;
}

int USB_HDR::nBytes()
{
    return 12; // don't trust sizeof
}

int USB_HDR::serialize(uint8_t* pBuf, int bufMax)
{
    int nTotal = 0;
    int n = 0;

    if ((n = Xfer::write(MagicNum, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(SeqNum, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(SeqInv, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(RouteID, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(TransferSize, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(Reserved, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(MsgID, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    return nTotal;
}

int USB_HDR::deserialize(uint8_t* pBuf, int bufLen)
{
    int n = 0;
    int nTotal = 0;

    if ((n = Xfer::read(&MagicNum, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::read(&SeqNum, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::read(&SeqInv, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::read(&RouteID, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::read(&TransferSize, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::read(&Reserved, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::read(&MsgID, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    return nTotal;
}

void USB_HDR::dump()
{
    printf("%12s    0x%02x\n", "magic",   MagicNum);
    printf("%12s    0x%02x\n", "seq",     SeqNum);
    printf("%12s    0x%02x\n", "seqInv",  SeqInv);
    printf("%12s    0x%02x\n", "RouteId", RouteID);
    printf("%12s    %d\n",   "size",    TransferSize);
    printf("%12s    0x%04x\n", "msgId",   MsgID);
}



//---------------------------------------------------------------------------
//---- ModCmd ---------------------------------------------------------------
//---------------------------------------------------------------------------

class ModCmd
{
public:
    ModCmd(const char* name=0);
    virtual ~ModCmd();

    USB_HDR     hdr;

    virtual int  serialize(uint8_t* pBuf, int bufMax);
    virtual void dump();

private:
    const char* _name;
};

ModCmd::ModCmd(const char* name)
    : _name(name)
{
}

ModCmd::~ModCmd()
{
}
    
int ModCmd::serialize(uint8_t* pBuf, int bufMax)
{
    int nHdr = hdr.serialize(pBuf, bufMax);
    if (nHdr != hdr.nBytes()) return 0;
    return nHdr;
}

void ModCmd::dump()
{
    printf("\n%s\n", _name);
    hdr.dump();
}



//---- ModCmdReadMem -----------------------------------------

class ModCmdReadMem : public ModCmd
{
public:
    ModCmdReadMem(uint16_t memSpace, uint32_t addr, uint32_t length);
   ~ModCmdReadMem();
    
    uint16_t    memSpace;
    uint16_t    unused;
    uint32_t    addr;
    uint32_t    length;

    int         serialize(uint8_t* pBuf, int bufMax);
    void        dump();
};

ModCmdReadMem::ModCmdReadMem(uint16_t memSpaceArg, uint32_t addrArg, uint32_t lengthArg)
    : ModCmd("ModCmdReadMem"),
      memSpace(memSpaceArg),
      unused(0),
      addr(addrArg),
      length(lengthArg)
{
    hdr.TransferSize = 24;
    hdr.MsgID = CMD_CODE_READMEM;
}

ModCmdReadMem::~ModCmdReadMem()
{
}

int ModCmdReadMem::serialize(uint8_t* pBuf, int bufMax)
{
    int n = ModCmd::serialize(pBuf, bufMax);
    if (!n) return 0;

    int nTotal = n;
    pBuf   += n;
    bufMax -= n;

    if ((n = Xfer::write(memSpace, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(unused, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(addr, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(length, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    return nTotal;
}

void ModCmdReadMem::dump()
{
    ModCmd::dump();
    printf("%12s    0x%04x\n", "memSpace", memSpace);
    printf("%12s    0x%04x\n", "addr",     addr);
    printf("%12s    %d\n",    "length",   length);
}



//---- ModCmdLoadMem -----------------------------------------

class ModCmdLoadMem : public ModCmd
{
public:
    ModCmdLoadMem(uint16_t memSpaceArg, uint32_t addrArg, uint8_t* pData=0, int lengthArg=0);
   ~ModCmdLoadMem();
    
    uint16_t    memSpace;
    uint16_t    unused;
    uint32_t    addr;
    uint32_t    length;

    int         serialize(uint8_t* pBuf, int bufMax);
    void        dump();

private:
    uint8_t*    _pData;
};

ModCmdLoadMem::ModCmdLoadMem(uint16_t memSpaceArg, uint32_t addrArg, uint8_t* pData, int lengthArg)
    : ModCmd("ModCmdLoadMem"),
      memSpace(memSpaceArg),
      unused(0),
      addr(addrArg),
      length(lengthArg),
      _pData(0)
{
    hdr.MsgID = CMD_CODE_LOADMEM;
    hdr.TransferSize = 0;

    if (length) {
        _pData = new uint8_t[length];
        memcpy(_pData, pData, length);
    }
}

ModCmdLoadMem::~ModCmdLoadMem()
{
    delete _pData;
}

int ModCmdLoadMem::serialize(uint8_t* pBuf, int bufMax)
{
    int n = ModCmd::serialize(pBuf, bufMax);
    if (!n) return 0;

    int nTotal = n;
    pBuf   += n;
    bufMax -= n;

    if ((n = Xfer::write(memSpace, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(unused, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(addr, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(length, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if (length && _pData) {
        for (uint32_t x = 0 ; x < length; x++) {
            if ((n = Xfer::write(_pData[x], &pBuf, &bufMax)) == 0) return 0;
            nTotal += n;
        }
    }
    return nTotal;
}

void ModCmdLoadMem::dump()
{
    ModCmd::dump();
    printf("%12s    0x%04x\n", "memSpace", memSpace);
    printf("%12s    0x%04x\n", "addr",     addr);
    printf("%12s    %d\n",    "length",   length);

    int n = length < 16 ? length : 16;
    for (int x = 0; x < n; x++) {
        printf("%02x ", _pData[x]);
        if (x == 7) printf("  ");
    }
    printf("\n");
}



//---- ModCmdModPower -----------------------------------------

class ModCmdModPower : public ModCmd
{
public:
    ModCmdModPower(uint16_t enable=0);
   ~ModCmdModPower();
    
    int         serialize(uint8_t* pBuf, int bufMax);
    void        dump();

private:
    uint16_t    _enable;
    uint16_t    _unused;
};

ModCmdModPower::ModCmdModPower(uint16_t enable)
    : ModCmd("ModCmdPower"), _enable(enable), _unused(0)
{
    hdr.TransferSize = 16;
    hdr.MsgID = CMD_CODE_SWITCHPWR;
}

ModCmdModPower::~ModCmdModPower()
{
}

int ModCmdModPower::serialize(uint8_t* pBuf, int bufMax)
{
    int n = ModCmd::serialize(pBuf, bufMax);
    if (!n) return 0;

    int nTotal = n;
    pBuf   += n;
    bufMax -= n;

    if ((n = Xfer::write(_enable, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_unused, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    return nTotal;
}

void ModCmdModPower::dump()
{
    ModCmd::dump();
    printf("%12s    0x%04x\n", "enable", _enable);
    printf("%12s    0x%04x\n", "unused", _unused);
}



//---- ModCmdMeasPump -----------------------------------------

class ModCmdMeasPump : public ModCmd
{
public:
    ModCmdMeasPump(uint16_t bufAddr, uint16_t bufLen, uint16_t steps, uint16_t pllSettle);
   ~ModCmdMeasPump();
    
    int         serialize(uint8_t* pBuf, int bufMax);
    void        dump();

private:
    uint16_t    _unused1;
    uint16_t    _bufAddr;
    uint16_t    _bufLen;
    uint16_t    _unused2;
    uint16_t    _unused3;
    uint16_t    _unused4;
    uint16_t    _steps;
    uint16_t    _slots;
    uint16_t    _unused5;
    uint16_t    _stepsPerPkt;
    uint16_t    _unused6;
    uint16_t    _pllSettle;
    uint16_t    _unused7;
    uint16_t    _unused8;
};

ModCmdMeasPump::ModCmdMeasPump(uint16_t bufAddr, uint16_t bufLen, uint16_t steps, uint16_t pllSettle)
    : ModCmd("ModCmdMeasPump"),
      _unused1(0),
      _bufAddr(bufAddr),
      _bufLen(bufLen),
      _unused2(0),
      _unused3(0),
      _unused4(0),
      _steps(steps),
      _slots(2),
      _unused5(0),
      _stepsPerPkt(1),
      _unused6(0),
      _pllSettle(pllSettle),
      _unused7(0),
      _unused8(0)
{
    hdr.TransferSize = 40;
    hdr.MsgID = CMD_CODE_MEASPUMP;
}

ModCmdMeasPump::~ModCmdMeasPump()
{
}

int ModCmdMeasPump::serialize(uint8_t* pBuf, int bufMax)
{
    int n = ModCmd::serialize(pBuf, bufMax);
    if (!n) return 0;

    int nTotal = n;
    pBuf   += n;
    bufMax -= n;

    if ((n = Xfer::write(_unused1, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_bufAddr, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_bufLen, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_unused2, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_unused3, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_unused4, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_steps, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_slots, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_unused5, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_stepsPerPkt, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_unused6, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_pllSettle, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_unused7, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::write(_unused8, &pBuf, &bufMax)) == 0) return 0;
    nTotal += n;

    return nTotal;
}

void ModCmdMeasPump::dump()
{
    ModCmd::dump();
    printf("%12s    0x%04x\n", "bufAddr",     _bufAddr);
    printf("%12s    %d\n",     "bufLen",      _bufLen);
    printf("%12s    %d\n",     "steps",       _steps);
    printf("%12s    %d\n",     "slots",       _slots);
    printf("%12s    %d\n",     "stepsPerPkt", _stepsPerPkt);
    printf("%12s    %d\n",     "pllSettle",   _pllSettle);
}


//---------------------------------------------------------------------------
//---- ModResp --------------------------------------------------------------
//---------------------------------------------------------------------------

class ModResp
{
public:
    ModResp(const char* name=0);
    virtual ~ModResp();

    USB_HDR     hdr;

    virtual int  deserialize(uint8_t* pBuf, int bufLen);    
    virtual void dump();

private:
    const char* _name;
};

ModResp::ModResp(const char* name)
    : _name(name)
{
}

ModResp::~ModResp()
{
}

int ModResp::deserialize(uint8_t* pBuf, int bufLen)
{
    int nHdr = hdr.deserialize(pBuf, bufLen);
    if (nHdr != hdr.nBytes()) return 0;
    return nHdr;
}

void ModResp::dump()
{
    printf("\n%s\n", _name);
    hdr.dump();
}


//---- ModRespStd --------------------------------------------

class ModRespStd : public ModResp
{
public:
    ModRespStd();
   ~ModRespStd();
    uint16_t    ccstat;
    uint16_t    devstat;

    int         deserialize(uint8_t* pBuf, int bufLen);
    void        dump();
};

ModRespStd::ModRespStd()
    : ModResp("ModRespStd")
{
}

ModRespStd::~ModRespStd()
{
}

int ModRespStd::deserialize(uint8_t* pBuf, int bufLen)
{
    int n = ModResp::deserialize(pBuf, bufLen);
    if (!n) return 0;

    int nTotal = n;
    pBuf   += n;
    bufLen -= n;

    if ((n = Xfer::read(&ccstat, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::read(&devstat, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    return nTotal;
}

void ModRespStd::dump()
{
    ModResp::dump();
    printf("%12s    0x%04x\n", "ccstat",  ccstat);
    printf("%12s    0x%04x\n", "devstat", devstat);
}


//---- ModRespReadMem ----------------------------------------

class ModRespReadMem : public ModResp
{
public:
    ModRespReadMem();
   ~ModRespReadMem();

    uint16_t    ccstat;
    uint16_t    devstat;
    uint32_t    length;

    int         deserialize(uint8_t* pBuf, int bufLen);
    void        dump();

private:
    uint8_t*    _pData;
};

ModRespReadMem::ModRespReadMem()
    : ModResp("ModRespReadMem"),
      _pData(0)
{
}

ModRespReadMem::~ModRespReadMem()
{
    delete _pData;
}

int ModRespReadMem::deserialize(uint8_t* pBuf, int bufLen)
{
    int n = ModResp::deserialize(pBuf, bufLen);
    if (!n) return 0;

    int nTotal = n;
    pBuf   += n;
    bufLen -= n;

    if ((n = Xfer::read(&ccstat, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::read(&devstat, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    if ((n = Xfer::read(&length, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    if (length) {
        _pData = new uint8_t[length];

        for (uint32_t x = 0 ; x < length; x++) {
            if ((n = Xfer::read(&_pData[x], &pBuf, &bufLen)) == 0) return 0;
            nTotal += n;
        }
    }
    return nTotal;
}

void ModRespReadMem::dump()
{
    ModResp::dump();
    printf("%12s    0x%04x\n", "ccstat",  ccstat);
    printf("%12s    0x%04x\n", "devstat", devstat);
    printf("%12s    %8d\n",    "length",  length);

    int n = length < 16 ? length : 16;
    for (int x = 0; x < n; x++) {
        printf("%02x ", _pData[x]);
        if (x == 7) printf("  ");
    }
    printf("\n");
}


//---- ModRespGumData --------------------------------------------

class ModRespGumData : public ModResp
{
public:
    ModRespGumData();
   ~ModRespGumData();
    uint16_t    ccstat;

    int         deserialize(uint8_t* pBuf, int bufLen);
    void        dump();

private:
    uint8_t*    _pData;
};

ModRespGumData::ModRespGumData()
    : ModResp("ModRespGumData"), _pData(0)
{
//    printf("ModRespGumData::ModRespGumData(): malloced = %d\n", malloced());
}

ModRespGumData::~ModRespGumData()
{
//    printf("ModRespGumData::~ModRespGumData(): malloced = %d\n", malloced());
    delete _pData;
}

int ModRespGumData::deserialize(uint8_t* pBuf, int bufLen)
{
//    printf("ModRespGumData::deserialize(): malloced = %d\n", malloced());
    int n = ModResp::deserialize(pBuf, bufLen);
    if (!n) return 0;

    int nTotal = n;
    pBuf   += n;
    bufLen -= n;

    if ((n = Xfer::read(&ccstat, &pBuf, &bufLen)) == 0) return 0;
    nTotal += n;

    int nRemaining = hdr.TransferSize - nTotal;
    _pData = new uint8_t[nRemaining];
    for (int x = 0; x < nRemaining; x++) {
        if ((n = Xfer::read(&_pData[x], &pBuf, &bufLen)) == 0) return 0;
        nTotal += n;
    }
    return nTotal;
}

void ModRespGumData::dump()
{
    ModResp::dump();
    printf("%12s    0x%04x\n", "ccstat",  ccstat);

}




//===========================================================================
//==== SlabInfo =============================================================
//===========================================================================

struct SlabInfoEntry
{
    SlabInfoEntry();
    virtual ~SlabInfoEntry();
    string      name;
    int         active_objs;
    int         obj_size;

    virtual void dump();
};

struct SlabInfoEntryDelta : public SlabInfoEntry
{
    SlabInfoEntryDelta();
   ~SlabInfoEntryDelta();

    int         active_objs_prev;
    int         active_objs_curr;
    void        dump();
};

typedef list<SlabInfoEntry*> SlabInfoList;

class SlabInfo 
{
public:
    SlabInfo();
   ~SlabInfo();

    bool          readState(const char* writeFile=0);
    void          addEntry(SlabInfoEntry* pSlabInfoEntry);

    SlabInfoList* pSlabInfoList();
    int           nGreaterThan(int ifActiveGreaterThan=-1, bool absValue=false, const char* matchName=0);
    void          dump(int ifActiveGreaterThan=-1, bool absValue=false);
    void          clear();
    
private:
    SlabInfoList _slabInfoList;

    void        getEntry(char* inbuf, int bufLen, SlabInfoEntry* pSlabInfoEntry);
};



//===========================================================================
//==== Misc =================================================================
//===========================================================================

void millisleep(double delayMs) 
{
    struct timespec ts;
    ts.tv_sec  = (uint32_t)(delayMs / 1000.0);
    ts.tv_nsec = (uint32_t)((delayMs - (float)ts.tv_sec * 1000.0) * 1e6);
    nanosleep(&ts, 0);
}

uint64_t getus()
{
    struct timeval  tv;
    struct timezone tz;
    gettimeofday(&tv, &tz);
    uint64_t us = (uint64_t) tv.tv_sec * (uint64_t) 1000000 + (uint64_t) tv.tv_usec;
    return us;
}


//===========================================================================
//==== Main =================================================================
//===========================================================================

void usage(const char* msg)
{
    if (msg) printf("Error: %s\n", msg);
    printf("usage: cicada_meas_test <cmd|evt> <nloops> <min sec before unplug float> <random unplug max sec float>\n");
    exit(1);
}


enum TEST_TYPE {
     TEST_TYPE_INVALID,
     TEST_TYPE_CMD,
     TEST_TYPE_EVT,
};
    
void cmdTest(int loops, float unplugMinSec, float unplugMaxSec);
void evtTest(int loops, float unplugMinSec, float unplugMaxSec);

int main(int argc, char** argv)
{
    if (argc != (1+4))
        usage("wrong nubmer of args");
    
    TEST_TYPE testType = TEST_TYPE_INVALID;
    if (!strcmp(argv[1], "cmd")) {
        testType = TEST_TYPE_CMD;
    }
    else if (!strcmp(argv[1], "evt")) {
        testType = TEST_TYPE_EVT;
    }
    else {
        usage("invalid test type: 'cmd' or 'int'");
    }

    int loops = 1;
    if (sscanf(argv[2], "%d", &loops) != 1) {
        usage("enter an integer >= 1");
    }
    if (loops < 1) {
        usage("enter an integer >= 1");
    }

    float unplugMinSec = 0;
    if (sscanf(argv[3], "%f", &unplugMinSec) != 1) {
        usage("enter an integer >= 0");
    }
    if (unplugMinSec < 0.0) {
        usage("enter an integer >= 0");
    }

    float unplugMaxSec = 0;
    if (sscanf(argv[4], "%f", &unplugMaxSec) != 1) {
        usage("enter an integer >= 0");
    }
    if (unplugMaxSec < 0.0) {
        usage("enter an integer >= 0");
    }

    printf("testType = %d, loops = %d, unplugMinSec = %8.3f, unplugMaxSec = %8.3f\n",
           testType, loops, unplugMinSec, unplugMaxSec);


    switch (testType) {
    case TEST_TYPE_CMD:
        cmdTest(loops, unplugMinSec, unplugMaxSec);
        break;

    case TEST_TYPE_EVT:
        evtTest(loops, unplugMinSec, unplugMaxSec);
        break;

    case TEST_TYPE_INVALID:
        usage("Invalid test type");
        break;
    }

    return 0;
}

bool slabInfoDiff(SlabInfo* p1, SlabInfo* p2, SlabInfo* pDiff);



//===========================================================================
//==== cmdTest and Helpers ==================================================
//===========================================================================

bool cmdAndResp(int fd, ModCmd* pModCmd, ModResp* pModResp);
pthread_t threadContext;
struct RandomAbortParms 
{
    float       unplugMinSec;
    float       unplugMaxSec;
};
void* randomAbortFunc(void* arg);

int globalLoopCount = 0;

void cmdTest(int loops, float unplugMinSec, float unplugMaxSec)
{
    bool ok = false;
    int fd = -1;

    SlabInfo* sifirst = new SlabInfo;
    sifirst->readState("/tmp/slabinfo.first");

    // command/response to write known values to DSP memory
    uint16_t membuf[] = { 1, 2, 3, 4, 5 };
    ModCmdLoadMem modCmdLoadMem(CM_MEM_MEASBLK, 0, (uint8_t*)membuf, sizeof(membuf));
    ModRespStd    modRespLoadMem;

    // command/response to read back that DSP memory
    ModCmdReadMem modCmdReadMem(CM_MEM_MEASBLK, 0, sizeof(membuf));
    ModRespReadMem modRespReadMem;

    // start random abort thread
    if (unplugMinSec > 0 || unplugMaxSec > 0) {
        RandomAbortParms* pParms = new RandomAbortParms;
        pParms->unplugMinSec = unplugMinSec;
        pParms->unplugMaxSec = unplugMaxSec;
        if (pthread_create(&threadContext, NULL, randomAbortFunc, (void*)pParms)) {
            printf("cmdTest: Error: couldn't start random abort thread\n");
            exit(1);
        }
        pthread_detach(threadContext);
    }
    
    SlabInfo* siprev = new SlabInfo;
    SlabInfo* sicurr = new SlabInfo;
    SlabInfo* sidiff = new SlabInfo;
    SlabInfo* sitemp = 0;

    for (globalLoopCount = 0; globalLoopCount < loops; globalLoopCount++) {

        do {
#if !defined(NO_IO)
            TRFLOW("open\n");
            if ((fd = open(cmdDev, O_RDWR)) < 0) {
                ok = false;
                break;
            }
#endif
            // set known pattern into DSP memory
            TRFLOW("ModCmd LoadMem\n");
            if (!(ok = cmdAndResp(fd, &modCmdLoadMem, &modRespLoadMem))) {
                break;
            }

            siprev->readState();

            for (; globalLoopCount < loops; globalLoopCount++) {

                if ((globalLoopCount % COUNT_UPDATE_ITERATIONS) == 0) {
                    printf("%d\n", globalLoopCount);
                }

                ok = cmdAndResp(fd, &modCmdReadMem, &modRespReadMem);
                if (!ok) {
                    printf("cmdTest: Error: cmdAndResp\n");
                    break;
                }
                if (modRespReadMem.ccstat != CMD_STAT_OK) {
                    printf("cmdTest: Error: ccstat = %d\n", modRespReadMem.ccstat);
                    ok = false;
                    break;
                }

                if ((globalLoopCount % SLABINFO_ITERATIONS) == (SLABINFO_ITERATIONS-1)) {

                    // update slabinfo stats and output if there's been a change
                    sicurr->readState();

                    sidiff->clear();
                    if ((ok = slabInfoDiff(siprev, sicurr, sidiff)) == true) {
                        int limit = 0;
                        if (sidiff->nGreaterThan(limit, ABS_COMPARISONS, SLABINFO_FIELD)) {
                            printf("%d\n", globalLoopCount);
                            sidiff->dump(limit);
                        }
                    }
                    else {
                        printf("slabInfoDiff() returned error\n");
                        break;
                    }

                    // exchange prev and curr for next time
                    sitemp = siprev;
                    siprev = sicurr;
                    sicurr = sitemp;
                    sitemp = 0;
                }
            }
        }
        while(0);
        
#if !defined(NO_IO)
        if (fd != -1) {
            TRFLOW("close()\n");
            close(fd);
            fd = -1;
        }
#endif
        // we've just had the plug pulled
        millisleep(500);
    }

    delete siprev;
    delete sicurr;
    delete sidiff;
    delete sitemp;

#ifdef OUTPUT_SLABINFO_TOTALS
    printf("\n\n**** Final Diffs from start ****\n");
    SlabInfo* sitotal = new SlabInfo;
    SlabInfo* silast  = new SlabInfo;
    silast->readState("/tmp/slabinfo.last");

    if ((ok = slabInfoDiff(sifirst, silast, sitotal)) == true) {
        int limit = 0;
        if (sitotal->nGreaterThan(limit, ABS_COMPARISONS, SLABINFO_FIELD)) {
            sitotal->dump(limit);
        }
    }
    delete sitotal;
    delete silast;
#endif
}


bool modReboot()
{
    bool ok = false;
    static const char* ctlpath = "/sys/class/cicada_meas/cicada_meas_cmd0/device/ctlmsg";
    static const char* rebootval = "85\n";
    int fd = open(ctlpath, O_WRONLY);
    if (fd >= 0) {
        write(fd, rebootval, strlen(rebootval));
        close(fd);
        ok = true;
    }
    return ok;
}


bool hubPower(bool enable, double delayMs)
{
    bool ok = false;

    do {
        static const char* hubPortNameSysNode = "/sys/class/cicada_meas/hub_port_name";
        int fdHubPortName = open(hubPortNameSysNode, O_WRONLY);
        if (fdHubPortName < 0) {
            printf("Error opening %s\n", hubPortNameSysNode);
            ok = false;
            break;
        }
        static const char* hubPortName = "1-1.2";
        int nChars = strlen(hubPortName);
        if (write(fdHubPortName, hubPortName, nChars) != nChars) {
            printf("Error writing %s\n", hubPortName);
            ok = false;
            break;
        }

        static const char* hubPortPowerSysNode = "/sys/class/cicada_meas/hub_power";
        int fdHubPortPower = open(hubPortPowerSysNode, O_WRONLY);
        if (fdHubPortPower < 0) {
            printf("Error opening %s\n", hubPortPowerSysNode);
            ok = false;
            break;
        }

        static const char* hubPortPowerEnable = 0;
        if (enable) {
            hubPortPowerEnable = "1";
        }
        else {
            hubPortPowerEnable = "0";
        }

        nChars = strlen(hubPortPowerEnable);
        if (write(fdHubPortPower, hubPortPowerEnable, nChars) != nChars) {
            printf("Error writing hub port power enable = %s\n", hubPortPowerEnable);
            ok = false;
            break;
        }
        if (delayMs) {
            millisleep(delayMs);
        }
    }
    while(0);

    return ok;
}



int abortCount = 0;

void* randomAbortFunc(void* arg)
{
    RandomAbortParms* pParms = (RandomAbortParms*) arg;
    double start = (double)getus();

    for (;;) {
        if (pParms->unplugMinSec > 0) {
            millisleep(pParms->unplugMinSec * 1000.0);
        }
        double additionalDelayMax = pParms->unplugMaxSec - pParms->unplugMinSec;
        if (additionalDelayMax > 0) {
            double additionalDelaySec = ((double)rand()/(double)RAND_MAX) * additionalDelayMax;
            millisleep(additionalDelaySec * 1000.0);
        }

        if (modReboot()) {
            double now = (double)getus();
//            double deltaSec = (now - start) / 1e6;
            start = now;
            abortCount++;
//            printf("**** abort: loop count = %d, abort count = %d, deltaSec = %6.3lf ****\n",
//                   globalLoopCount, abortCount, deltaSec);
        }
        else {
//            printf("randomAbortFunc: Error, can't open ctlmsg\n");
        }
    }
    return 0;
}


bool cmdAndResp(int fd, ModCmd* pModCmd, ModResp* pModResp)
{
    const int BUFMAX = 1024;
    uint8_t outBuf[BUFMAX];
    int nToWrite = pModCmd->serialize(outBuf, BUFMAX);

#ifdef DUMP_OUTDATA
    int xw = 0;
    printf("out: ");
    for (xw = 0; xw < USB_HDR::nBytes(); xw++) {
        printf("%02x ", outBuf[xw]);
    }
    printf("\n");
    printf("out: ");
    
    for ( ; xw < nToWrite; xw++) {
        printf("%02x ", outBuf[xw]);
    }
    printf("\n");
#else
    (void) nToWrite;
#endif

#if !defined(NO_IO)
    int nWritten = write(fd, outBuf, nToWrite);
    if (nWritten != nToWrite) {
        printf("Error: nWritten = %d, nToWrite = %d\n", nWritten, nToWrite);
        return false;
    }

    uint8_t inBuf[BUFMAX];
    int nRead = read(fd, inBuf, BUFMAX);
//    printf("nRead = %d\n", nRead);
    int nDeser = pModResp->deserialize(inBuf, nRead);
    if (!nDeser) {
        return false;
    }
#endif

#ifdef DUMP_INDATA
    int xr = 0;
    printf("in:  ");
    for (xr = 0; xr < USB_HDR::nBytes(); xr++) {
        printf("%02x ", inBuf[xr]);
    }
    printf("\n");
    printf("in:  ");

    for ( ; xr < nRead; xr++) {
        printf("%02x ", inBuf[xr]);
    }
    printf("\n");
#endif

    return true;
}




//===========================================================================
//==== evtTest and Helpers ==================================================
//===========================================================================

bool evtResp(int fd);


void evtTest(int loops, float unplugMinSec, float unplugMaxSec)
{
    bool ok = false;
    int fdCmd = -1;
    int fdEvt = -1;

    // make sure hub power is on
    hubPower(1, 1000);

    SlabInfo* sifirst = new SlabInfo;
    sifirst->readState("/tmp/slabinfo.first");

    // command/response to write known values to DSP memory
    uint16_t initbuf[] = { 1, 2, 3, 4, 5, 6, 7, 8 };
    ModCmdLoadMem modCmdLoadMem(CM_MEM_MEASBLK, 0, (uint8_t*)initbuf, sizeof(initbuf));
    ModRespStd    modRespLoadMem;

    // command/response to read back that DSP memory
    ModCmdReadMem modCmdReadMem(CM_MEM_MEASBLK, 0, sizeof(initbuf));
    ModRespReadMem modRespReadMem;

    // command/response to turn on module power
    ModCmdModPower modCmdModPowerOn(1);
    ModRespStd     modRespModPowerOn;

    // command/response to generate predictable interrupt endpoint activity
    ModCmdMeasPump modCmdMeasPump(0, sizeof(initbuf), 100, 20);
    ModRespStd     modRespMeasPump;


    // start random abort thread
    if (unplugMinSec > 0 || unplugMaxSec > 0) {
        RandomAbortParms* pParms = new RandomAbortParms;
        pParms->unplugMinSec = unplugMinSec;
        pParms->unplugMaxSec = unplugMaxSec;
        if (pthread_create(&threadContext, NULL, randomAbortFunc, (void*)pParms)) {
            printf("evtTest: Error: couldn't start random abort thread\n");
            exit(1);
        }
        pthread_detach(threadContext);
    }

    SlabInfo* siprev = new SlabInfo;
    SlabInfo* sicurr = new SlabInfo;
    SlabInfo* sidiff = new SlabInfo;
    SlabInfo* sitemp = 0;

    for (globalLoopCount = 0; globalLoopCount < loops; globalLoopCount++) {

        do {
#if !defined(NO_IO)
            TRFLOW("open cmd\n");
            if ((fdCmd = open(cmdDev, O_RDWR)) < 0) {
                ok = false;
                break;
            }

            TRFLOW("open evt\n");
            if ((fdEvt = open(evtDev, O_RDONLY)) < 0) {
                ok = false;
                break;
            }
#endif
            // turn on module power
            TRFLOW("ModCmd ModPower ON\n");
            if (!(cmdAndResp(fdCmd, &modCmdModPowerOn, &modRespModPowerOn))) {
                ok = false;
                break;
            }

            // set known pattern into DSP memory
            TRFLOW("ModCmd LoadMem\n");
            if (!(ok = cmdAndResp(fdCmd, &modCmdLoadMem, &modRespLoadMem))) {
                break;
            }

            siprev->readState();

            for (; globalLoopCount < loops; globalLoopCount++) {

                if ((globalLoopCount % COUNT_UPDATE_ITERATIONS) == 0) {
                    printf("%d\n", globalLoopCount);
                }

                TRFLOW("ModCmd MeasPump\n");
                ok = cmdAndResp(fdCmd, &modCmdMeasPump, &modRespMeasPump);
                if (!ok) {
//                    printf("evtTest: Error: cmdAndResp\n");
                    break;
                }
                if (modRespMeasPump.ccstat != CMD_STAT_OK) {
                    printf("evtTest: Error: ccstat = %d\n", modRespMeasPump.ccstat);
                    ok = false;
                    break;
                }

                TRFLOW("Get Gum Event Data\n");
                if (!(ok = evtResp(fdEvt))) {
//                    printf("evtTest: Error: evtResp\n");
                    break;
                }

//                if (SLABINFO_ALWAYS || (globalLoopCount % SLABINFO_ITERATIONS) == (SLABINFO_ITERATIONS-1)) {
                if (1) {
//                    printf("globalLoopCount = %d, malloced = %d\n", globalLoopCount, malloced());

                    // update slabinfo stats and output if there's been a change
                    sicurr->readState();

                    sidiff->clear();
                    if ((ok = slabInfoDiff(siprev, sicurr, sidiff)) == true) {
                        int limit = 0;
                        if (sidiff->nGreaterThan(limit, ABS_COMPARISONS, 0)) {
                            printf("\ncount = %d, abortCount = %d, malloced = %d\n",
                                   globalLoopCount, abortCount, malloced());
                            sidiff->dump(limit, ABS_COMPARISONS);
                        }
                    }
                    else {
                        printf("slabInfoDiff() returned error\n");
                        break;
                    }

                    // exchange prev and curr for next time
                    sitemp = siprev;
                    siprev = sicurr;
                    sicurr = sitemp;
                    sitemp = 0;
                }
            }
        }
        while(0);
        
#if !defined(NO_IO)
        if (fdCmd != -1) {
            TRFLOW("close cmd\n");
            close(fdCmd);
            fdCmd = -1;
        }
        if (fdEvt != -1) {
            TRFLOW("close evt\n");
            close(fdEvt);
            fdEvt = -1;
        }
#endif
        // we've just had the plug pulled
        millisleep(500);
    }

    // unplug module and leave unplugged
//    modReboot();
//    millisleep(10);
//    hubPower(0, 500);

    delete siprev;
    delete sicurr;
    delete sidiff;
    delete sitemp;

#ifdef OUTPUT_SLABINFO_TOTALS
    SlabInfo* sitotal = new SlabInfo;
    SlabInfo* silast  = new SlabInfo;
    silast->readState("/tmp/slabinfo.last");

    if ((ok = slabInfoDiff(sifirst, silast, sitotal)) == true) {
        int limit = 0;
        if (sitotal->nGreaterThan(limit, ABS_COMPARISONS)) {
            sitotal->dump(limit, ABS_COMPARISONS);
        }
    }
#endif
}



bool evtResp(int fd)
{
    const int BUFMAX = 1024;
    uint8_t inBuf[BUFMAX];
    bool ok = true;

    while (1) {
                    
#if !defined(NO_IO)
        int nRead = read(fd, inBuf, BUFMAX);
    //    printf("nRead = %d\n", nRead);
        ModRespGumData* pModRespGumData = new ModRespGumData;
        int nDeser = pModRespGumData->deserialize(inBuf, nRead);
        if (!nDeser) {
            delete pModRespGumData;
            ok = false;
            break;
        }
#endif

#ifdef DUMP_INDATA
        int xr = 0;
        printf("evt: ");
        for (xr = 0; xr < USB_HDR::nBytes(); xr++) {
            printf("%02x ", inBuf[xr]);
        }
        printf("\n");
        printf("evt:  ");

        for ( ; xr < nRead; xr++) {
            printf("%02x ", inBuf[xr]);
        }
        printf("\n");
#endif

        if (pModRespGumData->hdr.MsgID == INT_CODE_GUM_VEC ||
            pModRespGumData->hdr.MsgID == INT_CODE_GUM_LOG)
        {
//            printf("pModRespGumData->hdr.MsgID = 0x%x\n", pModRespGumData->hdr.MsgID);
        }
        else if (pModRespGumData->hdr.MsgID == INT_CODE_CMD_COMP) {
            delete pModRespGumData;
            ok = true;
            break;
        }
        delete pModRespGumData;
    }

    return ok;
}




//===========================================================================
//==== More SlabInfo ========================================================
//===========================================================================

SlabInfo::SlabInfo()
{
}


SlabInfo::~SlabInfo()
{
    clear();
}

void SlabInfo::clear()
{
    SlabInfoList::const_iterator pos;
    for (pos = _slabInfoList.begin(); pos != _slabInfoList.end(); pos++) {
        delete (*pos);
    }
    _slabInfoList.clear();
}


bool SlabInfo::readState(const char* writeFile)
{
    clear();

    FILE* fpRead = fopen("/proc/slabinfo", "r");
    if (!fpRead) return false;

    FILE* fpWrite = 0;
    if (writeFile) {
        fpWrite = fopen(writeFile, "w");
        if (!fpWrite) return false;
    }

    const char* slabInfoVersTag = "slabinfo - version: ";

    const int BUFSIZE = 1024;
    char buf[BUFSIZE];
    while(1) {
        if (!fgets(buf, BUFSIZE, fpRead)) {
            break;
        }

        if (fpWrite) {
            if (fputs(buf, fpWrite) == EOF) {
                break;
            }
        }

        // skip over comments
        if (buf[0] == '#') {
            continue;
        }

        // skip over slabinfo version
        if (!strncmp(buf, slabInfoVersTag, strlen(slabInfoVersTag))) {
            continue;
        }

        SlabInfoEntry* pSlabInfoEntry = new SlabInfoEntry;
        getEntry(buf, BUFSIZE, pSlabInfoEntry);
        _slabInfoList.push_back(pSlabInfoEntry);
    }
    fclose(fpRead);
    if (fpWrite) {
        fclose(fpWrite);
    }

    return true;
}


void SlabInfo::getEntry(char* inbuf, int bufLen, SlabInfoEntry* pSlabInfoEntry)
{
    // local copy of buffer because we're going to modify it
    char buf[bufLen];
    memcpy(buf, inbuf, bufLen);

    char* p      = buf;
    char* pStart = buf;

    enum STATE {
         STATE_FIND_START,
         STATE_FIND_END,
    };

    const int NFIELDS = 5;
    char* pFields[NFIELDS];
    pFields[0] = pStart;

    int fieldx = 0;
    STATE state = STATE_FIND_END;
    for (; p < (buf + bufLen); p++) {

        if (state == STATE_FIND_START) {
            if (*p == ' ') {
                continue;
            }
            pStart = p;
            state = STATE_FIND_END;
        }
        else if (state == STATE_FIND_END) {
            if (*p == ' ') {
                *p = '\0';
                pFields[fieldx++] = pStart;
                if (fieldx >= NFIELDS) {
                    break;
                }
                state = STATE_FIND_START;
            }
        }
    }

    assert(fieldx == NFIELDS);
    pSlabInfoEntry->name = pFields[0];
//    printf("**** %s %s %s %s %s\n", pFields[0], pFields[1], pFields[2], pFields[3], pFields[4]);

    assert(sscanf(pFields[1], "%d", &pSlabInfoEntry->active_objs) == 1);
    assert(sscanf(pFields[3], "%d", &pSlabInfoEntry->obj_size) == 1);
}

void SlabInfo::addEntry(SlabInfoEntry* pSlabInfoEntry)
{
    _slabInfoList.push_back(pSlabInfoEntry);
}


SlabInfoList* SlabInfo::pSlabInfoList()
{
    return &_slabInfoList;
}

int SlabInfo::nGreaterThan(int ifActiveGreaterThan, bool absValue, const char* matchName)
{
    int nGreaterThan = 0;
    SlabInfoList::const_iterator pos;
    for (pos = _slabInfoList.begin(); pos != _slabInfoList.end(); pos++) {
        if (matchName) {
            if (!strcmp(matchName, (*pos)->name.c_str())) {
                printf("matchName = '%s', name = '%s', active_objs = %d\n",
                       matchName, (*pos)->name.c_str(), (*pos)->active_objs);
                int n = (*pos)->active_objs;
                if (absValue) {
                    n = abs(n);
                }
                if (n > ifActiveGreaterThan) {
                    nGreaterThan++;
                }
            }
        }
        else {
            int n = (*pos)->active_objs;
            if (absValue) {
                n = abs(n);
            }
            if (n > ifActiveGreaterThan) {
                nGreaterThan++;
            }
        }
    }
    return nGreaterThan;
}

void SlabInfo::dump(int ifActiveGreaterThan, bool absValue)
{
    bool dumped = false;
    
    SlabInfoList::const_iterator pos;
    for (pos = _slabInfoList.begin(); pos != _slabInfoList.end(); pos++) {
        int n = (*pos)->active_objs;
        if (absValue) {
            n = abs(n);
        }
        if (n > ifActiveGreaterThan) {
            (*pos)->dump();
            dumped = true;
        }
    }

    if (dumped) {
//        printf("malloced = %d\n", malloced());
    }
}    

SlabInfoEntry::SlabInfoEntry()
{
}

SlabInfoEntry::~SlabInfoEntry()
{
}

void SlabInfoEntry::dump()
{
    printf("%-18s: active_objs = %8d, size = %8d\n",  name.c_str(), active_objs, obj_size);
}

SlabInfoEntryDelta::SlabInfoEntryDelta()
{
}

SlabInfoEntryDelta::~SlabInfoEntryDelta()
{
}


void SlabInfoEntryDelta::dump()
{
    printf("%-18s: active_objs: prev = %8d, curr = %8d, delta = %8d, size = %8d\n",
           name.c_str(), active_objs_prev, active_objs_curr, active_objs, obj_size);
}

bool slabInfoDiff(SlabInfo* pSlabInfo1, SlabInfo* pSlabInfo2, SlabInfo* pSlabInfoDiff)
{
    SlabInfoList* p1 = pSlabInfo1->pSlabInfoList();
    SlabInfoList* p2 = pSlabInfo2->pSlabInfoList();
    if (p1->size() != p2->size()) {
        return false;
    }

    SlabInfoList::const_iterator pos1 = p1->begin();
    SlabInfoList::const_iterator pos2 = p2->begin();
    for (uint32_t x = 0; x < p1->size(); x++, pos1++, pos2++) {
        if ((*pos1)->name == (*pos2)->name) {

            int delta = (*pos2)->active_objs - (*pos1)->active_objs;
            if (delta) {
                SlabInfoEntryDelta* pEntryDelta = new SlabInfoEntryDelta;
                pEntryDelta->name             = (*pos1)->name;
                pEntryDelta->active_objs_prev = (*pos1)->active_objs;
                pEntryDelta->active_objs_curr = (*pos2)->active_objs;
                pEntryDelta->active_objs      = delta;
                pEntryDelta->obj_size         = (*pos1)->obj_size;
                pSlabInfoDiff->addEntry(pEntryDelta);
            }
        }
    }
    return true;
}
