/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License V2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * --------------------------------------------------------------------------
 *
 * Adapted from snull.c --  the Simple Network Utility
 *
 * Copyright (C) 2001 Alessandro Rubini and Jonathan Corbet
 * Copyright (C) 2001 O'Reilly & Associates
 *
 * The source code in this file can be freely used, adapted,
 * and redistributed in source or binary form, so long as an
 * acknowledgment appears in derived source files.  The citation
 * should list that the code comes from the book "Linux Device
 * Drivers" by Alessandro Rubini and Jonathan Corbet, published
 * by O'Reilly & Associates.   No warranty is attached;
 * we cannot take responsibility for errors or fitness for use.
 *
 * --------------------------------------------------------------------------
 *
 * Network driver for usb-based cicada measurement modules
 *
 * Testing:
 * 	ping -s 1400 -p 00 -A -n -W 1 10.0.0.2 should drop very few (< 1%)
 * 	packets (assuming no other traffic).
 *
 * TODO:
 *
 * anchors
 *   look at email
 * look at usb skeleton driver
 * look at usb.h
 *
 * long-reach (speed on fiber): some way to specify when first bringing up interface?
 * 	userspace tool/lib to configure (like ifconfig), which sets speed
 * 	        using sysfs at the same time
 * 	ifconfig uses SIOCSIFADDR, SIOCGIFFLAGS, SIOCSIGFLAGS
 *
 * 	look at ethtool
 * 	look at how ppp specifies speed
 * 		pppd sets baud rate of underlying serial device
 *
 * make sure looking at all function return codes
 * make sure looking at all returned status bits
 *
 * change printk's to dev_* (see Chapter 13 in CodingStyle)
 *
 * handle tx timeout
 * TBD: reset device if hung up? must be coordinated by user-space...
 *
 * prevent ip fragmentation?
 *
 * Use netif_carrier_{on,off}?
 *
 * DONE:
 *
 * Need to increment ref count on netdev struct when in-flight URB
 *    has pointer to it (context field)? (dev_hold, dev_put)
 *
 *    net_device reference counts (Understanding Linux network internals,
 *       p. 158)
 *      register w/ netdev_chain notification chain
 *      receive NETDEV_UNREGISTER notification
 *      netdev_run_todo calls netdev_wait_allrefs
 *
 * try on ARM
 * CONFIG_DEBUG_SPINLOCK_SLEEP
 * CONFIG_DEBUG_MUTEXES
 *
 * critical section locking
 * TBD: tx ring need spinlock?
 */

#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/version.h>
#include <linux/usb.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/atomic.h>
#include "cicada_meas_drvdefs.h"

#define TRCH_MSG0	0x00000001
#define TRCH_MSG1	0x00000100
#define TRCH_MSG2	0x00000200
#define TRCH_MSG3	0x00000400
#define TRCH_MSG4	0x00000800

static unsigned int get_us(void)
{
	struct timeval tv;
	do_gettimeofday(&tv);
	return ((tv.tv_sec % 1000) * 1000000) + tv.tv_usec;
}


/* tracing can use netdev->priv when invalid.  This checks for trouble. */
static atomic_t netdev_priv_valid = ATOMIC_INIT(0);
#define NETDEV_VALID() (atomic_read(&netdev_priv_valid) ? 1 :	\
			(pr_err("Error: used netdev/priv out of scope %s, line %d\n", __FILE__, __LINE__), 0))

#define ERR_TR(fmt, args...) if (NETDEV_VALID() && *(priv->ptrace_enabled_chans) & TRCH_MSG0) \
				   priv->trace_record(TRCH_MSG0, fmt, ## args)

#define CBTRACE(fmt, args...) if (NETDEV_VALID() && *(priv->ptrace_enabled_chans) & TRCH_MSG1) \
					priv->trace_record(TRCH_MSG1, fmt, ## args)

#define DBG0(fmt, args...) if (NETDEV_VALID() && *(priv->ptrace_enabled_chans) & TRCH_MSG0) \
				     priv->trace_record(TRCH_MSG0, fmt, ## args)

#define DBG1(fmt, args...) if (NETDEV_VALID() && *(priv->ptrace_enabled_chans) & TRCH_MSG1) \
				     priv->trace_record(TRCH_MSG1, fmt, ## args)

#define DBG2(fmt, args...) if (NETDEV_VALID() && *(priv->ptrace_enabled_chans) & TRCH_MSG2) \
				     priv->trace_record(TRCH_MSG2, fmt, ## args)

#define DBG3(fmt, args...) if (NETDEV_VALID() && *(priv->ptrace_enabled_chans) & TRCH_MSG3) \
				     priv->trace_record(TRCH_MSG3, fmt, ## args)

/* very useful for debugging probe/disconnect, etc...
 * Module parameter enable_config_logging enables printk too.
 */
#define	CONFIG(fmt, args...) { \
		if (enable_config_logging) { printk(fmt "\n", ## args); } \
		if (NETDEV_VALID() && *(priv->ptrace_enabled_chans) & TRCH_MSG0) { \
			priv->trace_record(TRCH_MSG0, fmt, ## args); \
		} \
	}

/* use where "priv" is not accessible (no tracing) */
#define	LOG_CONFIG(fmt, args...) { \
		if (enable_config_logging) { printk(fmt "\n", ## args); } \
	}

/* We use a standard ethernet header (2*6 + 2) */
#define MIN_PACKET_DATA (ETH_HLEN + 1)

/* tx timeout in jiffies */
#define RCOMUSB_TIMEOUT (HZ) /* 1 second */

/* number of slots in tx ring buffer */
#define TX_RING_LEN 8

/* larger than largest message; smaller than page size (page size is 4k for
 * both x86 and ARM) */
#define MAX_URB_SIZE 2048

/* This appears to conflict with an existing net driver attribute called speed
   so I (Mike) created ifdefs and disabled it for now */
/*#define ENABLE_CICADA_SPEED_ATTR*/

static int tx_timeout = RCOMUSB_TIMEOUT;

/* the "ethernet" address */
static u8 rcom_dev_addr[ETH_ALEN];

/* there's only one class for all instances */
static struct class *cicada_rcom_class;
#define CICADA_RCOM_SYS_CLASS		"cicada_rcom"
#define CICADA_RCOM_PROC_NAME		"cicada_rcom"
#define CICADA_RCOM_NUM_MINORS		1	/* just one for cicada_rcom_evt */

struct comm_cfg_fields
{
	// Set by IOCTL_SET_CHAN_PRI (Default all = 0)
	uint8_t ChanPriA;    /* Highest Priority Channel # (1-5), LL: Tx port to use   */
	uint8_t ChanPriB;    /* Second highest priority chan #,   LL: Wl source to use */
	uint8_t ChanPriC;    /* Third  highest priority chan #,   LL: Rx port to use   */
	uint8_t ChanPriD;    /* Fourth highest priority chan # */
	uint8_t ChanPriE;    /* Lowest priority chan #         */

	// Set by IOCTL_SET_BAUD (Default = 0)
	uint32_t Baud;       /* LL: rate in bps for comm channel */
			     /* CU: 1950000 = fast (default), 85000 = slow (long reach) */
};
#define NUM_CHAN_PRI 5

struct evt_info;

struct rcom_priv {
	struct net_device *netdev;
	struct usb_device *usb_dev;
	struct usb_interface *usb_if;
	struct brb *tx_resp_brb;
	struct comm_cfg_fields comm_cfg;
	unsigned int model;
	unsigned int route;
	unsigned int bulk_in_endpointAddr;
	unsigned int bulk_out_endpointAddr;
	int response_success;
	int busy;
	int rx_pkt_retries;
	struct workqueue_struct *workq;
	struct urb *tx_ring[TX_RING_LEN];
	unsigned int write_ix;
	unsigned int read_ix;
	unsigned int pending_tx_bytes;
	struct work_struct work;
	struct semaphore sem; /* held when a command/response transaction with the
				 device is under way */
	wait_queue_head_t wq;
	struct net_device_stats stats;
	spinlock_t lock;
	uint8_t seqnum;
	uint8_t rx_pkt_waiting; /* device has received a packet, needs to be read */
	uint8_t tx_stopped; /* netif_stop_queue() has been called */
	int  (*bulk_submit_snd_urb)(void *mmstate, struct urb *urb,
				    gfp_t mem_flags, int routing_code);
	int  (*bulk_submit_rcv_brb)(struct brb *brb);
	void *mmstate;
	void (*cicada_meas_state_get)(void* mmstate);
	void (*cicada_meas_state_put)(void* mmstate);

	/* cicada_meas trace function and channel settings for unified tracing */
	uint32_t* ptrace_enabled_chans;
	void (*trace_record)(uint32_t chan, const char* fmt, ...);

	/* keep track of prev time and max delta time between rx pkt callback and
	   completion of readMesg USB command */
	unsigned int rxcb_intep_time;
	unsigned int rxcb_max_delta;

	dev_t devnum;
	struct device *chardev;
	struct evt_info *evtinfo;
};

static void fill_cicada_std_header(uint8_t *buf, struct rcom_priv *priv,
		uint32_t transfer_size,
		uint16_t message_id);
static void fill_cicada_sendmesg_cmd(uint8_t *buf, struct rcom_priv *priv,
		int len);
static void fill_cicada_readmesg_cmd(uint8_t *buf, struct rcom_priv *priv,
		int maxlen);
static void fill_cicada_configcomm_cmd(uint8_t *buf, struct rcom_priv *priv,
		int enable);

static int cicada_readmesg(struct net_device *netdev);
static int cicada_sendmesg(struct net_device *netdev, struct urb *tx_urb);
static int cicada_configcomm(struct net_device *netdev, int enable);

static void sendmesg_cmd_cb(struct urb *urb);
static void sendmesg_resp_cb(struct brb *brb);
static void readmesg_cmd_cb(struct urb *urb);
static void readmesg_resp_cb(struct brb *brb);
static void configcomm_cmd_cb(struct urb *urb);
static void std_resp_cb(struct brb *brb);
static void int_ep_cb(struct urb *int_urb, void *vnetdev);

static void restart_rx_tx(struct work_struct *work);

static int tx_ring_init(struct net_device *netdev);
static int tx_ring_write(struct net_device *netdev, struct sk_buff *skb);
static int tx_ring_read(struct net_device *netdev, struct urb **urb);

static int rcom_enable(struct net_device *netdev, int enable);
static int rcom_open(struct net_device *netdev);
static int rcom_stop(struct net_device *netdev);
static int rcom_ioctl(struct net_device *netdev, struct ifreq *req, int cmd);
/*static int rcom_config(struct net_device *dev, struct ifmap *map);*/
static int rcom_tx(struct sk_buff *skb, struct net_device *netdev);
static void rcom_tx_timeout(struct net_device *dev);
static struct net_device_stats *rcom_stats(struct net_device *dev);
static void rcom_uninit(struct net_device *netdev);
static void rcom_destructor(struct net_device *netdev);
static void rcom_cleanup(void);
static void rcom_init(struct net_device *netdev);

static int enable_config_logging = 0;

int cicada_rcom_probe(struct usb_device *usbdev, struct usb_interface *usbif,
		unsigned int bulk_in_endpointAddr,
		unsigned int bulk_out_endpointAddr,
		int *route,
		void (**cb)(struct urb *, void *),
		void **pnetdev,
		int  (*bulk_submit_snd_urb)(void *mmstate, struct urb *urb,
					    gfp_t mem_flags, int routing_code),
		int  (*bulk_submit_rcv_brb)(struct brb *brb),
		void *mmstate,
		void (*cicada_meas_state_get)(void* mmstate),
		void (*cicada_meas_state_put)(void* mmstate),
		uint32_t* ptrace_enabled_chans,
		void (*trace_record)(uint32_t chan, const char* fmt, ...),
		int enable_config_logging_arg
	    );
void cicada_rcom_disconnect(void *vnetdev);


/*
 * evt_info - evt related info
 *
 * This info must have a lifetime that is independent of the netdev struct and
 * its embedded "priv" area.  When the module is unplugged, cicada_rcom_disconnect()
 * will clean up as much as it can, including unregistering the netdev struct.
 */

typedef void (*cicada_meas_state_get_func)(void*);
typedef void (*cicada_meas_state_put_func)(void*);

struct evt_info
{
	struct kref kref;
	struct list_head active_list;
	struct cdev   *cdev;
	char *evtmsg;   /* event interrupt message */
	uint8_t evtint; /* device has received an event interrupt, ready to be read */
	struct mutex file_mutex, data_mutex;
	wait_queue_head_t evtwq;  /* for blocking read */

	void* mmstate;
	void (*cicada_meas_state_put)(void* mmstate);
};

static void remove_evt_info_active(struct evt_info *evtinfo);

static struct evt_info* evt_info_create(void* mmstate,
					cicada_meas_state_get_func measStateGetFunc,
					cicada_meas_state_put_func measStatePutFunc)
{
	struct evt_info *evtinfo = kzalloc(sizeof(struct evt_info), GFP_KERNEL);
	if (!evtinfo) {
		return NULL;
	}

	// this object is ref counted - count initialized to 1.
	kref_init(&evtinfo->kref);

	// embedded list node to support the "evt_info_active list"
	INIT_LIST_HEAD(&evtinfo->active_list);

	/* increment ref count on cicada_meas driver's state struct */
	(*measStateGetFunc)(mmstate);

	/* save, so we can decrement the ref count in evt_info_delete() */
	evtinfo->mmstate = mmstate;
	evtinfo->cicada_meas_state_put = measStatePutFunc;

	mutex_init(&evtinfo->file_mutex);
	mutex_init(&evtinfo->data_mutex);
	init_waitqueue_head(&evtinfo->evtwq);
	evtinfo->evtmsg = NULL;
	evtinfo->evtint = 0;

	LOG_CONFIG("evt_info_create(): evt_info = %p", evtinfo);
	return evtinfo;
}

/* called when evt_info ref count is decremented to zero */
static void evt_info_delete(struct kref *kref)
{
	struct evt_info *evtinfo = container_of(kref, struct evt_info, kref);
	cicada_meas_state_put_func measStatePutFunc = evtinfo->cicada_meas_state_put;
	void* mmstate = evtinfo->mmstate;

	LOG_CONFIG("evt_info_delete(): evt_info = %p", evtinfo);
	remove_evt_info_active(evtinfo);
	kfree(evtinfo);

	/* decrement ref count on cicada_meas driver's state struct */
	measStatePutFunc(mmstate);
}


/* evt_open() must use get_evt_info_active() to get the relevant
 * evt_info struct. As a result,  we must maintain this evt_info
 * infrastructure.
 */
struct active_evt_info {
	struct list_head	active_list;
	struct mutex		mutex;   /* protect list and shared data */
};

struct active_evt_info active_evtinfo;

/* init "active" list when module is loaded */
static void init_evt_info_active(void)
{
	INIT_LIST_HEAD(&active_evtinfo.active_list);
	mutex_init(&active_evtinfo.mutex);
}

static void add_evt_info_active(struct evt_info *evtinfo)
{
	LOG_CONFIG("add_evt_info_active() - evtinfo = %p", evtinfo);
	mutex_lock(&active_evtinfo.mutex);
	list_add_tail(&evtinfo->active_list, &active_evtinfo.active_list);
	mutex_unlock(&active_evtinfo.mutex);
}

static void remove_evt_info_active(struct evt_info *evtinfo)
{
	struct evt_info *list_evtinfo = NULL;
	struct list_head *ptr = NULL;
	struct list_head *next = NULL;

	mutex_lock(&active_evtinfo.mutex);
	list_for_each_safe(ptr, next, &active_evtinfo.active_list) {
		list_evtinfo = list_entry(ptr, struct evt_info, active_list);
		if (list_evtinfo && list_evtinfo == evtinfo) {
			LOG_CONFIG("remove_evt_info_active: evtinfo = %p", evtinfo);
			list_del(&(evtinfo->active_list));
			break;
		}
	}
	mutex_unlock(&active_evtinfo.mutex);
}

static struct evt_info* get_evt_info_active(struct cdev* cdev)
{
	struct evt_info *list_evtinfo = NULL;
	struct list_head *ptr = NULL;

	mutex_lock(&active_evtinfo.mutex);
	list_for_each(ptr, &active_evtinfo.active_list) {
		list_evtinfo = list_entry(ptr, struct evt_info, active_list);
		if (list_evtinfo && list_evtinfo->cdev == cdev) {
			mutex_unlock(&active_evtinfo.mutex);
			LOG_CONFIG("get_evt_info_active(): list_evtinfo = %p", list_evtinfo);
			return list_evtinfo;
		}
	}
	mutex_unlock(&active_evtinfo.mutex);
	LOG_CONFIG("get_evt_info_active(): evtinfo not found, return NULL");
	return NULL;
}



static void fill_cicada_std_header(uint8_t *buf, struct rcom_priv *priv,
		uint32_t TransferSize,
		uint16_t MsgID)
{
	struct USB_MSG_HEADER *hdr = (struct USB_MSG_HEADER *)buf;

	hdr->MagicNum = CICADA_CMD_MAGIC;

	/* net driver has its own private sequence number (i.e. not shared
	 * with command driver).  The device doesn't do anything with this
	 * except copy it into the associated response. */
	hdr->SeqNum = priv->seqnum++;
	hdr->SeqInv = ~hdr->SeqNum;
	hdr->RouteID = ROUTE_NET;
	hdr->TransferSize = cpu_to_le32(TransferSize);
	hdr->Reserved = 0;
	hdr->MsgID = cpu_to_le16(MsgID);
}

static void fill_cicada_sendmesg_cmd(uint8_t *buf, struct rcom_priv *priv,
		int len)
{
	struct USB_CMD_SENDMESG *hdr = (struct USB_CMD_SENDMESG *)buf;
	uint32_t TransferSize = sizeof(struct USB_CMD_SENDMESG) + len;

	DBG3("sendmesg: len: %d", TransferSize);
	DBG3("sendmesg: TransferSize (unpadded): %d", TransferSize);

	/* data must be zero-padded to multiple of 4 bytes */
	if (len % 4)
		TransferSize += (4 - (len % 4));

	DBG3("sendmesg: TransferSize (padded): %d", TransferSize);

	fill_cicada_std_header(buf, priv, TransferSize, CMD_CODE_SENDMESG);

	hdr->Spare1 = 0;
	hdr->Length = cpu_to_le32(len);
}

static void fill_cicada_readmesg_cmd(uint8_t *buf, struct rcom_priv *priv,
		int maxlen)
{
	struct USB_CMD_READMESG *hdr = (struct USB_CMD_READMESG *)buf;
	uint32_t TransferSize = sizeof(struct USB_CMD_READMESG);

	fill_cicada_std_header(buf, priv, TransferSize, CMD_CODE_READMESG);

	hdr->Spare1 = 0;
	hdr->MaxLength = cpu_to_le32(maxlen);
}

static void fill_cicada_configcomm_cmd(uint8_t *buf, struct rcom_priv *priv,
		int enable)
{
	struct USB_CMD_CONFIGMESG *hdr =
		(struct USB_CMD_CONFIGMESG *)buf;
	uint32_t TransferSize = sizeof(struct USB_CMD_CONFIGMESG);
	uint16_t comm_flags = 0;

	fill_cicada_std_header(buf, priv, TransferSize, CMD_CODE_CONFIGCOMM);

	if (enable) {
		comm_flags |= CMD_FLAG_MESGENA;  // Same as CMD_FLAG_LL_TX_EN
		comm_flags |= CMD_FLAG_LL_RX_EN; // RX is enabled separately on LL

		/* Nautilus does not yet support LONGREACH */
		if (priv->model >= 300 && priv->model < 400 &&
			priv->comm_cfg.Baud < 150000)
			comm_flags |= CMD_FLAG_LONGREACH;

		hdr->CommFlags = cpu_to_le16(comm_flags);

		hdr->ChanPriA = priv->comm_cfg.ChanPriA;
		hdr->ChanPriB = priv->comm_cfg.ChanPriB;
		hdr->ChanPriC = priv->comm_cfg.ChanPriC;
		hdr->ChanPriD = priv->comm_cfg.ChanPriD;
		hdr->ChanPriE = priv->comm_cfg.ChanPriE;

		hdr->Baud = priv->comm_cfg.Baud / 1000; // Convert to Kbps
	}
	else {
		hdr->CommFlags = 0;
		hdr->ChanPriA  = 0;
		hdr->ChanPriB  = 0;
		hdr->ChanPriC  = 0;
		hdr->ChanPriD  = 0;
		hdr->ChanPriE  = 0;
		hdr->Baud      = 0;
	}
}

/*
 * TX Ring Buffer
 *
 * Full when: write_ix + 1 == read_ix
 *
 * Empty when: write_ix == read_ix
 *
 * When ring becomes empty, start stack tx again (netif_wake_queue(netdev)).
 *
 * When ring becomes full, stop stack tx.
 *
 * In "full" ring, "extra" slot (one pointed to by write_ix) may be in process of
 * being transmitted.
 *
 * Relies on tx_ring_read only being called when it is okay to tx, since there
 * is no inherent check on whether tx is currently active.
 */

static inline int tx_ring_empty(struct rcom_priv *priv)
{
	return (priv->write_ix == priv->read_ix) ? 1 : 0 ;
}

static inline int tx_ring_full(struct rcom_priv *priv)
{
	return ((priv->write_ix + 1) % TX_RING_LEN == priv->read_ix) ? 1 : 0;
}

static int tx_ring_init(struct net_device *netdev)
{
	int i;
	struct rcom_priv *priv = netdev_priv(netdev);
	struct urb *tx_urb = NULL;
	uint8_t *tx_urb_buf = NULL;

	for (i = 0; i < TX_RING_LEN; i++) {
		tx_urb = usb_alloc_urb(0, GFP_KERNEL);
		tx_urb_buf = kmalloc(MAX_URB_SIZE, GFP_KERNEL);
		tx_urb->transfer_buffer = tx_urb_buf;
		tx_urb->transfer_flags |= URB_FREE_BUFFER;
		tx_urb->transfer_flags |= URB_ZERO_PACKET;
		priv->tx_ring[i] = tx_urb;
	}
	priv->write_ix = 0;
	priv->read_ix = 0;

	return 0;
}

static void tx_ring_free(struct net_device *netdev)
{
	int i;
	struct rcom_priv *priv = netdev_priv(netdev);

	for (i = 0; i < TX_RING_LEN; i++) {
		usb_free_urb(priv->tx_ring[i]);
		priv->tx_ring[i] = NULL;
	}
}

/*
 * Takes the given packet and copies it into the ring buffer for later
 * transmission.
 *
 * In all cases, the caller retains ownership of the skb.
 *
 * Returns: 0: packet was successfully copied into ring.
 *         -1: ring was full, packet not copied.
 *
 */
static int tx_ring_write(struct net_device *netdev, struct sk_buff *skb)
{
	struct rcom_priv *priv = netdev_priv(netdev);
	int pkt_len = 0;
	char *pkt_data = NULL;
	int pad_len = 0;
	int urb_buf_len = 0;
	struct urb *tx_urb = NULL;
	uint8_t *tx_urb_buf = NULL;
	unsigned long irqflags;

	pkt_data = skb->data;
	pkt_len = skb->len;

	DBG3("in tx_ring_write, pkt_len: %d", pkt_len);

	/* make sure we're not being asked to do scatter/gather I/O */
	BUG_ON(skb->data_len != 0);

	spin_lock_irqsave(&priv->lock, irqflags);

	/* check for full ring */
	if (tx_ring_full(priv)) {
		spin_unlock_irqrestore(&priv->lock, irqflags);
		return -1; /* FULL */
	}

	/*
	 * copy packet data into URB pointed to by the write_ix
	 */
	tx_urb = priv->tx_ring[priv->write_ix];

	priv->write_ix = (priv->write_ix + 1) % TX_RING_LEN;

	spin_unlock_irqrestore(&priv->lock, irqflags);

	tx_urb_buf = (uint8_t *) tx_urb->transfer_buffer;
	fill_cicada_sendmesg_cmd(tx_urb_buf, priv, pkt_len);
	memcpy(tx_urb_buf + sizeof(struct USB_CMD_SENDMESG), pkt_data, pkt_len);

	/* zero-pad data to multiple of 4 bytes as required by module firmware */
	if (pkt_len % 4) {
		pad_len = 4 - (pkt_len % 4);
		memset(tx_urb_buf + sizeof(struct USB_CMD_SENDMESG) + pkt_len,
				0, pad_len);
	}
	urb_buf_len = sizeof(struct USB_CMD_SENDMESG) + pkt_len + pad_len;
	DBG3("in tx_ring_write, pad_len: %d", pad_len);
	DBG3("in tx_ring_write, urb_buf_len: %d", urb_buf_len);

	usb_fill_bulk_urb(tx_urb, priv->usb_dev,
			usb_sndbulkpipe(priv->usb_dev, priv->bulk_out_endpointAddr),
			tx_urb_buf, urb_buf_len, sendmesg_cmd_cb, netdev);

	return 0;
}

/*
 * Reads a packet (actually a URB containing the packet data) from the ring
 * buffer and returns it so it can be sent to the device for transmission.
 *
 * Returns: 0: packet successfully obtained from ring
 *         -1: ring was empty
 */
static int tx_ring_read(struct net_device *netdev, struct urb **urb)
{
	struct rcom_priv *priv = netdev_priv(netdev);
	unsigned long irqflags;

	spin_lock_irqsave(&priv->lock, irqflags);

	if (priv->write_ix == priv->read_ix) {
		spin_unlock_irqrestore(&priv->lock, irqflags);
		return -1; /* EMPTY */
	}
	*urb = priv->tx_ring[priv->read_ix];
	priv->read_ix = (priv->read_ix + 1) % TX_RING_LEN;

	spin_unlock_irqrestore(&priv->lock, irqflags);

	return 0;
}

static void configcomm_cmd_cb(struct urb *urb)
{
#if 0
	if (urb->status &&
			!(urb->status == -ENOENT ||
				urb->status == -ECONNRESET ||
				urb->status == -ESHUTDOWN)) {
		usb_free_urb(urb);
	}
#endif
	if (urb->status)
		pr_warning("in configcomm_cmd_cb(), status=%d\n", urb->status);

	dev_put((struct net_device *)urb->context);
	usb_free_urb(urb);
}

/* completion handler for the Configure Communications response URB,
 * as well as any others (Start Comm, Stop Comm) which expect the
 * standard response.
 *
 * We get a standard Command Completion Report from the device.
 */
static void std_resp_cb(struct brb *brb)
{
	struct net_device *netdev = brb->context;
	struct rcom_priv *priv = netdev_priv(netdev);
	struct USB_RESP_STD *resp;

	if (brb->status) {
		priv->response_success = 0;
		pr_warning("in std_resp_cb(), status=%d\n", brb->status);
	}
	else {
		priv->response_success = 1;

		resp = (struct USB_RESP_STD *)brb->transfer_buffer;

		BUG_ON(resp->Head.MagicNum != CICADA_CMD_MAGIC);
		if (resp->Head.RouteID != ROUTE_NET ||
			le32_to_cpu(resp->Head.TransferSize) != 16 ||
			le16_to_cpu(resp->Head.MsgID) != DEV_RESP_STD ||
			le16_to_cpu(resp->CCStat) != CMD_STAT_OK) {

			dev_err(&netdev->dev, "*** Unexpected response in std_resp_cb\n");
			dev_err(&netdev->dev, "   routing id: %d\n", resp->Head.RouteID);
			dev_err(&netdev->dev, "   TransferSize (16): %d\n",
					le32_to_cpu(resp->Head.TransferSize));
			dev_err(&netdev->dev,
					"   MsgID (0x%x): 0x%x\n", DEV_RESP_STD,
					le16_to_cpu(resp->Head.MsgID));
			dev_err(&netdev->dev, "   CCStat (%d): %d\n", CMD_STAT_OK,
					le16_to_cpu(resp->CCStat));
		}

		DBG3("  routing id (2?): %d", resp->Head.RouteID);
		DBG3("  transfer size (16?): %d", le32_to_cpu(resp->Head.TransferSize));
		DBG3("  message id(320?): %d", le16_to_cpu(resp->Head.MsgID));
		DBG3("  ccstat: 0x%04x", le16_to_cpu(resp->CCStat));
		DBG3("  short stat: 0x%04x", le16_to_cpu(resp->OptionalDebug));
	}

	priv->busy = 0;
	up(&priv->sem);
	wake_up_interruptible(&priv->wq);

	dev_put((struct net_device *)brb->context);
	brb_free(brb);
}

/* allocate, fill out and submit URBs for Configure Communications command and
 * response.
 */
static int cicada_configcomm(struct net_device *netdev, int enable)
{
	int ret;
	struct urb *cmd_urb;
	void *cmd_urb_buf;
	struct brb *resp_brb;
	void *resp_brb_buf;
	struct rcom_priv *priv;

	priv = netdev_priv(netdev);
	priv->response_success = 0;
	CONFIG("cicada_configcomm: enable=%d", enable);

	/* allocate urb for command */
	cmd_urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!cmd_urb)
		return -1;

	/* allocate buffer for command */
	cmd_urb_buf = kmalloc(MAX_URB_SIZE, GFP_KERNEL);
	if (!cmd_urb_buf)
		return -1;
	cmd_urb->transfer_buffer = cmd_urb_buf;
	cmd_urb->transfer_flags |= URB_FREE_BUFFER;
	cmd_urb->transfer_flags |= URB_ZERO_PACKET;

	/* fill command */
	fill_cicada_configcomm_cmd(cmd_urb_buf, priv, enable);

	/* fill urb */
	usb_fill_bulk_urb(cmd_urb, priv->usb_dev,
			usb_sndbulkpipe(priv->usb_dev, priv->bulk_out_endpointAddr),
			cmd_urb_buf, sizeof(struct USB_CMD_CONFIGMESG),
			configcomm_cmd_cb, netdev);
	/* submit urb */
	dev_hold(netdev);
	ret = priv->bulk_submit_snd_urb(priv->mmstate, cmd_urb, GFP_KERNEL, ROUTE_NET);
	if (ret) {
		pr_notice("cicada_configcomm: line %d bulk_submit_snd_urb returned %d\n",
				__LINE__, ret);
		usb_free_urb(cmd_urb);
		dev_put(netdev);
		return -1;
	}

	/* allocate brb and buffer for response, then init */
	resp_brb = kmalloc(sizeof(struct brb), GFP_KERNEL);
	resp_brb_buf = kmalloc(MAX_URB_SIZE, GFP_KERNEL);
	brb_init(resp_brb, resp_brb_buf, MAX_URB_SIZE, std_resp_cb, priv->mmstate, netdev, ROUTE_NET);

	/* submit brb */
	dev_hold(netdev);
	ret = priv->bulk_submit_rcv_brb(resp_brb);
	if (ret) {
		pr_notice("cicada_configcomm: line %d bulk_submit_rcv_urb returned %d\n",
				__LINE__, ret);
		brb_free(resp_brb);
		dev_put(netdev);
		return -1;
	}
	return 0;
}

static int rcom_enable(struct net_device *netdev, int enable)
{
	struct rcom_priv *priv;
	priv = netdev_priv(netdev);

	if (down_interruptible(&priv->sem) != 0)
		return -1;

	/* TBD - we do this just to have something to give to
	 * wait_event_interruptible()... investigate whether this is really
	 * the best way to do things.
	 */
	priv->busy = 1;

	/* send "config communications" command to device, enable or disable packet
	 * reception. */
	if (cicada_configcomm(netdev, enable)) {
		pr_warning("rcom_enable: cicada_configcomm() returned error\n");
		priv->busy = 0;
		up(&priv->sem);
		return -1;
	}

	if (wait_event_interruptible(priv->wq, priv->busy == 0)) {
		/*up(&priv->sem);*/
		return -ERESTARTSYS;
	}
	/*up(&priv->sem);*/
	if (!priv->response_success) {
		return -1;
	}
	return 0;
}

/* rcom_open - called in response to ifconfig up
 *
 * Send "config communications" command to device to enable communications,
 * sleep waiting for the response.
 *
 * After returning, the kernel will set the IFF_UP bit in dev->flags.
 */
static int rcom_open(struct net_device *netdev)
{
	int retval = 0;
	struct rcom_priv *priv;
	priv = netdev_priv(netdev);

	DBG2("in rcom_open");

	/* Set some defaults */
	if (priv->model >= 200 && priv->model < 300) {
		priv->comm_cfg.Baud = 200000;
		priv->comm_cfg.ChanPriB = 2;

		if (priv->model == 202) {
		    priv->comm_cfg.ChanPriA = 2;
		}
		else {
		    priv->comm_cfg.ChanPriA = 1;
		}
	}
	else if (priv->model >= 300 && priv->model < 400) {
		priv->comm_cfg.Baud = 1950000;
	}
	else if (priv->model >= 400 && priv->model < 500) {
		/* Nautilus currently ignores baud rate in config comm command. */
		priv->comm_cfg.Baud = 0;
	}

	retval = rcom_enable(netdev, 1);
	if (retval == 0) {
		netif_start_queue(netdev);
	}

	return retval;
}

/* rcom_stop - called in response to ifconfig down
 *
 * Send "config communications" command to device to disable receiver, sleep
 * waiting for the response.
 *
 * After returning, the kernel will clear the IFF_UP bit in dev->flags. */
static int rcom_stop(struct net_device *netdev)
{
	struct rcom_priv *priv = netdev_priv(netdev);
	DBG2("in rcom_stop");

	/* TBD pp. 516 and 518 in the linux drivers book seem to contradict
	 * each other: use netif_tx_disable() or netif_stop_queue() when
	 * outside driver tx function? */
	netif_stop_queue(netdev);
	/*netif_tx_disable(netdev);*/

	/* TBD - this can be called either as a result of user-space doing an
	 * "ifconfig down", or from rcom_disconnect as a result of the device
	 * being unplugged.  In the latter case, it does not make sense to try
	 * to tell the device to disable communications. */

	return rcom_enable(netdev, 0);
}

/* allocate, fill out and submit URBs for generic CMD_CODE command and
 * standard response. (not for use with cmd_codes that have args or
 * specialized responses)
 *
 * Currently just used for CMD_CODE_START_COMM and STOP_COMM
 *
 */
static int cicada_sendcmd(struct net_device *netdev, uint16_t cmdCode)
{
	int ret;
	struct urb *cmd_urb;
	void *cmd_urb_buf;
	struct brb *resp_brb;
	void *resp_brb_buf;
	struct rcom_priv *priv;
	uint32_t TransferSize = sizeof(struct USB_MSG_HEADER);

	priv = netdev_priv(netdev);
	priv->response_success = 0;
	DBG2("cicada_sendcmd: cmdCode=%d", cmdCode);

	/* allocate urb for command */
	cmd_urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!cmd_urb)
		return -1;

	/* allocate buffer for command */
	cmd_urb_buf = kmalloc(MAX_URB_SIZE, GFP_KERNEL);
	if (!cmd_urb_buf)
		return -1;
	cmd_urb->transfer_buffer = cmd_urb_buf;
	cmd_urb->transfer_flags |= URB_FREE_BUFFER;
	cmd_urb->transfer_flags |= URB_ZERO_PACKET;

	/* fill command */
	fill_cicada_std_header(cmd_urb_buf, priv, TransferSize, cmdCode);

	/* fill urb */
	usb_fill_bulk_urb(cmd_urb, priv->usb_dev,
			usb_sndbulkpipe(priv->usb_dev, priv->bulk_out_endpointAddr),
			cmd_urb_buf, sizeof(struct USB_MSG_HEADER),
			configcomm_cmd_cb, netdev);
	/* submit urb */
	dev_hold(netdev);
	ret = priv->bulk_submit_snd_urb(priv->mmstate, cmd_urb, GFP_KERNEL, ROUTE_NET);
	if (ret) {
		pr_notice("cicada_sendcmd: line %d bulk_submit_snd_urb returned %d\n",
				__LINE__, ret);
		usb_free_urb(cmd_urb);
		dev_put(netdev);
		return -1;
	}

	/* allocate brb and buffer for response, then init */
	resp_brb = kmalloc(sizeof(struct brb), GFP_KERNEL);
	resp_brb_buf = kmalloc(MAX_URB_SIZE, GFP_KERNEL);
	brb_init(resp_brb, resp_brb_buf, MAX_URB_SIZE, std_resp_cb, priv->mmstate, netdev, ROUTE_NET);

	/* submit brb */
	dev_hold(netdev);
	ret = priv->bulk_submit_rcv_brb(resp_brb);
	if (ret) {
		pr_notice("cicada_sendcmd: line %d bulk_submit_rcv_urb returned %d\n",
				__LINE__, ret);
		brb_free(resp_brb);
		dev_put(netdev);
		return -1;
	}
	return 0;
}

// Send command that consists of just a command code
// Currently used for Start Comm and Stop Comm
static int rcom_send_cmd_code(struct net_device *netdev, uint16_t cmdCode)
{
	struct rcom_priv *priv;
	priv = netdev_priv(netdev);

	if (down_interruptible(&priv->sem) != 0)
		return -1;

	/* TBD - we do this just to have something to give to
	 * wait_event_interruptible()... investigate whether this is really
	 * the best way to do things.
	 */
	priv->busy = 1;

	if (cicada_sendcmd(netdev, cmdCode)) {
		pr_warning("rcom_enable: cicada_sendcmd() returned error\n");
		priv->busy = 0;
		up(&priv->sem);
		return -1;
	}

	if (wait_event_interruptible(priv->wq, priv->busy == 0)) {
		/*up(&priv->sem);*/
		return -ERESTARTSYS;
	}
	/*up(&priv->sem);*/
	if (!priv->response_success) {
		return -1;
	}
	return 0;
}

/* rcom_ioctl - called in response to device specific ioctl commands
 *
 *  SIOCDEVPRIVATE is deprecated, may look at alternatives.
 *  Possible cmds are SIOCDEVPRIVATE+0 to +15
 *
 *  Cmd data should be stored in the ifreq->ifr_data field.
 *
 *  Returns 0 for success. */
static int rcom_ioctl(struct net_device *netdev, struct ifreq *req, int cmd)
{
	struct rcom_priv *priv;
	priv = netdev_priv(netdev);

	// Determine which iotctl command was received
	switch (cmd) {
	case IOCTL_SET_CHAN_PRI: // Set Chan Pri Fields
	{
		uint8_t chan_pri[NUM_CHAN_PRI];

		// Copy cmd argument from user space.
		if (copy_from_user(&chan_pri, req->ifr_data, sizeof(uint8_t) * NUM_CHAN_PRI)) {
			return -EFAULT;
		}

		// Ignore unnecessary updates
		if (priv->comm_cfg.ChanPriA == chan_pri[0] &&
			priv->comm_cfg.ChanPriB == chan_pri[1] &&
			priv->comm_cfg.ChanPriC == chan_pri[2] &&
			priv->comm_cfg.ChanPriD == chan_pri[3] &&
			priv->comm_cfg.ChanPriE == chan_pri[4]) {
			return 0;
		}

		priv->comm_cfg.ChanPriA = chan_pri[0];
		priv->comm_cfg.ChanPriB = chan_pri[1];
		priv->comm_cfg.ChanPriC = chan_pri[2];
		priv->comm_cfg.ChanPriD = chan_pri[3];
		priv->comm_cfg.ChanPriE = chan_pri[4];

		break;
	}

	case IOCTL_SET_BAUD: // Set Baud
	{
		uint32_t baud;

		// Copy Baud from user space
		if (copy_from_user(&baud, req->ifr_data, sizeof(uint32_t))) {
			return -EFAULT;
		}

		// Ignore unnecessary updates
		if (priv->comm_cfg.Baud == baud) {
			return 0;
		}

		priv->comm_cfg.Baud = baud;

		break;
	}

	case IOCTL_DISABLE_COMM: // Disable Device communications
		priv->comm_cfg.Baud = 0;
		priv->comm_cfg.ChanPriA = 0;
		priv->comm_cfg.ChanPriB = 0;
		priv->comm_cfg.ChanPriC = 0;
		priv->comm_cfg.ChanPriD = 0;
		priv->comm_cfg.ChanPriE = 0;

		return rcom_enable(netdev, 0);

		break;

	case IOCTL_START_COMM: // Start Device communications - Mantis

		return rcom_send_cmd_code(netdev, CMD_CODE_START_COMM);

		break;

	case IOCTL_STOP_COMM: // Stop Device communications - Mantis

		return rcom_send_cmd_code(netdev, CMD_CODE_STOP_COMM);

		break;

	default:
		return -EINVAL; // Unknown command, indicate failure
	}

	// Enable RCOM with updated parameters
	return rcom_enable(netdev, 1);
}


#if 0
/* TBD - never used? */
static int rcom_config(struct net_device *dev, struct ifmap *map)
{
	pr_err("********* rcom_config called!!!!\n");

	return -EOPNOTSUPP;
}
#endif

/* submit URBs for SendMesg command and response
 *
 * This function should only be called when holding the priv->sem mutex.
 */
static int cicada_sendmesg(struct net_device *netdev, struct urb *tx_urb)
{
	int retval;
	void *tx_urb_buf;
	struct USB_CMD_SENDMESG *hdr;
	struct rcom_priv *priv = netdev_priv(netdev);

	/* must save packet length here so that we can update statistics when
	 * the tx is complete in sendmesg_resp_cb(). */
	tx_urb_buf = tx_urb->transfer_buffer;
	hdr = (struct USB_CMD_SENDMESG *)tx_urb_buf;
	priv->pending_tx_bytes = le32_to_cpu(hdr->Length);

	dev_hold(netdev);
	retval = priv->bulk_submit_snd_urb(priv->mmstate, tx_urb, GFP_ATOMIC, ROUTE_NET);
	if (retval) {
		pr_notice("cicada_rcom: line %d bulk_submit_snd_urb returned %d\n",
				__LINE__, retval);
		dev_put(netdev);
		if (retval == -ENODEV)
			return retval;
	}

	/* submit brb for response */
	dev_hold(netdev);
	retval = priv->bulk_submit_rcv_brb(priv->tx_resp_brb);
	if (retval) {
		pr_notice("cicada_rcom: line %d bulk_submit_rcv_urb returned %d\n",
				__LINE__, retval);
		dev_put(netdev);
		if (retval == -ENODEV)
			return retval;
	}

	return 0;
}

/* work queue function */
static void restart_rx_tx(struct work_struct *work)
{
	struct rcom_priv *priv = container_of(work, struct rcom_priv, work);
	struct net_device *netdev = priv->netdev;
	struct urb *tx_urb = NULL;
	int retval;

	down(&priv->sem);

	/* important to give rx a chance first */
	if (priv->rx_pkt_waiting) {
		CBTRACE("in restart_rx_tx, calling cicada_readmesg");
		priv->rx_pkt_waiting = 0;
		retval = cicada_readmesg(netdev);

		if (retval < 0) {
			up(&priv->sem);
		}

		return;
	}

	if (tx_ring_read(netdev, &tx_urb) == 0) {
		CBTRACE("in restart_rx_tx, calling cicada_sendmesg");
		retval = cicada_sendmesg(netdev, tx_urb);

		if (retval < 0) {
			up(&priv->sem);
			return;
		}

		if (priv->tx_stopped && tx_ring_empty(priv)) {
			priv->tx_stopped = 0;
			netif_wake_queue(netdev);
		}

		return;
	}

	CBTRACE("in restart_rx_tx, doing nothing");

	/* neither condition was true, so we didn't do anything, so
	 * reset busy flag.  If either rx or tx was done (above), their
	 * completion callback will call up(). */
	up(&priv->sem);
}

/*
 * Completion handler for URBs submitted for sending the SendMesg
 * command.  When this function is called, the usb core has finished
 * sending the command to the device, and we can free the URB.
 */
static void sendmesg_cmd_cb(struct urb *urb)
{
	struct rcom_priv *priv = netdev_priv((struct net_device *)urb->context);
	CBTRACE("urb completion: sendmesg_cmd");

	/* check status of URB */
	if (urb->status)
		pr_warning("in sendmesg_cmd_cb(), status=%d\n", urb->status);
	dev_put((struct net_device *)urb->context);
}

/*
 * Completion handler for URBs submitted to receive the response to a SendMesg
 * command.  When this function is called, the usb core has received the
 * response from the device, indicating that the device has finished
 * transmitting the packet.  If there are any more packets waiting to be sent,
 * we can send one now.  If we had called netif_stop_queue() before, and there
 * are now no more queued packets, we can call netif_wake_queue().
 */
static void sendmesg_resp_cb(struct brb *brb)
{
	struct net_device *netdev = brb->context;
	struct rcom_priv *priv = netdev_priv(netdev);
	struct USB_RESP_STD *resp;

	CBTRACE("urb completion: sendmesg_resp");

	/* check status of URB */
	if (brb->status)
		pr_warning("in sendmesg_resp_cb(), status=%d\n", brb->status);

	resp = (struct USB_RESP_STD *)brb->transfer_buffer;

	if (resp->Head.MagicNum != CICADA_CMD_MAGIC
			|| resp->Head.RouteID != ROUTE_NET
			|| le16_to_cpu(resp->Head.MsgID) != DEV_RESP_STD) {
		pr_err("*** Unexpected response in sendmesg_resp_cb\n");
		pr_err("  MagicNum (0x%x): 0x%x\n", CICADA_CMD_MAGIC, resp->Head.MagicNum);
		pr_err("  routing id %d\n", resp->Head.RouteID);
		pr_err("  message id (0x%x) 0x%x\n", DEV_RESP_STD,
				le16_to_cpu(resp->Head.MsgID));
	} else if (le16_to_cpu(resp->CCStat) != CMD_STAT_OK &&
		   le16_to_cpu(resp->CCStat) != CMD_STAT_EXEC) {
		   /* collisions may be reported as CMD_STAT_EXEC */

		/* Nautilus returns ccstat of CMD_STAT_QUEUE_BUSY to indicate busy */
		if (!(priv->model >= 400 && priv->model < 500 &&
			le16_to_cpu(resp->CCStat) == CMD_STAT_QUEUE_BUSY)) {
			pr_err("*** sendmesg error ***\n"
					"   CCStat = 0x%04x\n", le16_to_cpu(resp->CCStat));
		}

		priv->stats.tx_errors++;
	} else {
		priv->stats.tx_packets++;
		priv->stats.tx_bytes += priv->pending_tx_bytes;
	}

	dev_put(netdev);

	up(&priv->sem);

	/* check to see if any packets waiting to be transmitted or
	 * received */
	if (!tx_ring_empty(priv) || priv->rx_pkt_waiting) {
		if (!queue_work(priv->workq, &priv->work)) {
			DBG3("queue work: work already queued line %d", __LINE__);
		}
	}
}

static void readmesg_cmd_cb(struct urb *urb)
{
	unsigned int curr_time;
	unsigned int delta_time;
	struct rcom_priv *priv = netdev_priv((struct net_device *)urb->context);
	CBTRACE("urb completion: readmesg cmd");

	/* check status of URB */
	if (urb->status)
		DBG1("in readmesg_cmd_cb(), status=%d", urb->status);

	curr_time = get_us();
	delta_time = curr_time - priv->rxcb_intep_time;
	if (delta_time > priv->rxcb_max_delta) {
	    priv->rxcb_max_delta = delta_time;
	}
	DBG0("readmesg_cmd_cb: delta_time = %d us, max_delta_time = %d us", delta_time, priv->rxcb_max_delta);

	dev_put((struct net_device *)urb->context);
	usb_free_urb(urb);
}

/*
 * Completion handler for URBs submitted to receive the response to a
 * ReadMesg command.  When this function is called, the usb core has received
 * the response from the device, which contains the received data packet.
 */
static void readmesg_resp_cb(struct brb *brb)
{
	struct net_device *netdev = NULL;
	struct rcom_priv *priv = NULL;
	struct USB_RESP_READMESG *hdr;
	int len;
	struct sk_buff *skb = NULL;
	int retval = 0;

	if (!brb) {
		pr_err("null urb!\n");
		return;
	}

	netdev = brb->context;
	if (!netdev) {
		pr_err("null netdev!\n");
		brb_free(brb);
		return;
	}
	priv = netdev_priv(netdev);

	CBTRACE("urb completion: readmesg resp");

	/* check status of URB */
	if (brb->status)
		DBG1("in readmesg_cmd_cb(), status=%d", brb->status);

	dev_put(netdev);

	hdr = (struct USB_RESP_READMESG *)brb->transfer_buffer;

	if (!hdr) {
		pr_err("hdr == NULL!\n");
		goto done;
	}

	if (hdr->Head.MagicNum != CICADA_CMD_MAGIC
		|| hdr->Head.RouteID != ROUTE_NET
		|| le16_to_cpu(hdr->Head.MsgID) != DEV_RESP_READMESG) {

		/* Nautilus returns MsgId of DEV_RESP_STD to indicate busy */
		if (!(priv->model >= 400 && priv->model < 500 &&
				le16_to_cpu(hdr->Head.MsgID) == DEV_RESP_STD)) {
			pr_err("*** Unexpected response in readmesg_resp_cb\n");
			pr_err("  MagicNum (0x%x): 0x%x\n", CICADA_CMD_MAGIC, hdr->Head.MagicNum);
			pr_err("  routing id (0x%x): 0x%x\n", ROUTE_NET, hdr->Head.RouteID);
			pr_err("  message id (0x%x): 0x%x\n", DEV_RESP_READMESG,
					le16_to_cpu(hdr->Head.MsgID));
		}

		if (priv->model >= 200 && priv->model < 300 &&
				priv->rx_pkt_retries++ < 3 ) {
			priv->rx_pkt_waiting = 1; // Retry, pkt may still be in module queue
		}
		goto done;
	}

	if (le16_to_cpu(hdr->CCStat) != CMD_STAT_OK &&
		le16_to_cpu(hdr->CCStat) != CMD_STAT_EXEC) {
		/* CMD_STAT_EXEC indicates empty pkt queue */
		pr_err("*** readmesg error ***\n"
			"   CCStat = 0x%04x\n", le16_to_cpu(hdr->CCStat));

		if (priv->model >= 200 && priv->model < 300) {
			if (priv->rx_pkt_retries++ < 3) {
				priv->rx_pkt_waiting = 1; // Retry, pkt may still be in module queue
			}
		}
		goto done;
	}

	priv->rx_pkt_retries = 0; // Clear retry count
	len = le32_to_cpu(hdr->Length);

	DBG3("  CCStat: 0x%04x", le16_to_cpu(hdr->CCStat));
	DBG3("  short stat: 0x%04x", le16_to_cpu(hdr->OptionalDebug));
	DBG3("  comm stat: 0x%04x", le16_to_cpu(hdr->CommStat));

	if (len <= MAX_PACKET_DATA) {
		skb = dev_alloc_skb(len + 5); /* was + 2; network internals book shows + 5 */
	}
	else {
		skb = NULL;
		pr_err("---rx pkt len according to received hdr: %d\n", len);
	}

	if (!skb) {
		if (printk_ratelimit())
			pr_warning("readmesg_resp_cb: low on mem - packet dropped\n");
		priv->stats.rx_dropped++;
		ERR_TR("readmesg_resp_cb: skb = NULL, len = %d, dropped = %ld", len, priv->stats.rx_dropped);
		goto done;
	}
	skb_reserve(skb, 2); /* align IP on 16B boundary */
	memcpy(skb_put(skb, len), hdr->MesgData, len);

	skb->ip_summed = CHECKSUM_NONE;

	skb->protocol = eth_type_trans(skb, netdev);

	/*
	 * Set appropriate bits in 16 bit flags field if RLCP protocol and version
	 * match for use in tcpdump debugging.
	 *
	 * For copper, we add the last channel used.
	 */
	if (priv->model >= 300 && priv->model < 500 &&
			skb->protocol == 0xB588 && skb->data[0] == 4) {
		uint8_t byte = (skb->data[3] & 0xC7) | ((le16_to_cpu(hdr->CommStat) & 0xE0) >> 2);
		skb->data[3] = byte;
	}

	if (priv->model >= 200 && priv->model < 300 &&
			skb->protocol == 0xB588 && skb->data[0] == 4) {
		if (((skb->data[3] & 0x3) != priv->comm_cfg.ChanPriB) ||
				(skb->data[4] != priv->comm_cfg.ChanPriA)) {
			skb->data[3] |= 0x04;
		}
	}

	priv->stats.rx_packets++;
	priv->stats.rx_bytes += len;

	retval = netif_rx(skb);

	if (retval != NET_RX_SUCCESS && printk_ratelimit()) {
		if (retval == NET_RX_DROP)
			pr_warning("netif_rx returned NET_RX_DROP\n");
		else
			pr_warning("netif_rx returned %d\n", retval);
	}

done:
	brb_free(brb);

	up(&priv->sem);

	/* check to see if any packets waiting to be transmitted or
	 * received */
	if (!tx_ring_empty(priv) || priv->rx_pkt_waiting) {
		if (!queue_work(priv->workq, &priv->work)) {
			DBG3("queue work: work already queued line %d", __LINE__);
		}
	}
}

/* allocate, fill out and submit URBs for ReadMesg command and response
 *
 * This function should only be called when holding the priv->sem mutex.
 */
static int cicada_readmesg(struct net_device *netdev)
{
	int retval;
	struct urb *cmd_urb;
	void *cmd_urb_buf;
	struct brb *resp_brb;
	void *resp_brb_buf;
	struct rcom_priv *priv = netdev_priv(netdev);

	cmd_urb = usb_alloc_urb(0, GFP_KERNEL);
	cmd_urb_buf = kmalloc(MAX_URB_SIZE, GFP_KERNEL);
	cmd_urb->transfer_buffer = cmd_urb_buf;
	cmd_urb->transfer_flags |= URB_FREE_BUFFER;
	cmd_urb->transfer_flags |= URB_ZERO_PACKET;
	fill_cicada_readmesg_cmd(cmd_urb_buf, priv, MAX_URB_SIZE);
	usb_fill_bulk_urb(cmd_urb, priv->usb_dev,
			usb_sndbulkpipe(priv->usb_dev, priv->bulk_out_endpointAddr),
			cmd_urb_buf, sizeof(struct USB_CMD_READMESG),
			readmesg_cmd_cb, netdev);
	dev_hold(netdev);
	retval = priv->bulk_submit_snd_urb(priv->mmstate, cmd_urb, GFP_KERNEL, ROUTE_NET);
	if (retval) {
		DBG1("cicada_readmesg: line %d bulk_submit_snd_urb returned %d",
				__LINE__, retval);
		dev_put(netdev);
		if (retval == -ENODEV)
			return retval;
	}

	/* allocate brb and buffer for response, then init */
	resp_brb = kmalloc(sizeof(struct brb), GFP_KERNEL);
	resp_brb_buf = kmalloc(MAX_URB_SIZE, GFP_KERNEL);
	brb_init(resp_brb, resp_brb_buf, MAX_URB_SIZE, readmesg_resp_cb, priv->mmstate, netdev, ROUTE_NET);

	dev_hold(netdev);
	retval = priv->bulk_submit_rcv_brb(resp_brb);
	if (retval) {
		DBG1("cicada_readmesg: line %d bulk_submit_rcv_urb returned %d",
				__LINE__, retval);
		dev_put(netdev);
		if (retval == -ENODEV)
			return retval;
	}

	return 0;
}

/*
 * Callback invoked by the cicada_meas driver when an interrupt endpoint
 * message destined for us (the network driver) (as determined by the routing
 * id) is received.
 *
 * Currently this function is used to:
 *  - indicate that a packet has been received by the device,
 *  - handle a copper module Comm Info event interrupt message.
 *
 * This callback runs in the context of a workqueue.
 *
 * The URB passed in should not be referenced after this function returns.
 */
static void int_ep_cb(struct urb *int_urb, void *vnetdev)
{
	struct net_device *netdev = (struct net_device *)vnetdev;
	struct rcom_priv *priv = netdev_priv(netdev);
	const struct USB_MSG_HEADER *std_hdr = (struct USB_MSG_HEADER *)int_urb->transfer_buffer;

	uint8_t recv_err = 0;
	int retval;

	/* check status of URB */
	if (int_urb->status) {
		ERR_TR("*** in int_ep_cb(), status=%d", int_urb->status);
		return;
	}


	if (std_hdr->RouteID != ROUTE_NET) {
		pr_err("**********int_ep_cb: illegal RouteID: %u\n", std_hdr->RouteID);
	}

	else if (std_hdr->MsgID == INT_CODE_MESGRCV) {
		const struct USB_INT_MESGRCV *mr_hdr = (struct USB_INT_MESGRCV *)std_hdr;
		const uint16_t mesg_status = le16_to_cpu(mr_hdr->MesgStat);
		const uint16_t mesg_len = le16_to_cpu(mr_hdr->MesgLen);
		if (!(mesg_status & COMM_STAT_MESGRCVD)) {
			pr_err("*** int_ep_cb: mesgrcvd bit not set in mesg_stat\n");
			return;
		}

		if (mesg_status & COMM_STAT_MESGERR) {
			/* message received with error:  send a new config comm
			 * command to discard packet and re-enable receive.
			 */
			recv_err = 1;
			priv->stats.rx_errors++;
					ERR_TR("******** int_ep_cb: priv->stats.rx_errors = %d", priv->stats.rx_errors);
		}
		if (mesg_status & COMM_STAT_TYPE_ERR) {
			/* message received with error:  send a new config comm
			 * command to discard packet and re-enable receive.
			 */
			recv_err = 1;
			priv->stats.rx_frame_errors++;
					ERR_TR("******** int_ep_cb: priv->stats.rx_frame_errors = %d (not host-to-host)",
						   priv->stats.rx_frame_errors);
			}

		DBG0("int_ep_cb:		  msgstat = 0x%03x, pktlen = %3d", mesg_status, mesg_len);
			priv->rxcb_intep_time = get_us();

		if (mesg_len > MAX_PACKET_DATA)
			pr_err("rx pkt len according to int ep msg: %d", mesg_len);

		if (!recv_err) {
			/* TBD - try to get mutex and start read here, or just schedule
			 * restart_rx_tx? */
#if 1
			/* Set the rx_pkt_waiting flag first, in case the mutex is currently
			 * held by the restart_rx_tx thread.  After acquiring the mutex, we
			 * check the flag again in case the rx has been handled by
			 * restart_rx_tx. */

			priv->rx_pkt_waiting = 1;

			down(&priv->sem);

			if (priv->rx_pkt_waiting) {
				priv->rx_pkt_waiting = 0;
				retval = cicada_readmesg(netdev);
				if (retval < 0) {
					up(&priv->sem);
				}
			} else {
				up(&priv->sem);
			}

#else
			priv->rx_pkt_waiting = 1;
			queue_work(priv->workq, &priv->work);
#endif
		}
	}

	/* event interrupt message - no need to queue since only one message is
	 * possible per start command and only the last message is relevant anyway */
	else if (std_hdr->MsgID == INT_CODE_COMMINFO) {
		if (priv->evtinfo->evtmsg) {
			mutex_lock(&priv->evtinfo->data_mutex);
			memcpy(priv->evtinfo->evtmsg, std_hdr, sizeof(UsbIntCommInfo));
			mutex_unlock(&priv->evtinfo->data_mutex);
			priv->evtinfo->evtint = 1;

			wake_up_interruptible(&priv->evtinfo->evtwq);
		}
	}

	else {
		pr_err("**********int_ep_cb: illegal MsgID: %#x\n", std_hdr->MsgID);
	}
}

static inline void stop_tx(struct net_device *netdev, struct rcom_priv *priv)
{
	netif_stop_queue(netdev);
	priv->tx_stopped = 1;
}

/**
 * rcom_tx - transmit packet - called by the kernel.
 *
 * Actually just write the packet to the tx ring, then schedule the work queue
 * to start a transmit.
 */
static int rcom_tx(struct sk_buff *skb, struct net_device *netdev)
{
	int pkt_len = 0;
	struct rcom_priv *priv = netdev_priv(netdev);

	pkt_len = skb->len;

	if (pkt_len < MIN_PACKET_DATA) {
		pr_err("cicada_rcom: packet too short, dropping %d bytes\n",
				pkt_len);
		dev_kfree_skb(skb);
		return NETDEV_TX_OK;
	}
	if (pkt_len > MAX_PACKET_DATA) {
		pr_err("cicada_rcom: packet too long, dropping %d bytes\n",
				pkt_len);
		dev_kfree_skb(skb);
		return NETDEV_TX_OK;
	}

	/* Set appropriate bits in 16 bit flags field if our protocol and version */
	if (skb->data[12] == 0x88 && skb->data[13] == 0xB5 && skb->data[14] == 4) {
		switch (priv->comm_cfg.Baud) {
		case   85000:
			skb->data[16] = 1; break;
		case 1950000:
			skb->data[16] = 2; break;
		case  200000:
			skb->data[16] = 3; break;
		default:
			skb->data[16] = 0; break;
		}
		if (priv->model >= 200 && priv->model < 300) {
			skb->data[17] = priv->comm_cfg.ChanPriB & 0x3;
			skb->data[18] = priv->comm_cfg.ChanPriA;
		}
	}

	if (tx_ring_write(netdev, skb) == 0) {

		/* After writing packet to tx ring, if ring is full, tell the
		 * kernel to stop sending. */
		if (tx_ring_full(priv))
			stop_tx(netdev, priv);

		netdev->trans_start = jiffies;

		dev_kfree_skb(skb);

		if (!queue_work(priv->workq, &priv->work)) {
			DBG3(KERN_NOTICE "queue work: work already queued line %d",
					__LINE__);
		}

		return NETDEV_TX_OK;
	} else {
		/* According to Documentation/networking/netdevices.txt, it is
		 * probably a bug if we get here. */
		pr_warning("*** rcom_tx called with full tx queue\n");
		stop_tx(netdev, priv);
		return NETDEV_TX_BUSY;
	}
}

/* TODO - Deal with transmit timeout
 *
 * This is called by dev_watchdog() when the netif queue has been stopped for
 * too long.
 */
static void rcom_tx_timeout(struct net_device *dev)
{
	pr_notice("*** cicada_rcom: tx timeout\n");
	/* TODO - reset device? */
}

static struct net_device_stats *rcom_stats(struct net_device *dev)
{
	struct rcom_priv *priv = netdev_priv(dev);
	return &priv->stats;
}

/* TBD - needed? */
/* called after device is detached from the network */
static void rcom_uninit(struct net_device *netdev)
{
	struct rcom_priv *priv = netdev_priv(netdev);
	CONFIG("rcom_uninit");
}

/* called after last user reference disappears */
static void rcom_destructor(struct net_device *netdev)
{
	struct rcom_priv *priv = netdev_priv(netdev);
	void *mmstate = priv->mmstate;
	cicada_meas_state_put_func measStatePutFunc = priv->cicada_meas_state_put;

	CONFIG("rcom_destructor - entry");

	usb_put_dev(priv->usb_dev);
	usb_put_intf(priv->usb_if);

	brb_free(priv->tx_resp_brb);

	tx_ring_free(netdev);

	destroy_workqueue(priv->workq);

	/* netdev related functions are done with evtinfo */
	kref_put(&priv->evtinfo->kref, evt_info_delete);

	atomic_set(&netdev_priv_valid, 0);
	free_netdev(netdev);
	LOG_CONFIG("rcom_destructor - netdev/priv freed - do not use priv or tracing");

	measStatePutFunc(mmstate);
	LOG_CONFIG("rcom_destructor - exit");
}

static const struct net_device_ops rcom_dev_ops = {
	.ndo_uninit = rcom_uninit,
	.ndo_open = rcom_open,
	.ndo_stop = rcom_stop,
	.ndo_start_xmit = rcom_tx,
	.ndo_do_ioctl = rcom_ioctl,
	.ndo_get_stats = rcom_stats,
	.ndo_change_mtu = NULL,
	.ndo_tx_timeout = rcom_tx_timeout
};

static void rcom_cleanup(void)
{
	class_destroy(cicada_rcom_class);
}

#ifdef ENABLE_CICADA_SPEED_ATTR
static ssize_t show_speed(struct device *dev, struct device_attribute *attr,
		char *buf)
{
	struct net_device *netdev;
	struct rcom_priv *priv;
	int retval = 0;

	netdev = container_of(dev, struct net_device, dev);
	priv = netdev_priv(netdev);

	retval = sprintf(buf, "%d ", priv->speed);
	buf += retval;
	if (priv->speed == 0)
		retval += sprintf(buf, "(fast)\n");
	else
		retval += sprintf(buf, "(slow/long comm)\n");

	return retval;
}

static ssize_t set_speed(struct device *dev, struct device_attribute *attr,
		const char *buf, size_t count)
{
	uint32_t speed = 0;
	struct net_device *netdev;
	struct rcom_priv *priv;
	int retval = 0;

	netdev = container_of(dev, struct net_device, dev);
	priv = netdev_priv(netdev);

	retval = sscanf(buf, "%u", &speed);
	if (retval > 0) {
		priv->speed = speed;
		if (netdev->flags & IFF_UP)
			rcom_enable(netdev, 1);
	}

	return count;
}

static struct device_attribute dev_attr_speed;
static DEVICE_ATTR(speed, S_IRUSR|S_IWUSR, show_speed, set_speed);
#endif

/* invoked by alloc_netdev() */
/* TBD - proper division of stuff between here and cicada_rcom_probe */
static void rcom_init(struct net_device *netdev)
{
	struct rcom_priv *priv;
	int i;

	priv = netdev_priv(netdev);
	LOG_CONFIG("rcom_init() - entry");
	memset(priv, 0, sizeof(struct rcom_priv));
#if 0
	/*
	 * Make the usual checks: check_region(), probe irq, ...  -ENODEV
	 * should be returned if no device found.  No resource should be
	 * grabbed: this is done on open().
	 */
#endif

	ether_setup(netdev);
	netdev->tx_queue_len = 3; /* Override ether_setup (1000) TBD - ppp_generic.c uses 3 */
	netdev->flags &= ~IFF_MULTICAST;  /* Clear multicast bit set in ether_setup */

	// Set up dev_addr from serial number
	for (i=0; i < 6; i++) {
		netdev->dev_addr[i] = rcom_dev_addr[i];
	}

	netdev->dev_addr[0] |=  0x2; /* Make sure locally administered bit is set */
	netdev->dev_addr[0] &= ~0x1; /* Make sure uni/multicast bit is clear (unicast) */

	netdev->netdev_ops = &rcom_dev_ops;
	netdev->destructor = rcom_destructor;
	netdev->watchdog_timeo = tx_timeout;

	spin_lock_init(&priv->lock);
	LOG_CONFIG("rcom_init() - exit");
}


/*
 * cdev cannot be embedded in netdev->priv because calling cdev_del()
 * doesn't instantly stop using the cdev.
 */
static int evt_open(struct inode *inode, struct file *file)
{
	LOG_CONFIG("cicada_rcom::evt_open() - entry");

	if (inode && file) {
		struct evt_info *evtinfo = get_evt_info_active(inode->i_cdev);
		if (!evtinfo) {
			return -ENODEV;
		}

		if (mutex_trylock(&evtinfo->file_mutex) && mutex_trylock(&evtinfo->data_mutex)) {
			evtinfo->evtmsg = kzalloc(sizeof(UsbIntCommInfo), GFP_KERNEL);
			mutex_unlock(&evtinfo->data_mutex);
			evtinfo->evtint = 0;
			file->private_data = evtinfo;

			/* increment evtinfo ref to ensure that it will stay around
			   until evt_release() is called */
			kref_get(&evtinfo->kref);
			LOG_CONFIG("cicada_rcom::evt_open() - exit OK");
			return 0;
		}
		return -EBUSY;
	}

	return -ENODEV;
}

static int evt_release(struct inode *inode, struct file *file)
{
	LOG_CONFIG("cicada_rcom::evt_release() - entry - inode = %p, file = %p", inode, file);

	if (inode && file) {
		struct evt_info *evtinfo = (struct evt_info*) file->private_data;
		if (!evtinfo) {
			return -ENODEV;
		}

		mutex_lock(&evtinfo->data_mutex);
		kfree(evtinfo->evtmsg);
		evtinfo->evtmsg = NULL;
		mutex_unlock(&evtinfo->data_mutex);
		mutex_unlock(&evtinfo->file_mutex);

		/* evt related functions are now done with evtinfo */
		kref_put(&evtinfo->kref, evt_info_delete);
		LOG_CONFIG("cicada_rcom::evt_release() - exit OK");
		return 0;
	}

	return -ENODEV;
}

static ssize_t evt_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	/* non-null pointers */
	if (file && file->private_data) {
		struct evt_info *evtinfo = (struct evt_info*) file->private_data;
		if (evtinfo->evtmsg) {
			int retval;

			/* device has not yet received an event interrupt */
			if (!evtinfo->evtint) {

				/* non-blocking read - return 'try again' error */
				if (file->f_flags & O_NONBLOCK) {
					return -EAGAIN;
				}

				/* blocking read - wait until event */
				if (wait_event_interruptible(evtinfo->evtwq, evtinfo->evtint)) {
					return -ERESTARTSYS;
				}
			}

			/* read data and clear event interrupt flag if successful */
			mutex_lock(&evtinfo->data_mutex);
			retval = simple_read_from_buffer(buffer, count, ppos, evtinfo->evtmsg, sizeof(UsbIntCommInfo));
			mutex_unlock(&evtinfo->data_mutex);
			if (retval > 0) evtinfo->evtint = 0;
			return retval;
		}
	}

	return -ENODEV;
}

static uint32_t evt_poll(struct file *filp, poll_table *wait)
{
	uint32_t mask = 0;

	if (filp && filp->private_data) {
		struct evt_info *evtinfo = (struct evt_info*) filp->private_data;

		poll_wait(filp, &evtinfo->evtwq, wait);
		if (evtinfo->evtint) {
			mask |= POLLIN | POLLRDNORM;
		}
	}

	return mask;
}

static const struct file_operations evt_fops_table = {
	.owner	 = THIS_MODULE,
	.open	 = evt_open,
	.release = evt_release,
	.read	 = evt_read,
	.poll	 = evt_poll,
};


/**
 * cicada_rcom_probe - called by cicada_meas driver for each device found
 * @usbdev: the usb_dev of the corresponding measurement module
 * @usbif: the usb_interface of the corresponding measurement module
 * @bulk_in_endpointAddr: bulk in endpoint to use
 * @bulk_out_endpointAddr: bulk out endpoint to use
 * @route: the routing code indicating interrupt messages are destined for us,
 * 		filled in by this function
 * @cb: the interrupt endpoint callback function, filled in by this function
 * @pnetdev: the net_device for this network interface, filled in by this
 * 		function
 *
 * This should be called by the cicada_meas driver's probe function, to
 * create a new network interface.  We fill in the address of the callback
 * function used to pass interrupt messages to us, along with a pointer to our
 * net_device struct, which should be the second parameter when calling the
 * callback function.
 */
int cicada_rcom_probe(struct usb_device *usbdev, struct usb_interface *usbif,
		unsigned int bulk_in_endpointAddr,
		unsigned int bulk_out_endpointAddr,
		int *route,
		void (**cb)(struct urb *, void *),
		void **pnetdev,
		int  (*bulk_submit_snd_urb)(void *mmstate, struct urb *urb,
					    gfp_t mem_flags, int routing_code),
		int  (*bulk_submit_rcv_brb)(struct brb *brb),
		void *mmstate,
		void (*cicada_meas_state_get)(void* mmstate),
		void (*cicada_meas_state_put)(void* mmstate),
		uint32_t* ptrace_enabled_chans,
		void (*trace_record)(uint32_t chan, const char* fmt, ...),
		int enable_config_logging_arg)
{
	struct net_device *netdev;
	int result;
	struct brb *tx_resp_brb;
	void *tx_resp_brb_buf;
	struct rcom_priv *priv = NULL;
	uint32_t model = 0, serial = 0;
	int retval = 0;

	enable_config_logging = enable_config_logging_arg;

	LOG_CONFIG("cicada_rcom_probe() - entry");

	if (!usbdev || !usbif || !route || !cb || !pnetdev) {
		pr_err("cicada_rcom_probe called with null pointer\n");
		return -1;
	}

	/* Need to set this before the call to alloc_netdev */
	rcom_dev_addr[0] = 0x02; /* Set the locally administered bit */
	rcom_dev_addr[1] = 0x0c;
	/* The serial number field could be one of 3 different things:
	   300:123456:6:42 -> model:serial:rev:lia -> Copper with LIA plugged in at power up
	   300:345678:6    -> model:serial:rev     -> Copper without above
	   200:987654      -> model:serial         -> Loss Length
	   serial is a 7 digit production number: 1000000 - 9999999 and fits in 3 bytes
	   so the fourth byte will be a hash of model which works for the following model
	   numbers:
	   100..120, 200..220, 300..320, 400..420, 500..520
	*/
	if (!usbdev->serial) {
		pr_err("cicada_rcom_probe called with null serial pointer\n");
		return -EADDRNOTAVAIL;
	}
	retval = sscanf(usbdev->serial, "%u:%u", &model, &serial);
	if (retval == 2) {
		rcom_dev_addr[2] = ((((int)(model - 100) / 100) * 100) + ((model - 100) % 21)) & 0xff;
		rcom_dev_addr[3] = (serial & 0x00ff0000) >> 16;
		rcom_dev_addr[4] = (serial & 0x0000ff00) >>  8;
		rcom_dev_addr[5] = (serial & 0x000000ff) >>  0;
	}
	else {
		/* If we couldn't find a serial number, assign a random address */
		get_random_bytes(&rcom_dev_addr[2], 4);
		pr_warning("cicada_rcom_probe: serial number not OK; retval = %d; using random bytes", retval);
	}

	netdev = alloc_netdev(sizeof(struct rcom_priv), "cicada%d", NET_NAME_UNKNOWN, rcom_init);
	if (!netdev) {
		pr_err("cicada_rcom alloc failed\n");
		return -ENOMEM;
	}

	priv = netdev_priv(netdev);
	priv->netdev = netdev;
	priv->usb_dev = usb_get_dev(usbdev);
	priv->usb_if = usb_get_intf(usbif);
	priv->bulk_in_endpointAddr = bulk_in_endpointAddr;
	priv->bulk_out_endpointAddr = bulk_out_endpointAddr;
	priv->route = *route = ROUTE_NET;
	priv->model = model;
	priv->response_success = 0;
	priv->busy = 0;
	priv->seqnum = 1;
	priv->rx_pkt_waiting = 0;
	priv->tx_stopped = 0;
	priv->bulk_submit_snd_urb = bulk_submit_snd_urb;
	priv->bulk_submit_rcv_brb = bulk_submit_rcv_brb;
	priv->mmstate = mmstate;
	priv->cicada_meas_state_get = cicada_meas_state_get;
	priv->cicada_meas_state_put = cicada_meas_state_put;
	priv->cicada_meas_state_get(mmstate);
	priv->ptrace_enabled_chans = ptrace_enabled_chans;
	priv->trace_record = trace_record;
	priv->rxcb_intep_time = 0;
	priv->rxcb_max_delta = 0;

	init_waitqueue_head(&priv->wq);
	priv->workq = create_singlethread_workqueue("cicada_rcom");
	INIT_WORK(&priv->work, restart_rx_tx);

	sema_init(&priv->sem, 1);

	tx_ring_init(netdev);

	tx_resp_brb = kmalloc(sizeof(struct brb), GFP_KERNEL);
	priv->tx_resp_brb = tx_resp_brb;
	tx_resp_brb_buf = kmalloc(MAX_URB_SIZE, GFP_KERNEL);
	brb_init(tx_resp_brb, tx_resp_brb_buf, MAX_URB_SIZE, sendmesg_resp_cb, priv->mmstate, netdev, ROUTE_NET);

	/* increment ref cnt to represent the fact that cicada_meas will be
	 * holding a pointer to our netdev struct.  This reference is
	 * decremented again when cicada_meas calls cicada_rcom_disconnect(). */
	dev_hold(netdev);

	/* fill in address of interrupt endpoint callback */
	*cb = int_ep_cb;
	*pnetdev = netdev;

	result = register_netdev(netdev);
	if (result) {
		dev_err(&netdev->dev, "cicada_rcom: register_netdev failed\n");
		dev_put(netdev);
		rcom_destructor(netdev);
		return result;
	}

#ifdef ENABLE_CICADA_SPEED_ATTR
	if (device_create_file(&netdev->dev, &dev_attr_speed))
		dev_err(&netdev->dev, "device_create_file failed\n");
#endif

	/* setup cicada_rcom_evt */
	if (alloc_chrdev_region(&priv->devnum, 0, CICADA_RCOM_NUM_MINORS, CICADA_RCOM_PROC_NAME)) {
		pr_err("cicada_rcom_probe: alloc_chrdev_region failed\n");
	}
	else {
		/* USB unplug causes lifetime issues.  Priv must keep a pointer to evtinfo
		 * so reference counted evtinfo can live on after netdev/priv is deleted. */
		if ((priv->evtinfo = evt_info_create(priv->mmstate,
						 priv->cicada_meas_state_get,
						 priv->cicada_meas_state_put)) == NULL)
		{
			pr_err("cicada_rcom evt_info_init failed\n");
			return -ENOMEM;
		}

		/* Create evt char device and associate its fops table.
		 * "priv" must have pointer to cdev (cdev must not be embedded in priv).
		 */
		priv->evtinfo->cdev = cdev_alloc();
		priv->evtinfo->cdev->owner = THIS_MODULE;
		priv->evtinfo->cdev->ops = &evt_fops_table;

		if (cdev_add(priv->evtinfo->cdev, priv->devnum, CICADA_RCOM_NUM_MINORS)) {
			pr_err("cicada_rcom_probe: cdev_add failed\n");
			kref_put(&priv->evtinfo->kref, evt_info_delete);
			return -ENOMEM;
		}
		else {
			priv->chardev = device_create(cicada_rcom_class, &priv->usb_if->dev,
							       priv->devnum, NULL, "cicada_rcom_evt");
			if (IS_ERR(priv->chardev)) {
				pr_err("cicada_rcom_probe: device_create failed\n");
				cdev_del(priv->evtinfo->cdev);
				kref_put(&priv->evtinfo->kref, evt_info_delete);
				return -ENOMEM;
			}
		}

		/* some functions cannot access evtinfo using netdev/priv */
		add_evt_info_active(priv->evtinfo);
	}

	dev_info(&netdev->dev, "registered new device\n");
	atomic_set(&netdev_priv_valid, 1);
	CONFIG("cicada_rcom_probe() - exit");

	return 0;
}
EXPORT_SYMBOL(cicada_rcom_probe);

void cicada_rcom_disconnect(void *vnetdev)
{
	struct net_device *netdev = (struct net_device *)vnetdev;
	struct rcom_priv *priv = netdev_priv(netdev);

	CONFIG("cicada_rcom_disconnect() - entry");

	dev_info(&netdev->dev, "disconnected device\n");
#ifdef ENABLE_CICADA_SPEED_ATTR
	device_remove_file(&netdev->dev, &dev_attr_speed);
#endif
	wake_up_interruptible(&priv->evtinfo->evtwq);

	if (priv->chardev) device_destroy(cicada_rcom_class, priv->devnum);
	priv->chardev = NULL;

	/* cdev ops may still be used by file system after cdev_del() is called.
	 * The memory pointed to by evtinfo->cdev will be freed after all users
	 * have finished. */
	cdev_del(priv->evtinfo->cdev);
	priv->evtinfo->cdev = NULL;

	unregister_chrdev_region(priv->devnum, CICADA_RCOM_NUM_MINORS);

	/* cicada_meas has zeroed its reference to us */
	dev_put(netdev);

	/* Note: it waits in unregister_netdev() for all references to
	 * the netdev struct to be released */
	unregister_netdev(netdev);
}
EXPORT_SYMBOL(cicada_rcom_disconnect);

static int __init rcom_init_module(void)
{
	init_evt_info_active();

	cicada_rcom_class = class_create(THIS_MODULE, CICADA_RCOM_SYS_CLASS);
	if (!cicada_rcom_class) {
		pr_err("cicada_rcom class creation failed\n");
		return -ENODEV;
	}

	/* Uncomment to test kernel log levels */
#if 0
	/*
	 * Stock Ubuntu printk logging:
	 *     /var/log/messages: only INFO, NOTICE, WARNING
	 *     /var/log/kern.log: all
	 *     /var/log/syslog:   all
	 *     dmesg:             all
	 *     console (w/o X11): only ERR, CRIT, ALERT, EMERG
	 *
	 */
	printk(KERN_DEBUG "Testing KERN_DEBUG\n");
	printk(KERN_INFO "Testing KERN_INFO\n");
	printk(KERN_NOTICE "Testing KERN_NOTICE\n");
	printk(KERN_WARNING "Testing KERN_WARNING\n");
	printk(KERN_ERR "Testing KERN_ERR\n");
	printk(KERN_CRIT "Testing KERN_CRIT\n");
	printk(KERN_ALERT "Testing KERN_ALERT\n");
	printk(KERN_EMERG "Testing KERN_EMERG\n");
#endif

	pr_info("Loaded cicada_rcom driver\n");

	return 0;
}

module_init(rcom_init_module);
module_exit(rcom_cleanup);

module_param(tx_timeout, int, 0);

MODULE_LICENSE("GPL");
