/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/* #define DEBUG */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/gpio/consumer.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/power_supply.h>
#include <linux/types.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <linux/regulator/consumer.h>
#include <asm/gpio.h>
#include <linux/backlight.h>

struct led7706_data {
	struct device *dev;

	struct backlight_device *backlight;
	struct regulator *power_supply;
	struct gpio_desc *fault;

	unsigned long last_fault;
	struct delayed_work work;
};

static struct led7706_data *led7706_data;

static ssize_t enable_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led7706_data *data = led7706_data;

	return sprintf(buf, "%u\n", regulator_is_enabled(data->power_supply));
}

static ssize_t enable_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t n)
{
	struct led7706_data *data = led7706_data;
	unsigned int value;

	if (kstrtouint(buf, 2, &value) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	if (value) {
		if (regulator_enable(data->power_supply)) {
			dev_err(data->dev, "regulator_enable power-supply failed\n");
		}
	} else {
		if (regulator_disable(data->power_supply)) {
			dev_err(data->dev, "regulator_disable power-supply failed\n");
		}
	}

	return n;
}

static DEVICE_ATTR_RW(enable);

static ssize_t fault_show(struct device *dev,
			   struct device_attribute *attr,
			   char *buf)
{
	struct led7706_data *data = led7706_data;

	return sprintf(buf, "%u\n", gpiod_get_value(data->fault));
}

static DEVICE_ATTR_RO(fault);

static struct attribute *dev_attrs[] = {
	&dev_attr_enable.attr,
	&dev_attr_fault.attr,
	NULL
};

static const struct attribute_group dev_attr_group = {
	.attrs = dev_attrs,
};

static const struct of_device_id led7706_device_ids[] = {
	{ .compatible = "fnet,led7706-bl" },
	{}
};
MODULE_DEVICE_TABLE(of, led7706_device_ids);

static void led7706_work(struct work_struct *work)
{
	struct led7706_data *data =
		container_of(work, struct led7706_data, work.work);

	dev_dbg(data->dev, "%s\n", __func__);

	/* Backlight may fault on driving enable, disable
	 then re-enable interrupts to prevent false fault */
	disable_irq(gpiod_to_irq(data->fault));

	/* re-enable the backlight supply after the
	   scheduled delay time.
	*/
	if (regulator_enable(data->power_supply))
		dev_err(data->dev, "regulator_enable power-supply failed\n");

	enable_irq(gpiod_to_irq(data->fault));
}

static irqreturn_t led7706_handler(int irqno, void *_data)
{
	struct led7706_data *data = _data;
	unsigned long fault_delay;

	dev_dbg(data->dev, "%s\n", __func__);

	/* turn off the backlight supply.  if last fault was a
	   long time ago (more than 1.2 seconds), then make the
	   off time short.  if the faults are happening frequently,
	   then make the delay longer to avoid rapidly cycling
	   the supply when no display is attached.
	*/
	if (regulator_disable(data->power_supply)) {
			dev_err(data->dev, "regulator_disable power-supply failed\n");
	}

	if (time_after(jiffies, data->last_fault + msecs_to_jiffies(1200))) {
		dev_dbg(data->dev, "Backlight fault %lu\n", jiffies);
		fault_delay = msecs_to_jiffies(1);	/* 1ms */
	} else
		fault_delay = msecs_to_jiffies(1000);	/* 1 sec */

	/* schedule delay then re-enable the backlight */
	data->last_fault = jiffies;
	schedule_delayed_work(&data->work, fault_delay);

	return IRQ_HANDLED;
}

static int led7706_parse_dt(struct device *dev, struct led7706_data *data)
{
	struct device_node *np = dev->of_node;

	if (!np)
		return -ENOENT;

	if (!of_device_is_available(np))
		return -ENODEV;

	data->power_supply = devm_regulator_get(dev, "power");
	if (IS_ERR(data->power_supply)) {
		dev_err(dev, "could not get power-supply\n");
		return PTR_ERR(data->power_supply);
	}

	data->fault = devm_gpiod_get(dev, "fault", GPIOD_IN);
	if (IS_ERR(data->fault)) {
		dev_err(dev, "could not get fault gpio\n");
		return PTR_ERR(data->fault);
	}

	return 0;
}

static int led7706_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct led7706_data *data;
	struct device_node *backlight;
	struct device_node *lcd;
	struct platform_device *lcd_pdev;
	int ret;

	dev_dbg(dev, "%s\n", __func__);

	data = devm_kzalloc(&pdev->dev, sizeof(*data), GFP_KERNEL);
	if (!data) {
		dev_err(&pdev->dev, "Failed to alloc driver structure\n");
		return -ENOMEM;
	}

	ret = led7706_parse_dt(dev, data);
	if (ret) {
		dev_err(dev, "probe failed\n");
		return ret;
	}

	data->dev = dev;
	dev_set_drvdata(dev, data);
	led7706_data = data;

	data->last_fault = jiffies;
	INIT_DELAYED_WORK(&data->work, led7706_work);

	/* wait until the pwm-backlight has been probed */
	backlight = of_parse_phandle(dev->of_node, "pwm-backlight", 0);
	if (!backlight) {
		dev_err(dev, "failed to find 'pwm-backlight' node in device tree\n");
		return -ENODEV;
	}

	data->backlight = of_find_backlight_by_node(backlight);
	of_node_put(backlight);

	if (!data->backlight) {
		dev_err(dev, "failed to find backlight device\n");
		return -ENODEV;
	}

	if (!dev_get_drvdata(&data->backlight->dev))
		return -EPROBE_DEFER;

	/* wait until the lcd has been probed */
	lcd = of_parse_phandle(dev->of_node, "lcd", 0);
	if (lcd) {
		lcd_pdev = of_find_device_by_node(lcd);
		of_node_put(lcd);

		if (!lcd_pdev) {
			dev_err(dev, "failed to find lcd device\n");
			return -ENODEV;
		}

		if (!platform_get_drvdata(lcd_pdev))
			return -EPROBE_DEFER;
	} else
		dev_info(dev, "device tree 'lcd' node not used\n");

	/* Wait till after setting enable to turn on interrupts. Enable line
	 * serves dual purpose as IO rail voltage, and fault line will twitch as
	 * power supply is enabled */
	if (regulator_enable(data->power_supply))
		dev_err(data->dev, "regulator_enable power-supply failed\n");

	/* Register interrupts. Use low level trigger to catch any interrupt
	 * that occurred after enable but before handler was registered. */
	ret = devm_request_threaded_irq(dev, gpiod_to_irq(data->fault),
				NULL, led7706_handler,
				IRQF_TRIGGER_LOW | IRQF_ONESHOT,
				"led-fault", data);
	if (ret) {
		dev_err(dev, "failed to request led-fault IRQ %d: %d\n",
			gpiod_to_irq(data->fault), ret);
		return ret;
	}

	ret = sysfs_create_group(&data->backlight->dev.kobj, &dev_attr_group);
	if (ret) {
		dev_err(dev, "%s, sysfs_create_group failed\n", __func__);
		return -ENODEV;
	}


	dev_info(&pdev->dev, "probe succeded\n");

	return 0;
}

static int led7706_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;

	sysfs_remove_group(&dev->kobj, &dev_attr_group);
	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int led7706_suspend(struct platform_device *pdev,
			   pm_message_t state)
{
	struct device *dev = &pdev->dev;
	struct led7706_data *data = platform_get_drvdata(pdev);

	dev_dbg(&pdev->dev, "%s:\n", __func__);

	/* this may be heavy handed to free the interrupt
	   going into suspend, but if the power supply for
	   the fault signal is turned off the i.MX7 only
	   asserts the external PMIC-STBY-REQ signal for
	   2 milliseconds and partially terminates suspend.
	   freeing and requesting the irq on suspend / resume
	   seems to work around this issue.
	*/
	devm_free_irq(dev, gpiod_to_irq(data->fault), data);

	cancel_delayed_work_sync(&data->work);

	return 0;
}

static int led7706_resume(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct led7706_data *data = platform_get_drvdata(pdev);
	int ret;

	dev_dbg(&pdev->dev, "%s:\n", __func__);

	/* this may be heavy handed to request the interrupt
	   coming out of suspend, but if the power supply for
	   the fault signal is turned off the i.MX7 only
	   asserts the external PMIC-STBY-REQ signal for
	   2 milliseconds and partially terminates suspend.
	   freeing and requesting the irq on suspend / resume
	   seems to work around this issue.
	*/
	 ret = devm_request_threaded_irq(dev, gpiod_to_irq(data->fault),
				NULL, led7706_handler,
				IRQF_TRIGGER_LOW | IRQF_ONESHOT,
				"led-fault", data);
	if (ret) {
		dev_err(dev, "failed to request led-fault IRQ %d: %d\n",
			gpiod_to_irq(data->fault), ret);
		return ret;
	}

	return 0;
}
#endif

static struct platform_driver led7706_driver = {
	.driver = {
		.name = "led7706-bl",
		.of_match_table = led7706_device_ids,
	},
	.probe	  = led7706_probe,
	.remove	  = led7706_remove,
#ifdef CONFIG_PM_SLEEP
	.suspend  = led7706_suspend,
	.resume   = led7706_resume,
#endif
};

module_platform_driver(led7706_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("STMicro led7706 backlight control");

