/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/* #define DEBUG */

#include <linux/kobject.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/err.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/gpio/consumer.h>
#include <linux/of.h>

extern unsigned int system_rev;
extern int cicada_is_cpu_imx353(void);
extern u64 imx7d_uid(void);

struct cicada_hw_data {
	struct device *dev;

	struct gpio_desc *gpio_type_2;
	struct gpio_desc *gpio_type_1;
	struct gpio_desc *gpio_type_0;
	int type;

	struct gpio_desc *gpio_rev_3;
	struct gpio_desc *gpio_rev_2;
	struct gpio_desc *gpio_rev_1;
	struct gpio_desc *gpio_rev_0;
	int rev;
};
static struct cicada_hw_data *cicada_hw_data = NULL;

int cicada_hw_type(void)
{
	if (!cicada_hw_data) {
		pr_err("%s defered\n", __func__);
		return -EPROBE_DEFER;
	}

	return cicada_hw_data->type;
}
EXPORT_SYMBOL(cicada_hw_type);

/*
 * Hardware Rev -- used to designate significant changes to the hardware
 * on a board rev that must be detectable at run time by the firmware. A
 * possible example would be a new board rev with more LPDDR or Flash memory.
 */
#define BOARD_REV_1 (0 << 8)
#define BOARD_REV_2 (1 << 8)
#define BOARD_REV_3 (2 << 8)
#define BOARD_REV_4 (3 << 8)
#define BOARD_REV_5 (4 << 8)
#define BOARD_REV_6 (5 << 8)
#define BOARD_REV_7 (6 << 8)
#define BOARD_REV_8 (7 << 8)
#define BOARD_REV_9 (8 << 8)
#define BOARD_REV_10 (9 << 8)
#define BOARD_REV_11 (10 << 8)
#define BOARD_REV_12 (11 << 8)
#define BOARD_REV_13 (12 << 8)
#define BOARD_REV_14 (13 << 8)
#define BOARD_REV_15 (14 << 8)
#define BOARD_REV_16 (15 << 8)

static ssize_t hw_type_show(struct device *dev,
			  struct device_attribute *attr,
			  char *buf)
{
	struct cicada_hw_data *data = dev_get_drvdata(dev);
	return sprintf(buf, "%u\n", data->type);
}

static DEVICE_ATTR_RO(hw_type);

static ssize_t hw_rev_show(struct device *dev,
			   struct device_attribute *attr,
			   char *buf)
{
	struct cicada_hw_data *data = dev_get_drvdata(dev);
	return sprintf(buf, "%u\n", data->rev);
}

static DEVICE_ATTR_RO(hw_rev);

static ssize_t cpu_uid_show(struct device *dev,
			   struct device_attribute *attr,
			   char *buf)
{
	return sprintf(buf, "0x%016llx\n", imx7d_uid());
}

static DEVICE_ATTR_RO(cpu_uid);

static struct attribute *dev_attrs[] = {
	&dev_attr_hw_type.attr,
	&dev_attr_hw_rev.attr,
	&dev_attr_cpu_uid.attr,
	NULL
};

static const struct attribute_group dev_attr_group = {
	.attrs = dev_attrs,
};

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-hw-pins", .data = NULL },
	{}
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

static int cicada_hw_parse_dt(struct device *dev, struct cicada_hw_data *data)
{
	struct device_node *np = dev->of_node;

	if (!np)
		return -ENOENT;

	if (!of_device_is_available(np))
		return -ENODEV;

	/*
	* Hardware Id -- used to indicate different loading of parts on the same
	* board revision.  An example would be the Remote for Copper cable testing
	* which probably has parts NOLOADed to reduce cost.  The Ethernet port, USB
	* Host port, display backlight supply, microSD card socket and touch panel
	* interface are likely candidates to NOLOAD.
	*/
	data->gpio_type_2 = devm_gpiod_get(dev, "type-2", GPIOD_IN);
	if (IS_ERR(data->gpio_type_2)) {
		dev_err(dev, "could not get type-2 gpio\n");
		return PTR_ERR(data->gpio_type_2);
	}

	data->gpio_type_1 = devm_gpiod_get(dev, "type-1", GPIOD_IN);
	if (IS_ERR(data->gpio_type_1)) {
		dev_err(dev, "could not get type-1 gpio\n");
		return PTR_ERR(data->gpio_type_1);
	}

	data->gpio_type_0 = devm_gpiod_get(dev, "type-0", GPIOD_IN);
	if (IS_ERR(data->gpio_type_0)) {
		dev_err(dev, "could not get type-0 gpio\n");
		return PTR_ERR(data->gpio_type_0);
	}

	/*
	* Hardware Rev -- used to designate significant changes to the hardware
	* on a board rev that must be detectable at run time by the firmware. A
	* possible example would be a new board rev with more LPDDR or Flash memory.
	*/
	data->gpio_rev_3 = devm_gpiod_get(dev, "rev-3", GPIOD_IN);
	if (IS_ERR(data->gpio_rev_3)) {
		dev_err(dev, "could not get rev-3 gpio\n");
		return PTR_ERR(data->gpio_rev_3);
	}

	data->gpio_rev_2 = devm_gpiod_get(dev, "rev-2", GPIOD_IN);
	if (IS_ERR(data->gpio_rev_2)) {
		dev_err(dev, "could not get rev-2 gpio\n");
		return PTR_ERR(data->gpio_rev_2);
	}

	data->gpio_rev_1 = devm_gpiod_get(dev, "rev-1", GPIOD_IN);
	if (IS_ERR(data->gpio_rev_1)) {
		dev_err(dev, "could not get rev-1 gpio\n");
		return PTR_ERR(data->gpio_rev_1);
	}

	data->gpio_rev_0 = devm_gpiod_get(dev, "rev-0", GPIOD_IN);
	if (IS_ERR(data->gpio_rev_0)) {
		dev_err(dev, "could not get rev-0 gpio\n");
		return PTR_ERR(data->gpio_rev_0);
	}

	return 0;
}

static int cicada_hw_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct cicada_hw_data *data;
	int ret;

	dev_dbg(dev, "%s\n", __func__);
	
	if (cicada_hw_data) {
		dev_err(dev, "already allocated object\n");
		return -EINVAL;
	}

	data = devm_kzalloc(&pdev->dev, sizeof(*data), GFP_KERNEL);
	if (!data) {
		dev_err(&pdev->dev, "no memory\n");
		return -ENOMEM;
	}

	ret = cicada_hw_parse_dt(&pdev->dev, data);
	if (ret) {
		dev_err(&pdev->dev, "parce dt failed\n");
		return ret;
	}

	data->dev = dev;
	dev_set_drvdata(dev, data);

	cicada_hw_data = data;

	/* read and set the hardware rev, to query use:
	   board_is_rev(rev) macro */
	data->rev = (gpiod_get_value(data->gpio_rev_0) << 0) |
		    (gpiod_get_value(data->gpio_rev_1) << 1) |
		    (gpiod_get_value(data->gpio_rev_2) << 2) |
		    (gpiod_get_value(data->gpio_rev_3) << 3);

	/*
	* Hardware Type -- used to distinguish between Main and Remote units
	* where major blocks of hardware may not be installed on a Remote, or
	* an interface is initialized and used differently (like the display).
	* This function returns the following values:
	*	 0    -> Main
	*	 1    -> Remote
	*	 2..7 -> undefined
	*/
	data->type = (gpiod_get_value(data->gpio_type_0) << 0) |
		     (gpiod_get_value(data->gpio_type_1) << 1) |
		     (gpiod_get_value(data->gpio_type_2) << 2);

	dev_info(dev, "Cicada Platform Hardware Rev %d, Id %d\n", data->rev, data->type);

	system_rev &= ~(0x0f00);
	system_rev |= (data->rev << 8);

	ret = sysfs_create_group(&dev->kobj, &dev_attr_group);
	if (ret) {
		dev_err(dev, "%s, sysfs_create_group failed\n", __func__);
		return -ENODEV;
	}

	return 0;
}

static int cicada_hw_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	sysfs_remove_group(&dev->kobj, &dev_attr_group);
	return 0;
}

static struct platform_driver cicada_hw_driver = {
	.driver = {
		.name = "cicada-hw-pins",
		.of_match_table = cicada_device_ids,
	},
	.probe	= cicada_hw_probe,
	.remove	= cicada_hw_remove,
};

static int __init cicada_hw_init(void)
{
	return platform_driver_probe(&cicada_hw_driver, cicada_hw_probe);
}
subsys_initcall(cicada_hw_init);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("Cicada Hardware Type");

