/*
 * MMA7660FC Three-Axis Digital Accelerometer
 *
 * Copyright (C) 2009 ?????????????????????????????
 * Licensed under the GPL-2 or later.
 */

#include <linux/device.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/i2c.h>
#include <linux/irq.h>
#include <linux/slab.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/of_device.h>

/* MMA7660 Register Map */
#define XOUT_REG	0x00	/* R   6-bit output value X */
#define YOUT_REG	0x01	/* R   6-bit output value Y */
#define ZOUT_REG	0x02	/* R   6-bit output value Z */
#define TILT_REG	0x03	/* R   tilt status */
#define SRST_REG	0x04	/* R   sample rate status */
#define SPCNT_REG	0x05	/* R/W sleep count */
#define INTSU_REG	0x06	/* R/W interrupt setup */
#define MODE_REG	0x07	/* R/W mode */
#define SR_REG		0x08	/* R/W sample rate */
#define PDET_REG	0x09	/* R/W tap/pulse detection */
#define PD_REG		0x0A	/* R/W tap/pulse debounce count */

/* XOUT, YOUT, ZOUT register definitions */
#define XYZ_ALERT	(1 << 6)
#define XYZ_SHIFT	2
#define XYZ_MIN		-32
#define XYZ_MAX		32

/* TILT register bit / bit field definitions */
#define TILT_BAFRO(x)		(x & 0x03)
#define TILT_BAFRO_FRONT	0x01
#define TILT_BAFRO_BACK		0x02
#define TILT_POLA(x)	((x >> 2) & 0x07)
#if 0
/* hardware orientation */
#define TILT_POLA_LEFT	0x01
#define TILT_POLA_RIGHT	0x02
#define TILT_POLA_DOWN	0x05
#define TILT_POLA_UP	0x06
#else
/* Mantis orientation */
#define TILT_POLA_LEFT	0x06
#define TILT_POLA_RIGHT	0x05
#define TILT_POLA_DOWN	0x02
#define TILT_POLA_UP	0x01
#endif
#define TILT_TAP	(1 << 5)
#define TILT_ALERT	(1 << 6)
#define TILT_SHAKE	(1 << 7)

/* SRST register bit definitions */
#define SRST_AMSRS	(1 << 0)
#define SRST_AWSRS	(1 << 1)

/* INTSU register bit definitions */
#define INTSU_FBINT	(1 << 0)
#define INTSU_PLINT	(1 << 1)
#define INTSU_PDINT	(1 << 2)
#define INTSU_ASINT	(1 << 3)
#define INTSU_GINT	(1 << 4)
#define INTSU_SHINTZ	(1 << 5)
#define INTSU_SHINTY	(1 << 6)
#define INTSU_SHINTX	(1 << 7)

/* MODE register bit definitions */
#define MODE_MODE	(1 << 0)
#define MODE_TON	(1 << 2)
#define MODE_AWE	(1 << 3)
#define MODE_ASE	(1 << 4)
#define MODE_SCPS	(1 << 5)
#define MODE_IPP	(1 << 6)
#define MODE_IAH	(1 << 7)

/* SR register bit / bit field definitions */
#define SR_AMSR(x)	(x & 0x07)
#define AMSR_AMPD	0
#define AMSR_AM64	1
#define AMSR_AM32	2
#define AMSR_AM16	3
#define AMSR_AM8	4
#define AMSR_AM4	5
#define AMSR_AM2	6
#define AMSR_AM1	7
#define SR_AWSR(x)	((x & 0x03) << 3)
#define AWSR_AW32	0
#define AWSR_AW16	1
#define AWSR_AW8	2
#define AWSR_AW1	3
#define SR_FILT(x)	((x & 0x07) << 5)
#define FILT_NODB	0
#define FILT_DB2	1
#define FILT_DB3	2
#define FILT_DB4	3
#define FILT_DB5	4
#define FILT_DB6	5
#define FILT_DB7	6
#define FILT_DB8	7

struct mma7660_data {
	struct i2c_client	*client;
	struct device		*dev;
	struct input_dev	*idev;
	struct gpio_desc	*irq1;
	struct gpio_desc	*irq2;
	int			opened;
	int			last_key;
	unsigned int		debug_level;
};

struct mma7660_rawdata {
	s8 x;
	s8 y;
	s8 z;
	u8 tilt;
};

static int mma7660_write_reg(struct i2c_client *client, u8 command, u8 value)
{
	int ret;
	int i;

	for (i = 0; i < 3; ++i) {
		ret = i2c_smbus_write_byte_data(client, command, value);
		if (ret == 0)
			break;
		else
			dev_dbg(&client->dev, "%s, retry %d\n", __func__, i+1);

		udelay(10);
	}
	if (ret < 0)
		dev_dbg(&client->dev, "%s, failed\n", __func__);

	return ret;
}

static int mma7660_get_data(struct i2c_client *client, struct mma7660_rawdata *rdata)
{
	u8 data[4]; /* xout, yout, zout, tilt */
	int ret;

	memset(rdata, 0, 4);

	ret = i2c_smbus_read_i2c_block_data(client, XOUT_REG, 4, data);
	if (ret != 4) {
		dev_err(&client->dev, "%s: read error (%d)\n", __func__, ret);
		return -EIO;
	}

	rdata->x = (s8)(data[0] & 0x3f);
	rdata->y = (s8)(data[1] & 0x3f);
	rdata->z = (s8)(data[2] & 0x3f);
	rdata->tilt = data[3];

	return 0;
}

/* touch screen irq handler */
static irqreturn_t mma7660_handler(int irq, void *_data)
{
	struct mma7660_data *data = (struct mma7660_data *)_data;
	struct input_dev *idev = data->idev;
	struct mma7660_rawdata rawdata;
	int key = 0;

	if (mma7660_get_data(data->client, &rawdata)) {
		dev_dbg(data->dev, "%s, mma7660_get_data failed\n", __func__);
		goto exit;
	}

	if (data->debug_level > 0) {
		char buffer[100];
		int i = 0;
		buffer[0] = 0;
		switch TILT_BAFRO(rawdata.tilt) {
		case TILT_BAFRO_FRONT:
			i += sprintf(&buffer[i], "front ");
			break;
		case TILT_BAFRO_BACK:
			i += sprintf(&buffer[i], "back ");
			break;
		default:
			i += sprintf(&buffer[i], "unknown ");
			break;
		}

		switch TILT_POLA(rawdata.tilt) {
		case TILT_POLA_LEFT:
			i += sprintf(&buffer[i], "left ");
			break;
		case TILT_POLA_RIGHT:
			i += sprintf(&buffer[i], "right ");
			break;
		case TILT_POLA_UP:
			i += sprintf(&buffer[i], "up ");
			break;
		case TILT_POLA_DOWN:
			i += sprintf(&buffer[i], "down ");
			break;
		default:
			i += sprintf(&buffer[i], "unknown ");
			break;
		}

		if (rawdata.tilt & TILT_SHAKE)
			i += sprintf(&buffer[i], "shake ");

		if (rawdata.tilt & TILT_TAP)
			i += sprintf(&buffer[i], "tap ");

		dev_dbg(&data->client->dev, "%s %u, tilt(%02x), (%3d, %3d, %3d) %s\n",
			__func__,
		       jiffies_to_msecs(jiffies),
		       rawdata.x, rawdata.y, rawdata.z,
		       rawdata.tilt,
		       buffer);
	}

	switch TILT_POLA(rawdata.tilt) {
	case TILT_POLA_LEFT:
		key = BTN_WEST;
		break;
	case TILT_POLA_RIGHT:
		key = BTN_EAST;
		break;
	case TILT_POLA_UP:
		key = BTN_NORTH;
		break;
	case TILT_POLA_DOWN:
		key = BTN_SOUTH;
		break;
	default:
		break;
	}

	input_report_key(idev, key, true);
	input_sync(idev);
	input_report_key(idev, key, false);

	input_sync(idev);
	
exit:
	return IRQ_HANDLED;
}

static ssize_t debug_store(struct device *dev, struct device_attribute *attr,
			   const char *buf, size_t n)
{
	struct mma7660_data *data = dev_get_drvdata(dev);
	unsigned int value;
	if (kstrtouint(buf, 10, &value) != 0) {
		dev_err(dev, "%s, invalid value\n", __func__);
		return -EINVAL;
	}

	data->debug_level = value;

	return n;
}

static DEVICE_ATTR_WO(debug);

static struct attribute *dev_attrs[] = {
	&dev_attr_debug.attr,
	NULL
};

static const struct attribute_group dev_attr_group = {
	.attrs = dev_attrs,
};

static int mma7660_dev_open(struct input_dev *dev)
{
	struct mma7660_data *data = input_get_drvdata(dev);

/*	dev_dbg(&data->client->dev, "%s\n", __func__); */

	if (data->opened > 0) {
		data->opened++;
		return 0;
	}

	data->opened = 1;

	return 0;
}

static void mma7660_dev_close(struct input_dev *dev)
{
	struct mma7660_data *data = input_get_drvdata(dev);

/*	dev_dbg(&data->client->dev, "%s\n", __func__); */

	if (data->opened == 0) {
		dev_info(&data->client->dev, "unbalanced close\n");
		return;
	}

	data->opened--;
}

static int mma7660_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	struct mma7660_data *data;
	struct device *dev = &client->dev;
	u8 test;
	int ret;

	dev_dbg(dev, "%s\n", __func__);

	ret = i2c_smbus_read_i2c_block_data(client, XOUT_REG, 1, &test);
	if (ret != 1)
		return ret;
		
	data = devm_kzalloc(dev, sizeof(*data), GFP_KERNEL);
	if (!data)
		return PTR_ERR(data);

	data->irq1 = devm_gpiod_get(dev, "irq1", GPIOD_IN);
	if (IS_ERR(data->irq1)) {
		dev_err(dev, "could not get irq1 gpio\n");
		return PTR_ERR(data->irq1);
	}

	data->irq2 = devm_gpiod_get(dev, "irq2", GPIOD_IN);
	if (IS_ERR(data->irq2)) {
		dev_err(dev, "could not get irq2 gpio\n");
		return PTR_ERR(data->irq2);
	}

	data->dev = dev;
	data->client = client;
	data->last_key = 999999;

	/* initialize device */
	mma7660_write_reg(client, MODE_REG, 0);
	mma7660_write_reg(client, SR_REG,
			  SR_FILT(FILT_DB4) | SR_AWSR(AWSR_AW8) |
			  SR_AMSR(AMSR_AM8));
	mma7660_write_reg(client, INTSU_REG,
			  INTSU_FBINT | INTSU_PLINT |
			  INTSU_SHINTX | INTSU_SHINTY | INTSU_SHINTZ);
	mma7660_write_reg(client, MODE_REG,
			  MODE_MODE | MODE_AWE | MODE_ASE);

	data->idev = devm_input_allocate_device(dev);
	if (IS_ERR(data->idev)) {
		dev_err(dev, "devm_input_allocate_device failed\n");
		return PTR_ERR(data->idev);
	}

	dev_set_drvdata(dev, data);
	i2c_set_clientdata(client, data);
	input_set_drvdata(data->idev, data);

	data->idev->name = "mma7660 accelerometer";
	data->idev->phys = "mma7660";
	data->idev->open = mma7660_dev_open;
	data->idev->close = mma7660_dev_close;

	__set_bit(EV_KEY, data->idev->evbit);
	__set_bit(KEY_UP, data->idev->keybit);
	__set_bit(KEY_DOWN, data->idev->keybit);
	__set_bit(KEY_LEFT, data->idev->keybit);
	__set_bit(KEY_RIGHT, data->idev->keybit);

	ret = input_register_device(data->idev);
	if (ret) {
		dev_err(dev, "input_register_device failed\n");
		return ret;
	}

	ret = sysfs_create_group(&dev->kobj, &dev_attr_group);
	if (ret) {
		dev_err(dev, "%s, sysfs_create_group failed\n", __func__);
		return ret;
	}

	ret = devm_request_threaded_irq(dev, gpiod_to_irq(data->irq1),
				NULL, mma7660_handler,
				IRQF_TRIGGER_FALLING | IRQF_ONESHOT,
				"mma7660", data);
	if (ret) {
		dev_err(dev, "failed to request IRQ %d: %d\n",
			gpiod_to_irq(data->irq1), ret);
		return ret;
	}

	dev_info(dev, "probed, chip @ 0x%x (%s)\n",
		 client->addr, client->adapter->name);

	return 0;
}

static int mma7660_remove(struct i2c_client *client)
{
	struct device *dev = &client->dev;
	sysfs_remove_group(&dev->kobj, &dev_attr_group);
	return 0;
}

static const struct i2c_device_id i2c_device_ids[] = {
	{ "mma7660", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, i2c_device_ids);

static const struct of_device_id device_ids[] = {
	{ .compatible = "fnet,mma7660" },
	{},
};
MODULE_DEVICE_TABLE(of, device_ids);

struct i2c_driver mma7660_driver = {
	.driver = {
		.name = "mma7660",
		.of_match_table = device_ids,
	},
	.probe = mma7660_probe,
	.remove = mma7660_remove,
	.id_table = i2c_device_ids,
};

module_i2c_driver(mma7660_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("MMA7660 accelerometer driver");
