/*
 * Copyright (c) (2011) Fluke Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/printk.h>
#include <linux/module.h>
#include <linux/pm.h>
#include <linux/irq.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/reboot.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>

#define PWDN_ACTIVE_MAGIC_NUMBER	4465284

struct cicada_pwr_btn_data {
	struct device *dev;

	int wakeup;
	int suspend;
	int pwr_rdy;

	struct gpio_desc *pwr_dwn_rqst;
	struct gpio_desc *pwr_kill;
	struct gpio_desc *cpu_active;
	struct platform_device *pdev;
};

struct cicada_pwr_btn_data *cicada_pwr_btn_data;

static const struct of_device_id cicada_device_ids[] = {
	{ .compatible = "fnet,cicada-pwr-btn", .data = NULL },
	{}
};
MODULE_DEVICE_TABLE(of, cicada_device_ids);

static ssize_t pdwn_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	return sprintf(buf, "%u\n", PWDN_ACTIVE_MAGIC_NUMBER);
}

static ssize_t pdwn_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t n)
{
	struct cicada_pwr_btn_data *data = dev_get_drvdata(dev);
	static int active_state = 0;
	unsigned int value = simple_strtol(buf, NULL, 10);

	if (value != PWDN_ACTIVE_MAGIC_NUMBER) {
		printk(KERN_ERR "pwdn_active_store: Invalid value\n");
		return -EINVAL;
	}

	active_state = !active_state;
	pr_debug("%s %d\n", __func__, active_state);

	if (data->cpu_active)
		gpiod_set_value(data->cpu_active, active_state);

	return n;
}

static DEVICE_ATTR_RW(pdwn);

static void power_kill(void)
{
	pr_debug("%s\n", __func__);
	if (cicada_pwr_btn_data->pwr_kill)
		gpiod_set_value(cicada_pwr_btn_data->pwr_kill, 1);
}

static irqreturn_t cicada_pwr_btn_handler(int irqno, void *_data)
{
	struct cicada_pwr_btn_data *data = (struct cicada_pwr_btn_data *)_data;

	pr_debug("%s\n", __func__);

	if (!data->suspend)
		//orderly_poweroff(true);
		kernel_power_off();

	/* TBD: the concept of imx-active is to signal the PIC micro
	   that the imx is still alive and in the process of shutting
	   down.  a forced power off by the user would have a long timeout
	   in this case (maybe 30 seconds??).  if the imx is dead and
	   does not toggle this output, then the forced power off happens
	   much more quickly (maybe 10 seconds??) */
	if (data->cpu_active)
		gpiod_set_value(data->cpu_active, 1);

	return IRQ_HANDLED;
}

static int cicada_pwr_btn_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct cicada_pwr_btn_data *data;
	int ret = 0;

	dev_dbg(dev, "%s\n", __func__);

	if (!np)
		return -EINVAL;

	if (!of_device_is_available(np))
		return -ENODEV;

	data = devm_kzalloc(dev, sizeof(*data), GFP_KERNEL);
	if (!data) {
		dev_err(dev, "could not allocate object\n");
		return -ENOMEM;
	}

	/* kill power */
	data->pwr_kill = devm_gpiod_get(dev, "pwr-kill", GPIOD_OUT_LOW);
	if (IS_ERR(data->pwr_kill)) {
		dev_info(dev, "could not get pwr-kill gpio\n");
		data->pwr_kill = NULL;
	}

	/* set power off function */
	pm_power_off = power_kill;

	/* toggle during shutdown, output */
	data->cpu_active = devm_gpiod_get(dev, "cpu-active", GPIOD_OUT_LOW);
	if (IS_ERR(data->cpu_active)) {
		dev_info(dev, "could not get cpu-active gpio\n");
		data->cpu_active = NULL;
	}

	/* power down request */
	data->pwr_dwn_rqst = devm_gpiod_get(dev, "irq", GPIOD_IN);
	if (IS_ERR(data->pwr_dwn_rqst)) {
		dev_err(dev, "could not get irq gpio\n");
		return PTR_ERR(data->pwr_dwn_rqst);
	}

	data->dev = dev;
	dev_set_drvdata(dev, data);

	cicada_pwr_btn_data = data;

	data->wakeup = 1;
	data->suspend = 0;

	device_init_wakeup(dev, data->wakeup);

	ret = device_create_file(dev, &dev_attr_pdwn);
	if (ret) {
		dev_err(dev, "%s, device_create_file failed\n", __func__);
		return ret;
	}

	/* Register interrupts */
	ret = devm_request_threaded_irq(dev, gpiod_to_irq(data->pwr_dwn_rqst),
				NULL, cicada_pwr_btn_handler,
				IRQF_TRIGGER_FALLING | IRQF_ONESHOT,
				"pwr_dwn_rqst", data);
	if (ret) {
		dev_err(dev, "failed to request pwr_dwn_rqst IRQ %d: %d\n",
			gpiod_to_irq(data->pwr_dwn_rqst), ret);
		return ret;
	}

	return 0;
}

static int cicada_pwr_btn_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	device_remove_file(dev, &dev_attr_pdwn);

	return 0;
}

static struct platform_driver cicada_pwr_btn_driver = {
	.driver = {
		.name = "cicada-pwr-btn",
		.owner = THIS_MODULE,
		.of_match_table = cicada_device_ids,
	},
	.probe	  = cicada_pwr_btn_probe,
	.remove   = cicada_pwr_btn_remove,
};

module_platform_driver(cicada_pwr_btn_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("FNet");
MODULE_DESCRIPTION("Cicada Power Button");



