/*******************************************************************************
* Copyright (c) 2004, 2012 Alternative Ideas Corporation. All rights reserved.
* This program and the accompanying materials are protected by United
* States and International copyright laws. They may not be inspected,
* copied, modified, transmitted, executed, or used in any way without
* the express written permission of the copyright holder.
*******************************************************************************/

/**
* Implementation of standard shell context menu for Windows OS
*/

extern "C" {
#include "BaseFileSystemWrapper.h"
#include "FileSystemWindowsShared.h"
#include "FileSystem.h"
}

#include <windows.h>
#include <shobjidl.h> //For IShellItemImageFactory
#include <shlobj.h> // For SHGetDesktopFolder
#include <strsafe.h> // For StringCchCopyW

// It is not supported on OS less than Windows Vista
typedef HRESULT (* SHCreateItemFromParsingNameProc) (PCWSTR, IBindCtx*, REFIID, void**);

static SHCreateItemFromParsingNameProc pfSHCreateItemFromParsingName = NULL;
static BOOL g_isInitInt = FALSE;

static void initStatic() {
    HMODULE h;
    if (g_isInitInt) {
        return;
    }
    // get handle to kernel32
    if (GetModuleHandleExW((GET_MODULE_HANDLE_EX_FLAG_FROM_ADDRESS |
        GET_MODULE_HANDLE_EX_FLAG_UNCHANGED_REFCOUNT),
        (LPCWSTR)&SHGetDesktopFolder, &h) != 0) {
            // Get address of CreateSymbolicLinkW (>= Windows Vista)
            pfSHCreateItemFromParsingName = (SHCreateItemFromParsingNameProc)GetProcAddress(h, "SHCreateItemFromParsingName");
            DW_LOG(L"Init SHCreateItemFromParsingName");
    } else {
        DW_LOG(L"Cannot load Shell32 module");
    }
    g_isInitInt = TRUE;
}


// http://msdn.microsoft.com/en-us/library/aa289172%28VS.71%29.aspx
// http://www.camaswood.com/tech/get-an-hbitmap-file-preview-using-iextractimage/
static jlong getThumbnailXP(JNIEnv* env, const SIZE& size, WCHAR* nPathDir, WCHAR* nFile) {

    // Get IShellFolder of desktop (root) folder -
    // we need it only to resolve required path
    IShellFolder* pDesktop = NULL;
    HRESULT hr = SHGetDesktopFolder(&pDesktop);
    jlong jBitmapOut = 0;
    if (SUCCEEDED(hr)) {
        // Get PIDL of directory
        LPITEMIDLIST pListDir = NULL;
        hr = pDesktop->ParseDisplayName(NULL, NULL, nPathDir, NULL, &pListDir, NULL);
        if (SUCCEEDED(hr)) {
            // Get IShellFolder of nPathDir
            IShellFolder* pSub = NULL;
            hr = pDesktop->BindToObject(pListDir, NULL, IID_IShellFolder, (void**)&pSub);
            if (SUCCEEDED(hr)) {
                // Get PIDL of file
                LPITEMIDLIST pListFile = NULL;
                hr = pSub->ParseDisplayName(NULL, NULL, nFile, NULL, &pListFile, NULL);
                if (SUCCEEDED(hr)) {
                    // Get IExtractImage
                    LPCITEMIDLIST pidl = pListFile; // this is const
                    IExtractImage* pIExtract = NULL;
                    hr = pSub->GetUIObjectOf(NULL, 1, &pidl, IID_IExtractImage, NULL, (void**)& pIExtract);
                    if (SUCCEEDED(hr)) {
                        // The IEIFLAG_ORIGSIZE flag tells it to use the original aspect
                        // ratio for the image size. The IEIFLAG_QUALITY flag tells the 
                        // interface we want the image to be the best possible quality.
                        DWORD flags = IEIFLAG_ORIGSIZE | IEIFLAG_QUALITY; 
                        // The recommended color depth in units of bits per pixel
                        DWORD colorDepth = 32;

                        // Gets a path to the image that is to be extracted.
                        OLECHAR pathBuffer[MAX_PATH];
                        hr = pIExtract->GetLocation(pathBuffer, MAX_PATH, NULL, &size, colorDepth, &flags);         
                        if (SUCCEEDED(hr)) {
                            HBITMAP hBitmap = NULL;
                            hr = pIExtract->Extract(&hBitmap);
                            // It is possible for Extract to fail if there is no thumbnail image
                            // so we will not throw exception here
                            if (SUCCEEDED(hr)) {
                                jBitmapOut = (jlong)hBitmap;
                            } else {
                                DW_LOG(L"IExtractImage.Extract failed with hr=0x%X", hr);
                            }
                        } else {
                            DW_LOG(L"Get IExtractImage failed with hr=0x%X", hr);
                        }
                        pIExtract->Release();
                    } else {
                        // TODO Milen: Should we throw exception or silently return 0
                        DW_LOG(L"Get GetUIObjectOf failed with hr=0x%X", hr);
                        //throwExceptionFromErrorWithMsg(env, hr, "GetUIObjectOf failed:", nFile);
                    }
                    // When it is no longer needed, it is the responsibility of the caller to free this resource
                    CoTaskMemFree(pListFile);
                } else {
                    throwExceptionFromErrorWithMsg(env, hr, "Resolve file failed:", nFile);
                }
                pSub->Release();
            } else {
                throwExceptionFromErrorWithMsg(env, hr, "BindToObject failed:", nPathDir);
            }
            // When it is no longer needed, it is the responsibility of the caller to free this resource
            CoTaskMemFree(pListDir);
        } else {
            throwExceptionFromErrorWithMsg(env, hr, "Resolve of directory failed:", nPathDir);
        }
        pDesktop->Release();
    } else {
        throwExceptionFromErrorWithMsg(env, hr, "GetDesktopFolder failed:", NULL);
    }
    
    return jBitmapOut;
}


/*
* Class:     com_deltopia_internal_io_win32_FileSystem
* Method:    nGetThumbnail
* Signature: (Ljava/lang/String;III)J
*/
JNIEXPORT jlong JNICALL Java_com_deltopia_internal_io_win32_FileSystem_nGetThumbnail
    (JNIEnv* env, jobject jFileSystem, jstring jPath, jint maxWidth, jint maxHeight, jint flags) {

    DW_LOG(L">>> Get Thumbnail");
    HRESULT hr = S_OK;

    // In theory this method should not be called if InitStatic was failed, but for safety
    if (g_InitRes != LIB_LOAD_SUCCESS) {
        JNU_ThrowIllegalStateException(env, "Not initialized");
        return 0;
    }
    // Check input arguments
    if (NULL == jPath) {
        JNU_ThrowNullPointerException(env, "Null path");
        return 0;
    }

    WCHAR* nPath = (WCHAR*)env->GetStringChars(jPath, NULL);
    if (nPath == NULL) {
        JNU_ThrowIllegalArgumentException(env, "Invalid input path");
        return NULL;
    }

    SIZE size = { maxWidth, maxHeight };
    jlong hBitmapOut = 0;

    // IShellItemImageFactory is since Vista, so we will have fallback for XP
    if (NULL != pfSHCreateItemFromParsingName) {
        DW_LOG(L"Use SHCreateItemFromParsingName");
        // Getting the IShellItemImageFactory interface pointer for the file.
        IShellItemImageFactory *pImageFactory;
        //hr = SHCreateItemFromParsingName((WCHAR*)nPath, NULL, IID_PPV_ARGS(&pImageFactory));
        hr = (*pfSHCreateItemFromParsingName)((WCHAR*)nPath, NULL, IID_PPV_ARGS(&pImageFactory));
        if (SUCCEEDED(hr)) {
            //size - Max size of the image
            HBITMAP hbmp;
            hr = pImageFactory->GetImage(size, flags, &hbmp);
            if (SUCCEEDED(hr)) {
                hBitmapOut = (jlong)hbmp;
            } else {
                throwExceptionFromError(env, hr, (WCHAR*)nPath);
            }
            pImageFactory->Release();
        } else {
            JNU_ThrowIOException(env, "Cannot create shell item from path");
        }
    } else {
        // nPath=F:\_Temp\Sample.png
        WCHAR* nFile = wcsrchr(nPath, '\\');
        size_t pos = nFile != NULL ? (nFile - nPath) : -1;
        size_t len = wcslen(nPath);
        if (pos > 0 && pos < (len - 1)) {
            DW_LOG(L"Path separator path=%s, pos=%d, len=%d", nPath, pos, len);
            WCHAR* nPathDir = (WCHAR*)malloc(sizeof(WCHAR)*(pos + 1));
            // nPathDir=F:\_Temp
            if (nPathDir != NULL) {
                hr = StringCchCopyN(nPathDir, pos + 1, nPath, pos);
                if (SUCCEEDED(hr)) {
                    // \Sample.png -> Sample.png
                    nFile++; 
                    DW_LOG(L"Get thumbnail full=%s, dir=%s, file=%s", nPath, nPathDir, nFile);
                    hBitmapOut = getThumbnailXP(env, size, nPathDir, nFile);
                } else {
                    throwExceptionFromErrorWithMsg(env, hr, "Copy path failed:", nPath);
                }
            } else {
                JNU_ThrowOutOfMemoryError(env, "Cannot allocate native paths array");
            }
        } else {
            JNU_ThrowIllegalArgumentException(env, "Invalid path");
        }
    }
    env->ReleaseStringChars(jPath, (const jchar*)nPath);

    DW_LOG(L"<<< Get Thumbnail %ld", hBitmapOut);
    return hBitmapOut;
}