/*******************************************************************************
* Copyright (c) 2004, 2012 Alternative Ideas Corporation. All rights reserved.
* This program and the accompanying materials are protected by United
* States and International copyright laws. They may not be inspected,
* copied, modified, transmitted, executed, or used in any way without
* the express written permission of the copyright holder.
*******************************************************************************/

// Implementation of standard shell context menu for Windows OS
#include "ShellMenu.h"


// References:
// http://netez.com/2xExplorer/shellFAQ/bas_context.html
// http://msdn.microsoft.com/en-us/library/windows/desktop/bb776095%28v=vs.85%29.aspx
// http://www.codeproject.com/Articles/4025/Use-Shell-ContextMenu-in-your-applications
// http://blogs.msdn.com/b/oldnewthing/archive/2004/09/20/231739.aspx
// http://www.codeproject.com/Articles/22012/Explorer-Shell-Context-Menu
// http://stackoverflow.com/questions/3777121/how-to-access-windows-shell-context-menu-items
// http://stackoverflow.com/questions/11346987/open-windows-explorer-shell-context-menu



#define MIN_ID 10000
#define MAX_ID 20000


#define DW_SAFE_RELEASE(_Pointer)               \
    if (NULL != _Pointer) {                     \
        _Pointer->Release();                    \
        _Pointer = NULL;                        \
    }

#define SAFE_DELETE(_Pointer)                   \
    if (NULL != _Pointer) {                     \
        delete _Pointer;                        \
        _Pointer = NULL;                        \
    }


ShellMenu* g_pShellMenu = NULL;

ShellMenu::ShellMenu(JNIEnv *env, HWND hWnd, HMENU hMenu)
{
    DW_ASSERT(NULL!=env);
    DW_ASSERT(NULL!=hWnd);
    m_env = env;
    m_hWnd = hWnd;

    m_nItemsCount = 0;
    m_psfFolder = NULL;
    m_pidlArray = NULL;
    m_iMenuType = 0;
    m_pContextMenu = NULL;
    m_OldWndProc = NULL;

    // If input menu is NULL create one as root
    if( NULL == hMenu ) {
        m_hMenu = ::CreatePopupMenu ();
        // Mark to destroy in destructor
        m_bDestroyMenu = TRUE;
    } else {
        m_bDestroyMenu = FALSE;
        m_hMenu = hMenu;
    }
    g_pShellMenu = this;
    DW_LOG(L"+++ Create 0x%X", this);
}

ShellMenu::~ShellMenu()
{
    DW_LOG(L"--- Destroy 0x%X", this);
    g_pShellMenu = NULL;

    // Restore old windows procedure
    if (m_OldWndProc) { 
        SetWindowLongPtr (m_hWnd, GWLP_WNDPROC, (LONG_PTR)m_OldWndProc);
        m_OldWndProc = NULL;
    }

    // Free all allocated data
    DW_SAFE_RELEASE(m_psfFolder);

    if (m_pidlArray) {
        FreePIDLArray (m_pidlArray);
        m_pidlArray = NULL;
    }

    if(m_bDestroyMenu && m_hMenu) {
        DestroyMenu( m_hMenu );
        m_hMenu = NULL;
    }

    DW_SAFE_RELEASE(m_pContextMenu);
}


LRESULT CALLBACK ShellMenu::HookWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    //DW_LOG(L"\tWndProc %d (0x%X) wParam=0x%X lParam=0x%X", message, message, wParam, lParam);
    LRESULT lResult = 0;
    if (g_pShellMenu->HandleMessage(message, wParam, lParam, &lResult)) {
        return lResult;
    }

    // Call original WndProc of the subclassed window, instead of DefWindowProc()
    // in order to prevent undefined behavior of window
    return ::CallWindowProc(g_pShellMenu->m_OldWndProc, 
        g_pShellMenu->m_hWnd, 
        message, 
        wParam, 
        lParam); // Vista fix
}

BOOL ShellMenu::HandleMessage(const UINT message, const WPARAM wParam, const LPARAM lParam, LRESULT* pResultOut)
{
    *pResultOut = 0;
    //DW_LOG(L"\t HandleMessage %d (0x%X) wParam=0x%X lParam=0x%X", message, message, wParam, lParam);
    HRESULT hr;

    switch (message) {
    // Implement IContextMenu3 if your shortcut menu extension needs to process the WM_MENUCHAR message.
    case WM_MENUCHAR:  
        if (3 == m_iMenuType) {
            hr = ((IContextMenu3*)m_pContextMenu)->HandleMenuMsg2 (message, wParam, lParam, pResultOut);
            if (SUCCEEDED(hr)) {
                return TRUE;
            }
        }
        break;

    case WM_DRAWITEM:
    case WM_MEASUREITEM:
        if (wParam) {
            break; // if wParam != 0 then the message is not menu-related
        }

    // WM_INITMENUPOPUP must be handled, else on XP the menu behaves unpredictably
    // e.g. crashes, misplaces menu items, mistargets commands, etc.
    case WM_INITMENUPOPUP:
        if (m_iMenuType >= 2) {
            hr = ((IContextMenu2*)m_pContextMenu)->HandleMenuMsg (message, wParam, lParam);
            if (SUCCEEDED(hr)) {
                // inform caller that we handled WM_INITPOPUPMENU by ourself
                *pResultOut = (message == WM_INITMENUPOPUP ? 0 : TRUE);
                return TRUE; 
            } else {
                //DW_LOG(L"IContextMenu2.HandleMenuMsg failed 0x%X on message=%d", hr, message);
            }
        }
        break;

    case WM_COMMAND:
        DW_LOG(L"WM_COMMAND %d %d", wParam, lParam);
        if (HIWORD(wParam) ==0) { //Menu
            int iCmd = LOWORD(wParam);
            DW_LOG(L"Menu WM_COMMAND %d %d %d", wParam, lParam, iCmd);
            if (iCmd >= MIN_ID && iCmd <= MAX_ID) {
                if (InvokeCommand(iCmd)) {
                    return TRUE;
                }
            }
        }
        break;

    default:
        break;
    }
    return FALSE;
}


// Returns the higest/newest available version of IContextMenu for th previously
// cached PIDLs (i.e. SetObjects must have already been called.)
// On success: returns an integer from 13, indicating the version of the return 
// IContextMenu menu pointer.
// On failure: returns 0 and throws a Java exception
int ShellMenu::GetContextMenu (LPCONTEXTMENU* ppContextMenu)
{
    DW_ASSERT(NULL != m_pidlArray);
    DW_ASSERT(m_nItemsCount>0);

    *ppContextMenu = NULL;
    LPCONTEXTMENU pContextMenu1 = NULL;
    int iMenuType = 0;

    // Call GetUIObjectOf to retrieve a context menu. A new COM object is created 
    // for file items specified by m_pidlArray. Each IContextMenu is closely linked
    // to the folder items used to initialize it, which are eventually used as the 
    // arguments of whatever command is to be executed later. First retrieve the
    // normal IContextMenu interface (every object should have it)
    HRESULT hr = m_psfFolder->GetUIObjectOf (NULL, 
        m_nItemsCount, 
        (LPCITEMIDLIST *) m_pidlArray, 
        IID_IContextMenu, 
        NULL, 
        (void**) &pContextMenu1);

    if ( SUCCEEDED(hr) ) {   
        DW_LOG(L"Got IContextMenu ver 1");

        // We already have an IContextMenu interface, now try obtaining any of the 
        // higher version interfaces as each "higher-version" interface derives 
        // from its older version. IContextMenu3 was introduced for keyboard shortcuts.
        hr = pContextMenu1->QueryInterface (IID_IContextMenu3, (void**)ppContextMenu);
        if ( SUCCEEDED(hr) ) {
            DW_LOG(L"Got IContextMenu ver 3");
            iMenuType = 3;
        } else {
            // If IContextMenu3 failed, try IContextMenu2 - it was introduced for owner drawn menus
            hr = pContextMenu1->QueryInterface (IID_IContextMenu2, (void**)ppContextMenu);
            if (SUCCEEDED(hr)) {
                DW_LOG(L"Got IContextMenu ver 2");
                iMenuType = 2;
            } else {
                iMenuType = 1;
                // no higher IContextMenu versions found, version 1 will be the output
                *ppContextMenu = pContextMenu1; 
            }
        }

        if (iMenuType > 1) {
            // we can now release version 1 interface, cause we got a higher one
            pContextMenu1->Release(); 
        }                     
    } else {
        DW_LOG(L"GetUIObjectOf failed with hr=0x%X", hr);
        throwExceptionFromErrorWithMsg(m_env, hr, "Create context menu failed:", NULL);
    }

    return iMenuType;
}


void ShellMenu::AttachContextMenu(const BOOL bReplaceWinProc) {
    DW_LOG(L">>> AttachContextMenu");

    // Try to get the highest IContextMenu version available
    LPCONTEXTMENU pContextMenu = NULL;   // common pointer to IContextMenu and higher version interface
    int iMenuType = GetContextMenu(&pContextMenu);
    if (m_env->ExceptionCheck()) {
        // Not even IContextMenu version 1 was not found
        return;
    }
    DW_LOG(L"GetContextMenu type=%d", iMenuType);
    DW_ASSERT(iMenuType > 0);
    DW_ASSERT(pContextMenu);
    m_pContextMenu = pContextMenu;
    m_iMenuType = iMenuType;

    // Next step is to get an actual menu filled with items relevant to the selected
    // object. menuIndex indicates that new menu items (shell item) should be inserted
    // right at the end of latest item in hMenu. However, when it is time to execute
    // a command, the command identifier (ID) is used instead of the subitem offset
    // in the menu.
    const int menuIndex = ::GetMenuItemCount(m_hMenu);
    // CMF_NORMAL -  Indicates normal operation. A shortcut menu extension, namespace
    // extension, or drag-and-drop handler can add all menu items. In fact it is 0x00,
    // so we can skip. CMF_EXPLORE - The Windows Explorer tree window is present
    HRESULT hr = pContextMenu->QueryContextMenu(m_hMenu, 
        menuIndex, 
        MIN_ID, 
        MAX_ID, 
        CMF_NORMAL | CMF_EXPLORE);

    // Milen: If successful, QueryContextMenu returns an HRESULT value that has
    // its severity value set to SEVERITY_SUCCESS and its code value set to the
    // offset of the largest command identifier that was assigned, plus one. For
    // example, if idCmdFirst is set to 5 and you add three items to the menu 
    // with command identifiers of 5, 7, and 8, the return value should be 
    // MAKE_HRESULT(SEVERITY_SUCCESS, 0, 8 - 5 + 1). Otherwise, it returns a COM
    // error value.
    if (SUCCEEDED(hr)) {
        DW_LOG(L"QueryContextMenu succeeded, startIndex=%d", menuIndex);
        // subclass window to handle menurelated messages in CShellContextMenu
        if (bReplaceWinProc && iMenuType > 1) {  
            // only subclass if its version 2 or 3
            m_OldWndProc = (WNDPROC) SetWindowLongPtr (m_hWnd, GWLP_WNDPROC, (LONG_PTR) HookWndProc);
            DW_LOG(L"SetWindowLongPtr");
        } else {
            m_OldWndProc = NULL;
        }
    } else {
        DW_LOG(L"QueryContextMenu failed with hr=0x%X", hr);
        throwExceptionFromErrorWithMsg(m_env, hr, "Query context menu failed:", NULL);
    }

}

BOOL ShellMenu::InvokeCommand(int iCmd) {
    POINT pt = { 0, 0};
    ClientToScreen(m_hWnd, &pt);
    CMINVOKECOMMANDINFOEX info;
    ::ZeroMemory(&info, sizeof(info));
    info.cbSize = sizeof(info);
    info.fMask = CMIC_MASK_PTINVOKE; // remember the point for "properties"
    if (GetKeyState(VK_CONTROL) < 0) { // send key states (for delete command)
        info.fMask |= CMIC_MASK_CONTROL_DOWN;
    }
    if (GetKeyState(VK_SHIFT) < 0) {
        info.fMask |= CMIC_MASK_SHIFT_DOWN;
    }          
    info.hwnd = m_hWnd;
    // The returned idCommand belongs to shell menu entries so execute the related
    //  command. Note that this method receives an offset, not a command.
    info.lpVerb  = MAKEINTRESOURCEA(iCmd - MIN_ID);
    info.lpVerbW = MAKEINTRESOURCEW(iCmd - MIN_ID);
    info.nShow = SW_SHOWNORMAL;
    info.ptInvoke = pt; // pass the point to "properties"
    HRESULT hr = m_pContextMenu->InvokeCommand((LPCMINVOKECOMMANDINFO)&info);
    if (FAILED(hr)) {
        DW_LOG(L"InvokeCommandEx FAILED cmd=%d (0x%X) hr=0x%X", iCmd, iCmd, hr);
        DW_LOG_ERR(m_env, L"InvokeCommandEx FAILED", hr, L"Multiple paths");
        return FALSE;
    } else {
        DW_LOG(L"InvokeCommandEx %d (0x%X)", iCmd, iCmd);
        return TRUE;
    }
}

void ShellMenu::SetObjects( LPWSTR* paths, int count )
{
    DW_ASSERT(NULL != paths);
    DW_ASSERT(count>0);
    DW_ASSERT(NULL == m_psfFolder);
    DW_ASSERT(NULL == m_pidlArray);

    HRESULT hr = S_OK;
    int nValidCount = 0;
    for (int i = 0; i < count; i++) {
        LPITEMIDLIST pidl = NULL;
        SFGAOF sfgao;

        // SHParseDisplayName translates a Shell namespace object's display name
        // into a pointer to an item identifier list (PIDL) and returns the attributes
        // of the object.
        // ParseDisplayName is not expected to handle the relative path or parent
        //  folder indicators (".\" or "..\") so we should pass only canonical 
        // paths from java. IShellFolder::ParseDisplayName implicitly validates 
        // the existence of the item unless that behavior is overridden by a special
        // bind context parameter. So method will return error if item does not exist.
        // We will make not-strict i.e. the whole method fails only if ALL items fail.
        // http://msdn.microsoft.com/en-us/library/windows/desktop/bb762236%28v=vs.85%29.aspx
        // http://msdn.microsoft.com/en-us/library/windows/desktop/bb775090%28v=vs.85%29.aspx

        hr = SHParseDisplayName(paths[i],
            NULL,     // hwnd - use only if displaying a dlg or a msg box.
            &pidl,    // receives the item identifier list; NULL in the case of error.
            0,
            &sfgao); 

        if (SUCCEEDED(hr)) {
            DW_LOG(L"SHParseDisplayName %d: %s", i, paths[i]);

            LPITEMIDLIST pidlItem = NULL;  
            IShellFolder* psfFolder = NULL;
            // Get relative PIDL via SHBindToParent
            hr = SHBindToParent(pidl, IID_IShellFolder, (void **) &psfFolder, (LPCITEMIDLIST *) &pidlItem);
            if (SUCCEEDED(hr)) {
                LPITEMIDLIST* pidlArrayNew = (LPITEMIDLIST *) realloc (m_pidlArray, 
                    (nValidCount + 1) * sizeof (LPITEMIDLIST));
                if (NULL != pidlArrayNew) {
                    m_pidlArray = pidlArrayNew;
                    m_pidlArray[nValidCount] = CopyPIDL (pidlItem);   // copy relative pidl to pidlArray
                    // Milen: Note  SHBindToParent() does not allocate a new PIDL - it
                    // simply receives a pointer through this parameter - therefore 
                    // pidlItem needs not be freed it i.e. no ::CoTaskMemFree(pidlItem);
                    nValidCount++;
                }  else {
                    DW_LOG(L"Cannot allocate new PIDL array");
                    JNU_ThrowOutOfMemoryError(m_env, "Cannot allocate new PIDL array");
                }
                // Free the IShellFolder return from SHBindToParent
                // Keep first one to be used in GetUIObjectOf in GetContextMenu
                if (NULL == m_psfFolder) {
                    m_psfFolder = psfFolder;
                } else {
                    DW_SAFE_RELEASE(psfFolder);
                }
            } else {
                DW_LOG_ERR(m_env, L"SHBindToParent failed", hr, paths[i]);
            }
            // Milen: Do we need this? SHParseDisplayName does not mention this?
            // The topic should include whether and how the ID list returned in
            // ppidl should be freed by the caller. 
            ::CoTaskMemFree(pidl);
        } else {
            DW_LOG_ERR(m_env, L"SHParseDisplayName failed", hr, paths[i]);
        }
        // Stop on exception
        if (m_env->ExceptionCheck()) {
            break;
        }
    }
    m_nItemsCount = nValidCount;
    if (!m_env->ExceptionCheck() && nValidCount<=0) {
        // hr will be the error from latest path
        throwExceptionFromErrorWithMsg(m_env, hr, "No valid paths found:", NULL);
    }
    DW_LOG(L"<<< SetObjects %d items", m_nItemsCount);
}


void ShellMenu::FreePIDLArray(LPITEMIDLIST *pidlArray)
{
    if (NULL == pidlArray) {
        return;
    }

    const size_t nSize = _msize (pidlArray) / sizeof (LPITEMIDLIST);
    if (nSize != m_nItemsCount) {
        DW_LOG(L"PIDL array size %d differs from items count %d", nSize, m_nItemsCount);
    }
    for (size_t i = 0; i < nSize; i++) {
        // Allocated in CopyPIDL
        free(pidlArray[i]);
    }
    // Allocated in SetObjects by realloc
    free (pidlArray);
}


LPITEMIDLIST ShellMenu::CopyPIDL (LPCITEMIDLIST pidl, int cb)
{
    if (cb == -1) {
        cb = GetPIDLSize (pidl); // Calculate size of list.
    }
    // TODO Milen: I cannot find good information how to copy/free PIDL 
    // This code looks strange!
    LPITEMIDLIST pidlRet = (LPITEMIDLIST) calloc (cb + sizeof (USHORT), sizeof (BYTE));
    if (pidlRet != NULL) {
        CopyMemory(pidlRet, pidl, cb);
    }

    return (pidlRet);
}


UINT ShellMenu::GetPIDLSize (LPCITEMIDLIST pidl)
{  
    if (!pidl) {
        return 0;
    }

    int nSize = 0;
    LPITEMIDLIST pidlTemp = (LPITEMIDLIST) pidl;
    while (pidlTemp->mkid.cb) {
        nSize += pidlTemp->mkid.cb;
        pidlTemp = (LPITEMIDLIST) (((LPBYTE) pidlTemp) + pidlTemp->mkid.cb);
    }

    return nSize;
}