// ShellContextMenu.cpp: Implementierung CShellContextMenu.
//
//////////////////////////////////////////////////////////////////////

#include "ShellContextMenu.h"

extern "C" {
#include "BaseFileSystemWrapper.h"
#include "FileSystemWindowsShared.h"
}

/**
* References:
* http://netez.com/2xExplorer/shellFAQ/bas_context.html
* http://msdn.microsoft.com/en-us/library/windows/desktop/bb776095%28v=vs.85%29.aspx
* http://www.codeproject.com/Articles/4025/Use-Shell-ContextMenu-in-your-applications
* http://blogs.msdn.com/b/oldnewthing/archive/2004/09/20/231739.aspx
* http://www.codeproject.com/Articles/22012/Explorer-Shell-Context-Menu
* http://stackoverflow.com/questions/3777121/how-to-access-windows-shell-context-menu-items
* http://stackoverflow.com/questions/11346987/open-windows-explorer-shell-context-menu
*/

/////////////////////////////////////////////////////////////////////
// class to show shell context menu of files/folders/shell objects
// developed by R. Engels 2003
/////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

#define MIN_ID 1
#define MAX_ID 10000

IContextMenu2 * g_IContext2 = NULL;
IContextMenu3 * g_IContext3 = NULL;
WNDPROC g_OldWndProc; // Vista fix

CShellContextMenu::CShellContextMenu(JNIEnv *env)
{
    m_psfFolder = NULL;
    m_pidlArray = NULL;
    m_hMenu = NULL;
    m_env = env;
}

CShellContextMenu::~CShellContextMenu()
{
    // free all allocated datas
    if(!m_psfFolder) {
        m_psfFolder->Release ();
        m_psfFolder = NULL;
    }

    //   TODO RE12/27/05 This has been commented out because it produced random crashes. To be investigated!
    FreePIDLArray (m_pidlArray);
    m_pidlArray = NULL;

    if(m_hMenu) {
        DestroyMenu( m_hMenu );
        m_hMenu = NULL;
    }
}



/** 
 * This function determines which version of IContextMenu is available for those objects 
 * (always the highest one) and returns that interface
 * Should be called only if SetObjects was called 
 * On success: returns an integer from 13, indicating the version of the menu pointer. 
 * On failure: returns 0 and throw Java exception */
int CShellContextMenu::GetContextMenu (LPCONTEXTMENU* ppContextMenu)
{
    DW_ASSERT(m_pidlArray);
    DW_ASSERT(m_nItemsCount);

    *ppContextMenu = NULL;
    LPCONTEXTMENU pContextMenu1 = NULL;
    int iMenuType = 0;

    /** Call GetUIObjectOf to retrieve a context menu
     * a new COM object is created for file items specified by m_pidlArray
     * Each IContextMenu is closely linked with the folder items used to initialize it, 
     * which are eventually used as the arguments of whatever command is to be executed later.
     * first we should retrieve the normal IContextMenu interface (every object should have it) 
     * Unfortunately */
    HRESULT hr = m_psfFolder->GetUIObjectOf (NULL, m_nItemsCount, (LPCITEMIDLIST *) m_pidlArray, 
                                          IID_IContextMenu, NULL, (void**) &pContextMenu1);

    if ( SUCCEEDED(hr) ) {   
        DW_LOG(L"Got IContextMenu ver 1");

        /* Since we got an IContextMenu interface we can now obtain the higher version interfaces via that 
         * each "higher-version" interface derives from the previous in the list 
         * IContextMenu3 was introduced for keyboard shortcuts */
        hr = pContextMenu1->QueryInterface (IID_IContextMenu3, (void**)ppContextMenu);
        if ( SUCCEEDED(hr) ) {
            DW_LOG(L"Got IContextMenu ver 3");
            iMenuType = 3;
        } else {
            /* If IContextMenu3 failed, try IContextMenu2 - it was introduced for owner drawn menus */
            hr = pContextMenu1->QueryInterface (IID_IContextMenu2, (void**)ppContextMenu);
            if (SUCCEEDED(hr)) {
                DW_LOG(L"Got IContextMenu ver 2");
                iMenuType = 2;
            } else {
                iMenuType = 1;
                /* no higher versions were found, version 1 will be the output */
                *ppContextMenu = pContextMenu1; 
            }
        }

        if (iMenuType > 1) {
            /* we can now release version 1 interface, cause we got a higher one */
            pContextMenu1->Release(); 
        }                     
    } else {
        DW_LOG(L"GetUIObjectOf failed with hr=0x%X", hr);
        throwExceptionFromErrorWithMsg(m_env, hr, "Create context menu failed:", NULL);
    }
    return iMenuType;
}


LRESULT CALLBACK CShellContextMenu::HookWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
   switch (message)
   {
   case WM_MENUCHAR:   // only supported by IContextMenu3
      if (g_IContext3)
      {
         LRESULT lResult = 0;
         g_IContext3->HandleMenuMsg2 (message, wParam, lParam, &lResult);
         return (lResult);
      }
      break;

   case WM_DRAWITEM:
   case WM_MEASUREITEM:
      if (wParam)
         break; // if wParam != 0 then the message is not menu-related

   case WM_INITMENUPOPUP:
      if (g_IContext2)
         g_IContext2->HandleMenuMsg (message, wParam, lParam);
      else   // version 3
         g_IContext3->HandleMenuMsg (message, wParam, lParam);
      return (message == WM_INITMENUPOPUP ? 0 : TRUE); // inform caller that we handled WM_INITPOPUPMENU by ourself
      //break;

   default:
      break;
   }

    // call original WndProc of window to prevent undefined bevhaviour of window
    return ::CallWindowProc (g_OldWndProc , hWnd, message, wParam, lParam); // Vista fix
}


UINT CShellContextMenu::ShowContextMenu( HWND hWnd, HMENU hMenu, POINT point )
{

    /* Try to get IContextMenu higher version possible */
    LPCONTEXTMENU pContextMenu = NULL;   // common pointer to IContextMenu and higher version interface
    int iMenuType = GetContextMenu(&pContextMenu);
    if (m_env->ExceptionCheck()) {
        /* Even IContextMenu version 1 is not found */
        return 0;
    }
    DW_ASSERT(iMenuType > 0);
    DW_ASSERT(pContextMenu);

    /* If input menu is NULL create one as root */
    if( !hMenu ) {
        hMenu = ::CreatePopupMenu ();
        /* Cache only to destroy in destructor */
        m_hMenu = hMenu;
    }

    /* The next step is to get an actual menu filled with items relevant to the selected object.  */
    /* menuIndex indicates that new menu items (shell item) should be inserted right at the end of latest item in hMenu
    * However, when it is high time to execute some command, the command identifier (ID) is used instead of 
    * the subitem offset in the menu.*/
    const int menuIndex = ::GetMenuItemCount( hMenu );
    /* CMF_NORMAL -  Indicates normal operation. A shortcut menu extension, namespace extension, or drag-and-drop handler can add all menu items.
    * In fact it is 0x00, so we can skip.
    * CMF_EXPLORE - The Windows Explorer tree window is present */ 
    HRESULT hr = pContextMenu->QueryContextMenu( hMenu, menuIndex, MIN_ID, MAX_ID, CMF_NORMAL | CMF_EXPLORE);
    /*Yoli - both xp and win7 returns different HRESULT-s here; on 
    * Milen: If successful, QueryContextMenu returns an HRESULT value that has its severity value set to 
    SEVERITY_SUCCESS and its code value set to the offset of the largest command identifier that was assigned,
    plus one. For example, if idCmdFirst is set to 5 and you add three items to the menu with command identifiers
    of 5, 7, and 8, the return value should be MAKE_HRESULT(SEVERITY_SUCCESS, 0, 8 - 5 + 1). 
    Otherwise, it returns a COM error value. */

    UINT idCommand = 0;
    if (SUCCEEDED(hr)) {
        /* subclass window to handle menurelated messages in CShellContextMenu  */
        if (iMenuType > 1) {  
            /* only subclass if its version 2 or 3 */
            g_OldWndProc = (WNDPROC) SetWindowLongPtr (hWnd, GWLP_WNDPROC, (LONG_PTR) HookWndProc);
            if (iMenuType == 2) {
                g_IContext2 = (LPCONTEXTMENU2) pContextMenu;
            } else {   // version 3
                g_IContext3 = (LPCONTEXTMENU3) pContextMenu;
            }
        } else {
            g_OldWndProc = NULL;
        }

        DW_LOG(L"TrackPopupMenu on %dx%d, hWnd=0x%X, hMenu=0x%X", point.x, point.y, hWnd, hMenu);
        ::SetLastError(0);
        idCommand = TrackPopupMenu( hMenu, TPM_RETURNCMD | TPM_LEFTALIGN, point.x, point.y, 0, hWnd, NULL );
        const DWORD nError = !idCommand ? ::GetLastError() : 0;

        if (g_OldWndProc) { // unsubclass
            SetWindowLongPtr (hWnd, GWLP_WNDPROC, (LONG_PTR) g_OldWndProc);
        }

        /** 
         * If TPM_RETURNCMD is specified as we did in the uFlags parameter, the return value is 
         * the menu-item identifier of the item that the user selected. 
         * If the user cancels the menu without making a selection, 
         * or if an error occurs, the return value is zero. */
        if (!idCommand) {
            DW_LOG(L"TrackPopupMenu on %dx%d, hWnd=0x%X, hMenu=0x%X failed with error=0x%X", 
                point.x, point.y, hWnd, hMenu, nError);
            /* TODO Milen: It failed with 'Popup menu already active' and in fact InvokeCommand is not called */
/*
            if (nError != 0) {
                throwExceptionFromErrorWithMsg(m_env, nError, "Show context menu failed:", NULL);
            }
*/
        } else if (idCommand >= MIN_ID && idCommand <= MAX_ID) {  
            /* Returned idCommand belongs to shell menu entries 
            * so execute related command. Note that this method receives offset not command */
            InvokeCommand (pContextMenu, idCommand - MIN_ID);
            idCommand = 0;
        } else {
            DW_LOG(L"Unhandled command %d", idCommand);
        }
    } else {
        DW_LOG(L"QueryContextMenu failed with hr=0x%X", hr);
        throwExceptionFromErrorWithMsg(m_env, hr, "Query context menu failed:", NULL);
    }

    pContextMenu->Release();
    g_IContext2 = NULL;
    g_IContext3 = NULL;
    g_OldWndProc = NULL;

    return idCommand;
}


void CShellContextMenu::InvokeCommand (LPCONTEXTMENU pContextMenu, UINT idCmdOffset)
{
    CMINVOKECOMMANDINFO cmi;
    ::ZeroMemory(&cmi, sizeof(cmi));
    cmi.cbSize = sizeof (CMINVOKECOMMANDINFO);
    cmi.lpVerb = (LPSTR)MAKEINTRESOURCE(idCmdOffset);
    cmi.nShow = SW_SHOWNORMAL;

    pContextMenu->InvokeCommand (&cmi);
}


void CShellContextMenu::SetObjects( LPCTSTR* paths, int dim )
{
   DW_ASSERT(dim>0);
    // free all allocated datas
    if( m_psfFolder ) {
        m_psfFolder->Release ();
        m_psfFolder = NULL;
    }
    if (m_pidlArray) {
        FreePIDLArray (m_pidlArray);
        m_pidlArray = NULL;
    }

    // get IShellFolder interface of Desktop (root of shell namespace)
    IShellFolder * psfDesktop = NULL;
    HRESULT hr = SHGetDesktopFolder (&psfDesktop);   // needed to obtain full qualified pidl

    // ParseDisplayName creates a PIDL from a file system path relative to the IShellFolder interface
    // but since we use the Desktop as our interface and the Desktop is the namespace root
    // that means that it's a fully qualified PIDL, which is what we need
    LPITEMIDLIST pidl = NULL;

    WCHAR wchPath[MAX_PATH+1];
    // TODO Milen: Check return value
    hr = StringCchCopyW(wchPath, MAX_PATH+1, paths[0]);
    //if (IsTextUnicode ((LPVOID)paths[0], lstrlen (paths[0]), NULL))
    //   lstrcpy ((char *) wchPath, paths[0]);
    //else
    //   MultiByteToWideChar (CP_ACP, 0, paths[0], -1, wchPath,
    //                           sizeof (wchPath) / sizeof (WCHAR));

    hr = psfDesktop->ParseDisplayName (NULL, 0, wchPath, NULL, &pidl, NULL);

    // now we need the parent IShellFolder interface of pidl, and the relative PIDL to that interface
    LPITEMIDLIST pidlItem = NULL;   // relative pidl
    hr = SHBindToParentEx (pidl, IID_IShellFolder, (void **) &m_psfFolder, NULL);
    // TODO Milen Do we need this, was commented
    free (pidlItem);

    // get interface to IMalloc (need to free the PIDLs allocated by the shell functions)
    LPMALLOC lpMalloc = NULL;
    hr = SHGetMalloc (&lpMalloc);
    lpMalloc->Free (pidl);

    // now we have the IShellFolder interface to the parent folder specified in the first element in strArray
    // since we assume that all objects are in the same folder (as it's stated in the MSDN)
    // we now have the IShellFolder interface to every objects parent folder

    IShellFolder * psfFolder = NULL;
    //nItems = strArray.GetSize ();
    m_nItemsCount = dim;
    for (int i = 0; i < dim; i++)
    {
        // TODO Milen: Check return value
        hr = StringCchCopyW(wchPath, MAX_PATH+1, paths[i]);
        //if (IsTextUnicode ((LPVOID)paths[i], lstrlen (paths[i]), NULL))
        //   lstrcpy ((char *) wchPath, paths[i]);
        //else
        //   MultiByteToWideChar (CP_ACP, 0, paths[i], -1, wchPath,
        //                    sizeof (wchPath) / sizeof (WCHAR));

        hr = psfDesktop->ParseDisplayName (NULL, 0, wchPath, NULL, &pidl, NULL);

        m_pidlArray = (LPITEMIDLIST *) realloc (m_pidlArray, (i + 1) * sizeof (LPITEMIDLIST));
        // get relative pidl via SHBindToParent
        hr = SHBindToParentEx (pidl, IID_IShellFolder, (void **) &psfFolder, (LPCITEMIDLIST *) &pidlItem);
        m_pidlArray[i] = CopyPIDL (pidlItem);   // copy relative pidl to pidlArray
        free (pidlItem);
        lpMalloc->Free (pidl);      // free pidl allocated by ParseDisplayName
        psfFolder->Release ();
    }
    lpMalloc->Release ();
    psfDesktop->Release ();
}


void CShellContextMenu::FreePIDLArray(LPITEMIDLIST *pidlArray)
{
    if (!pidlArray)
        return;

    size_t iSize = _msize (pidlArray) / sizeof (LPITEMIDLIST);

    for (size_t i = 0; i < iSize; i++)
        free (pidlArray[i]);
    free (pidlArray);
}


LPITEMIDLIST CShellContextMenu::CopyPIDL (LPCITEMIDLIST pidl, int cb)
{
    if (cb == -1) {
        cb = GetPIDLSize (pidl); // Calculate size of list.
    }

    LPITEMIDLIST pidlRet = (LPITEMIDLIST) calloc (cb + sizeof (USHORT), sizeof (BYTE));
    if (pidlRet)
        CopyMemory(pidlRet, pidl, cb);

    return (pidlRet);
}


UINT CShellContextMenu::GetPIDLSize (LPCITEMIDLIST pidl)
{  
    if (!pidl) {
        return 0;
    }

    int nSize = 0;
    LPITEMIDLIST pidlTemp = (LPITEMIDLIST) pidl;
    while (pidlTemp->mkid.cb) {
        nSize += pidlTemp->mkid.cb;
        pidlTemp = (LPITEMIDLIST) (((LPBYTE) pidlTemp) + pidlTemp->mkid.cb);
    }
    return nSize;
}


// this is workaround function for the Shell API Function SHBindToParent
// SHBindToParent is not available under Win95/98
HRESULT CShellContextMenu::SHBindToParentEx (LPCITEMIDLIST pidl, REFIID riid, VOID **ppv, LPCITEMIDLIST *ppidlLast)
{
    HRESULT hr = 0;
    if (!pidl || !ppv)
        return E_POINTER;

    int nCount = GetPIDLCount (pidl);
    if (nCount == 0)   // desktop pidl of invalid pidl
        return E_POINTER;

    IShellFolder * psfDesktop = NULL;
    SHGetDesktopFolder (&psfDesktop);
    if (nCount == 1)   // desktop pidl
    {
        if ((hr = psfDesktop->QueryInterface(riid, ppv)) == S_OK)
        {
            if (ppidlLast) 
                *ppidlLast = CopyPIDL (pidl);
        }
        psfDesktop->Release ();
        return hr;
    }

    LPBYTE pRel = GetPIDLPos (pidl, nCount - 1);
    LPITEMIDLIST pidlParent = NULL;
    pidlParent = CopyPIDL (pidl, pRel - (LPBYTE) pidl);
    IShellFolder * psfFolder = NULL;

    if ((hr = psfDesktop->BindToObject (pidlParent, NULL, IID_IShellFolder, (void **) &psfFolder)) != S_OK)
    {
        free (pidlParent);
        psfDesktop->Release ();
        return hr;
    }
    if ((hr = psfFolder->QueryInterface (riid, ppv)) == S_OK)
    {
        if (ppidlLast)
            *ppidlLast = CopyPIDL ((LPCITEMIDLIST) pRel);
    }
    free (pidlParent);
    psfFolder->Release ();
    psfDesktop->Release ();

    return hr;
}


LPBYTE CShellContextMenu::GetPIDLPos (LPCITEMIDLIST pidl, int nPos)
{
    if (!pidl) {
        return 0;
    }

    int nCount = 0;

    BYTE * pCur = (BYTE *) pidl;
    while (((LPCITEMIDLIST) pCur)->mkid.cb)	{
        if (nCount == nPos)
            return pCur;
        nCount++;
        pCur += ((LPCITEMIDLIST) pCur)->mkid.cb;   // + sizeof(pidl->mkid.cb);
    }
    if (nCount == nPos) {
        return pCur;
    }

    return NULL;
}


int CShellContextMenu::GetPIDLCount (LPCITEMIDLIST pidl)
{
    if (!pidl) {
        return 0;
    }

    int nCount = 0;
    BYTE*  pCur = (BYTE *) pidl;
    while (((LPCITEMIDLIST) pCur)->mkid.cb) {
        nCount++;
        pCur += ((LPCITEMIDLIST) pCur)->mkid.cb;
    }

    return nCount;
}