/*******************************************************************************
 * Copyright (c) 2004, 2012 Alternative Ideas Corporation. All rights reserved.
 * This program and the accompanying materials are protected by United
 * States and International copyright laws. They may not be inspected,
 * copied, modified, transmitted, executed, or used in any way without
 * the express written permission of the copyright holder.
 *******************************************************************************/

/**
 * Implementation of standard shell context menu for Windows OS
 */

extern "C" {
#include "BaseFileSystemWrapper.h"
#include "FileSystemWindowsShared.h"
#include "FileSystem.h"
}

//#include "ShellContextMenu.h"
#include "ShellMenu.h"


// DeltaWalker custom JNI implementation


JNIEXPORT jlong JNICALL Java_com_deltopia_internal_io_win32_FileSystem_nShellMenuAttach
  (JNIEnv *env, jobject jFileSystem, jlong controlHandle, jlong menuHandle, 
  jobjectArray jPaths, jboolean replaceWinProc, jboolean throwOnInvalidFile) {

   DW_LOG(L">>> SCM create");
   HRESULT hr = S_OK;

   // In theory this method should not be called if InitStatic was failed, but for safety
   if (g_InitRes != LIB_LOAD_SUCCESS) {
      JNU_ThrowIllegalStateException(env, "Not initialized");
      return 0;
   }
   // Check input arguments
   if (NULL == jPaths) {
      JNU_ThrowNullPointerException(env, "paths");
      return 0;
   }
   HWND hWnd;
   // We allow null, then get active window
   if (!controlHandle) {
      DW_LOG(L"Null window handle, active window will be used");
      hWnd = ::GetForegroundWindow();
      if (!hWnd) {
         JNU_ThrowIllegalStateException(env, "Cannot get active window");
         return 0;
      }
   } else {
      hWnd = (HWND)controlHandle;
      if (FALSE == IsWindow(hWnd)) {
         JNU_ThrowIllegalStateException(env, "Invalid control handle");
         return 0;
      }
   }
   // Can be null  too
   HMENU hMenu = (HMENU)menuHandle;

   // Get required paths to native strings, check for nulls too
   // First get array length
   const jsize count = env->GetArrayLength(jPaths);
   if (count <= 0) {
      JNU_ThrowIllegalArgumentException(env, "Empty paths array");
      return 0;
   }
   // Allocate arrays of WCHAR* pointers for all paths
   LPWSTR* nPaths = (LPWSTR*)malloc(sizeof(LPWSTR*)*count);
   if (NULL == nPaths) {
      JNU_ThrowOutOfMemoryError(env, "Cannot allocate native paths array");
      return 0;
   }

   // Now get paths
   int i;
   for (i = 0; i < count; i++) {
      jstring jPath = (jstring)env->GetObjectArrayElement(jPaths, i);
      if (NULL == jPath) {
         DW_LOG(L"SCM get path element failed on %d item", i);
         JNU_ThrowIllegalArgumentException(env, "Null element");
         break;
      }
      const jchar* nPath  = env->GetStringChars(jPath, NULL);
      if (NULL != nPath) {
         jsize length = env->GetStringLength(jPath);
         if (length > 0) {
            WCHAR* nPathCopy = (WCHAR*)malloc(sizeof(WCHAR)*(length+1));
            if (NULL != nPathCopy) {
               hr = StringCchCopyW(nPathCopy, length+1, (WCHAR*)nPath);
               if (SUCCEEDED(hr)) {
                  DW_LOG(L"SCM added path %s", nPathCopy);
                  nPaths[i] = nPathCopy;
               } else {
                  throwExceptionFromError(env, hr, (WCHAR*)nPath);
               }
            } else {
               JNU_ThrowOutOfMemoryError(env, "Cannot allocate native path");
            }
         } else {
            JNU_ThrowIllegalArgumentException(env, "Empty path");
         }
         env->ReleaseStringChars(jPath, nPath);
      } else {
         DW_LOG(L"SCM get string path failed on %d item", i);
         JNU_ThrowIllegalArgumentException(env, "Null path");
      }
      if (env->ExceptionCheck()) {
         break;
      }
   }

   ShellMenu* pSCM = NULL;
   if (!env->ExceptionCheck()) {
      DW_LOG(L"SCM Show with %d items", count);

      pSCM = new ShellMenu(env, hWnd, hMenu);
      if (NULL != pSCM) {
        pSCM->SetObjects(nPaths, count);
        if (!env->ExceptionCheck()) {
            pSCM->AttachContextMenu(JNI_TRUE == replaceWinProc);
        }
      } else {
          JNU_ThrowOutOfMemoryError(env, "Cannot allocate ShellMenu");
      }
   }

   // On exception release object which will release all corresponded resources
   if (env->ExceptionCheck()) {
       if (pSCM) {
           delete pSCM;
           pSCM = NULL;
       }
   }

   //Release input paths
   int j;
   for (j = 0; j < i; j++) {
      free((void*)nPaths[j]);
   }
   free(nPaths);

   DW_LOG(L"<<< SCM create");
   return (jlong)pSCM;
}


JNIEXPORT void JNICALL Java_com_deltopia_internal_io_win32_FileSystem_nShellMenuDestroy
  (JNIEnv *env, jobject jFileSystem, jlong nativeObject) {

   // In theory this method should not be called if InitStatic was failed, but for safety
   if (g_InitRes != LIB_LOAD_SUCCESS) {
      JNU_ThrowIllegalStateException(env, "Not initialized");
      return;
   }
   // Check input arguments
   if (NULL == nativeObject) {
      JNU_ThrowNullPointerException(env, "nativeObject");
      return;
   }
   ShellMenu* pSCM = (ShellMenu*)nativeObject;
   delete pSCM;
}

//
//JNIEXPORT jlong JNICALL Java_com_deltopia_internal_io_win32_FileSystem_nShellMenuHandleMsg
//  (JNIEnv *env, jobject jFileSystem, jlong nativeObject, jint jMsg, jlong jWParam, jlong jLParam) {
//   // In theory this method should not be called if InitStatic was failed, but for safety
//   if (g_InitRes != LIB_LOAD_SUCCESS) {
//      JNU_ThrowIllegalStateException(env, "Not initialized");
//      return 0;
//   }
//   // Check input arguments
//   if (NULL == nativeObject) {
//      JNU_ThrowNullPointerException(env, "nativeObject");
//      return 0;
//   }
//
//   ShellMenu* pSCM = (ShellMenu*)nativeObject;
//   // TODO Milen: How to handle 32/64b?
//   const int wParam = jWParam;
//   const int lParam = jLParam;
//   LRESULT lResult = 0;
//   // TODO Milen: We should return result instead to throw exception
//   if (!pSCM->HandleMessage(jMsg, wParam, lParam, &lResult)) {
//       JNU_ThrowIOException(env, "Message not for shell menu");
//   }
//   return lResult;
//}
