/*******************************************************************************
 * Copyright (c) 2004, 2012 Alternative Ideas Corporation. All rights reserved.
 * This program and the accompanying materials are protected by United
 * States and International copyright laws. They may not be inspected,
 * copied, modified, transmitted, executed, or used in any way without
 * the express written permission of the copyright holder.
 *******************************************************************************/
import java.io.BufferedReader;
import java.io.File;
import java.io.FileDescriptor;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.net.ConnectException;
import java.net.Socket;
import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

public class Launcher {
    public final class LaunchConst {
        public static final String APP_FILE_NAME = ".DeltaWalker"; //$NON-NLS-1$

        public static final String ARG_MERGED = "-merged="; //$NON-NLS-1$

        /**
         * When specified, a new instance of the application will be opened (as
         * opposed to connecting to an already running instance).
         */
        public static final String ARG_MI = "-mi"; //$NON-NLS-1$

        public static final String ARG_WAIT = "-wait"; //$NON-NLS-1$

        /**
         * Allows the caller to specify the time, in milliseconds, the launcher
         * should wait before returning control to the caller. Default value is
         * 1s (1000ms).
         */
        public static final String ARG_WAIT_FOR = "-wait="; //$NON-NLS-1$

        /**
         * A flag to AllowSetForegroundWindow specifying that all processes will
         * be enabled to set the foreground window. This flag rather belongs to
         * the <code>OS</code> class.
         */
        public static final int ASFW_ANY = -1;

        public static final int BIND_TO_PORT_ATTEMPT_COUNT = 10;

        public static final String DONE = "done"; //$NON-NLS-1$

        public static final String EMPTY = ""; //$NON-NLS-1$

        public final static String HANDSHAKE_REQUEST = "Hi, DeltaWalker, how are you doing?"; //$NON-NLS-1$

        public final static String HANDSHAKE_RESPONSE = "Doing well, thank you."; //$NON-NLS-1$

        public static final int INVALID_PORT = -1;

        public static final String LOCAL_HOST = "localhost"; //$NON-NLS-1$

        public static final String OPEN_NEW_APP_WINDOW = "OPEN_NEW_APP_WINDOW"; //$NON-NLS-1$

        /*
         * Windows does it again! Although ';' is universally used as a path
         * separator on Windows, it IS a valid character in a file name!!! Beef
         * up the path separator to decrease the risk of collisions. Be careful
         * to avoid special Regex characters! Keep in mind that while Windows
         * doesn't allow ':' in a file name, ':' follows the drive letter...
         */
        public static final String PATH_SEPARATOR = ":;:"; //$NON-NLS-1$

        public static final int STARTING_PORT = 7777;

        /**
         * Prevents the creation of <code>LaunchConstants</code> instances.
         */
        private LaunchConst() {
        }
    }

    private static final String DW_EXE;

    private static final String DW_PATH;

    private static final String OS_NAME;

    /**
     * The working folder (PWD) including a trailing path separator. <br>
     * NB. Calling System.getProperty("user.dir") inside the launcher (dw.jar)
     * returns what's widely considered the working directory. In contrast,
     * calling it from inside DW's code base always returns DW's install
     * directory.
     */
    private static final String USER_DIR;

    static {
        System.setOut(new PrintStream(new FileOutputStream(FileDescriptor.out)));
        OS_NAME = System.getProperty("os.name"); //$NON-NLS-1$
        DW_PATH = System.getProperty("dw.path"); //$NON-NLS-1$
        USER_DIR = System.getProperty("user.dir") + File.separator; //$NON-NLS-1$

        DW_EXE = "DeltaWalker" + (OS_NAME.contains("Windows") ? ".exe" : ""); //$NON-NLS-1$  //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
    }

    public static boolean absolutelyExists(File file) {
        return (file.isAbsolute() && file.exists());
    }

    /**
     * Tests for existence only if the specified abstract path is absolute. If
     * "path" is not absolute and there is a file named "File" or "Untitled" in
     * the default directory e.g. "C:\Program Files\eclipse", exists() would
     * return true, which is not what one would expect.
     */
    public static boolean absolutelyExists(String path) {
        final File file = new File(path);
        return absolutelyExists(file);
    }

    public static String getSafeString(String string) {
        return (string == null) ? "" : string;
    }

    public static boolean isEmpty(String string) {
        return (string == null) || (string.length() == 0);
    }

    /**
     * Tests whether the protocol/schema of the specified URI is amongst the
     * ones supported by the Apache Commons VFS (defined earlier in this class)
     */
    public static boolean isRemoteUri(final String uriString) {
        if (isEmpty(uriString)) {
            return false;
        }
        final String uri = uriString.toLowerCase().replace('\\', '/');
        /*
         * Use String#contains() instead of String#startsWith() to account for
         * composite schemes like zip:http:// or tar:gz:http://
         */
        return uri.contains("ftp://") || uri.contains("http://") //$NON-NLS-1$  //$NON-NLS-2$
                || uri.contains("http://") || uri.contains("webdav://") //$NON-NLS-1$ //$NON-NLS-2$
                || uri.contains("smb://"); //$NON-NLS-1$
    }

    public static void main(String args[]) {
        Launcher launcher = new Launcher();
        launcher.launch(args);
    }

    private int foundPort = LaunchConst.INVALID_PORT;

    ObjectInputStream in;

    String message;

    ObjectOutputStream out;

    /** The pathname of the file used for port # exchange. */
    private File portFile;

    private boolean wait = false;

    /**
     * The time, in milliseconds, the launcher should wait before returning
     * control to the caller. Default value is 1000ms.
     */
    private long waitFor = 1000;

    public Launcher() {
    }

    private String[] argumentsAsArray(List<String> args) {
        boolean append = !DW_PATH.endsWith(File.separator);
        String pathName = DW_PATH + (append ? File.separator : "") + DW_EXE; //$NON-NLS-1$

        List<String> cmdArgs = new ArrayList<String>();
        cmdArgs.add(pathName);

        for (String arg : args) {
            /*
             * If launched via a shell/batch file there are empty strings that
             * need to be filtered out, else they'll confuse DW.
             */
            if (!isEmpty(arg)) {
                cmdArgs.add(arg);
            }
        }

        return cmdArgs.toArray(new String[cmdArgs.size()]);
    }

    /**
     * Looks for a port the DW server socket is listening to. Scanning ports
     * that aren't available is time-consuming. Use a file to exchange the port
     * number b/n running DW instances. Each instance must refresh the file to
     * ensure it's not an orphan left behind a crashed app.
     */
    private int findServerPort() {
        if (foundPort != LaunchConst.INVALID_PORT) {
            return foundPort;
        }

        /*
         * Scanning ports that aren't available is time-consuming. Use a file to
         * look for a port we could talk to, starting at
         * <code>STARTING_PORT</code>.
         */
        File portFile = getPortFile();

        /*
         * Don't rely on an IO exception to learn whether a port file exists, as
         * for a very first app instance this file is not created yet.
         */
        if (!absolutelyExists(portFile)) {
            return LaunchConst.INVALID_PORT;
        }

        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new FileReader(portFile));

            String inputLine = getSafeString(reader.readLine()).trim();
            if (inputLine.length() > 0) {
                int candidatePort = Integer.parseInt(inputLine);
                if (isServerAlive(candidatePort)) {
                    return (foundPort = candidatePort);
                }
            }
        } catch (IOException x) {
        } catch (NumberFormatException x) {
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException x) {
                    /* Ignore the exception */
                }
            }
        }

        /*
         * The port file doesn't contain the port of an alive DeltaWalker
         * server.
         */
        portFile.delete();

        return LaunchConst.INVALID_PORT;
    }

    /** Gets the file used for port exchange. */
    public File getPortFile() {
        if (portFile == null) {
            String home = System.getProperty("user.home");
            home += "/.deltawalker";

            portFile = new File(home, LaunchConst.APP_FILE_NAME);
        }

        return portFile;
    }

    /**
     * Returns the resource args, optionally including the RESULT URI arg
     * 
     * @param args
     * @param incMergedArg
     *            Include the ARG_MERGED argument in the list of returned
     *            resource args.
     * @return
     */
    private List<String> getResourceArgs(String[] args, boolean incMergedArg) {
        List<String> resourceArgs = new ArrayList<String>();
        for (String arg : args) {
            arg = arg.trim();
            if (arg.contains(LaunchConst.ARG_WAIT_FOR)) {
                String value = arg.substring(LaunchConst.ARG_WAIT_FOR.length(),
                        arg.length()).trim();
                waitFor = Long.parseLong(value);
                continue;
            }
            if (arg.equalsIgnoreCase(LaunchConst.ARG_MI)) {
                continue;
            }
            if (arg.equalsIgnoreCase(LaunchConst.ARG_WAIT)) {
                wait = true;
                continue;
            }
            if (!incMergedArg && arg.contains(LaunchConst.ARG_MERGED)) {
                continue;
            }

            /*
             * Many SCMs provide only the resource file/folder names when
             * launching an external diff tool, assuming the diff tool would
             * take care of prepending the working folder when necessary. The
             * following few lines do just that.
             */
            if (!isEmpty(arg) && !isRemoteUri(arg) && !absolutelyExists(arg)) {
                String path = USER_DIR + arg;
                /*
                 * Update the URI only if a missing working folder is the real
                 * culprit.
                 */
                if (absolutelyExists(path)) {
                    arg = path;
                }
            }

            resourceArgs.add(arg);
        }

        return resourceArgs;
    }

    public boolean isDeltaWalkerRunning() {
        return (findServerPort() != LaunchConst.INVALID_PORT);
    }

    /**
     * Checks whether the server on the specified port is alive and of the type
     * we know what to do with, i.e. DeltaWalker
     */
    private boolean isServerAlive(int port) {
        try {
            Socket client = new Socket(LaunchConst.LOCAL_HOST, port);

            PrintWriter writer = new PrintWriter(client.getOutputStream(), true);
            BufferedReader reader = new BufferedReader(new InputStreamReader(
                    client.getInputStream()));

            writer.println(LaunchConst.HANDSHAKE_REQUEST);

            /* Wait to hear back from server. */
            String inputLine = getSafeString(reader.readLine());
            writer.close();
            reader.close();
            client.close();

            return inputLine.equals(LaunchConst.HANDSHAKE_RESPONSE);
        } catch (ConnectException ex) {
            /* The ConnectException seems to be benign and safe to ignore. */
        } catch (IOException ex) {
        }

        return false;
    }

    public void launch(String args[]) {
        if ((args == null) || (args.length == 0)) {
            System.out
                    .println("Launching DeltaWalker with no files and/or folders for comparison.");
        }
        if (isEmpty(DW_PATH)) {
            System.out
                    .println("Please set the dw.path system property like so: \n"
                            + "java -Ddw.path=\"full_path_to_DeltaWalker_executable\" -jar dw.jar");
            return;
        }
        /*
         * Determine whether a new application instance should be started (e.g.
         * when the application has been started with a specific command-line
         * argument that forces a new instance). Otherwise - i.e. explicit
         * creation of a new application instance is not requested and an
         * existing application instance is found - reuse that instance.
         */
        boolean launchNew = false;
        for (String arg : args) {
            if (arg.equals(LaunchConst.ARG_MI)) {
                launchNew = true;
                break;
            }
        }
        /*
         * TODO: we'll need to pass along all command line args not just the
         * resources.
         */
        List<String> resourceArgs = getResourceArgs(args, true);
        if (launchNew || !isDeltaWalkerRunning()) {
            launchDeltaWalker(resourceArgs);
        } else {
            /*
             * When valid resources are passed as command line arguments and
             * there already is a running app instance they will be opened for
             * comparison in that instance (as opposed to in a new app instance)
             * regardless of the value of <code>singleton</code>. Pass along
             * both the resource args as well as any DeltaWalker-specific args.
             */
            sendLaunchArguments(resourceArgs);
        }
    }

    private void launchDeltaWalker(List<String> args) {
        try {
            String[] argArray = argumentsAsArray(args);
            Process dwProcess = Runtime.getRuntime().exec(argArray);

            /*
             * The class Process provides methods for performing input from the
             * process (via getInputStream()), performing output to the process
             * (via getInputStream()), waiting for the process to complete,
             * checking the exit status of the process, and destroying (killing)
             * the process.
             */
            if (wait) {
                BufferedReader stdInput = new BufferedReader(
                        new InputStreamReader(dwProcess.getInputStream()));

                BufferedReader stdError = new BufferedReader(
                        new InputStreamReader(dwProcess.getErrorStream()));

                /* read the output from the command */
                String str = null;
                while ((str = stdInput.readLine()) != null) {
                    System.out.println(str);
                }

                /* read any errors from the attempted command */
                while ((str = stdError.readLine()) != null) {
                    System.out.println(str);
                }
                System.exit(0);
            } else {
                /*
                 * Wait 1s then start checking whether DW is up and running and
                 * once it is, give it another 0.5s to finish
                 * initialization/open the files, then return/exit.
                 */
                int interval = 501; // repeat 4x every 0.5s
                final int[] counter = new int[] { 0 };
                Timer timer = new Timer();
                timer.scheduleAtFixedRate(new TimerTask() {
                    public void run() {
                        /* If more than 5sec. have elapsed, exit anyway. */
                        if ((isDeltaWalkerRunning() && (counter[0] >= 4))
                                || (counter[0] >= 10)) {
                            System.exit(0);
                        }
                        counter[0]++;
                    }
                }, waitFor, interval);
            }
        } catch (IOException ex) {
            System.out.println("Couldn't launch DeltaWalker: " + ex);
            ex.printStackTrace();

            System.exit(-1);
        }
    }

    private void sendLaunchArguments(List<String> args) {
        int port = findServerPort();
        if (port == LaunchConst.INVALID_PORT) {
            return;
        }

        try {
            final Socket client = new Socket(LaunchConst.LOCAL_HOST, port);
            final PrintWriter writer = new PrintWriter(
                    client.getOutputStream(), true);
            final BufferedReader reader = new BufferedReader(
                    new InputStreamReader(client.getInputStream()));

            if (args.isEmpty()) {
                writer.println(LaunchConst.OPEN_NEW_APP_WINDOW);
            } else {
                StringBuffer argString = new StringBuffer();
                for (String arg : args) {
                    /*
                     * If launched via a shell/batch file there are empty
                     * strings that need to be filtered out, else they'll
                     * confuse DW.
                     */
                    if (!isEmpty(arg)) {
                        argString.append(arg);
                        argString.append(LaunchConst.PATH_SEPARATOR);
                    }
                }
                if (wait) {
                    argString.append(LaunchConst.ARG_WAIT);
                }

                writer.println(argString);
            }

            /* Wait to hear back from server. */
            String line;
            while ((line = reader.readLine()) != null) {
                /*
                 * We should wait for a done confirmation specific to the
                 * resources we sent.
                 */
                if (line.equals(LaunchConst.DONE)) {
                    break;
                }
            }
            writer.close();
            reader.close();
            client.close();
        } catch (IOException ex) {
            System.out.println(ex.getMessage());
        }
    }
}
