/*
 * Decompiled with CFR 0.152.
 */
package com.prosc.aws;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.prosc.aws.AwsRequest;
import com.prosc.aws.ResponseStyle;
import com.prosc.aws.exception.AWSJsonException;
import com.prosc.aws.exception.AWSRuntimeException;
import com.prosc.aws.exception.AWSThrottlingException;
import com.prosc.aws.exception.AccessDeniedException;
import com.prosc.aws.exception.AwsInternalException;
import com.prosc.aws.exception.InvalidParameterValueException;
import com.prosc.aws.exception.OperationAbortedException;
import com.prosc.aws.exception.PermanentRedirectException;
import com.prosc.aws.exception.RedirectException;
import com.prosc.aws.exception.ServerClockMismatchException;
import com.prosc.aws.exception.ServiceUnavailableException;
import com.prosc.aws.exception.SignatureDoesNotMatchException;
import com.prosc.io.IOUtils;
import com.prosc.xml.DomUtils;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class AwsResult {
    private static final Logger log = Logger.getLogger(AwsResult.class.getName());
    private static final DocumentBuilderFactory DOCUMENT_BUILDER_FACTORY = DocumentBuilderFactory.newInstance();
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private final HttpURLConnection urlConnection;
    private final ResponseStyle responseStyle;
    private final ResponseStyle errorStyle;
    private final AwsRequest<?> request;
    private InputStream responseStream;
    private long responseLength;
    private Document xmlDocument;
    private String responseAsString;
    private int httpStatusCode = Integer.MAX_VALUE;
    private String errorCode;
    private boolean didValidate = false;

    public AwsResult(HttpURLConnection connection, ResponseStyle responseStyle, AwsRequest<?> request) {
        this(connection, responseStyle, responseStyle == ResponseStyle.none ? ResponseStyle.raw : responseStyle, request);
    }

    public AwsResult(HttpURLConnection connection, ResponseStyle responseStyle, ResponseStyle errorStyle, AwsRequest<?> request) {
        this.urlConnection = connection;
        this.responseStyle = responseStyle;
        this.errorStyle = errorStyle;
        this.request = request;
    }

    public AwsResult downloadInterruptibly() throws InterruptedException, IOException {
        IOUtils.openConnectionInterruptibly(this.urlConnection);
        return this.download();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public AwsResult download() throws IOException {
        if (this.httpStatusCode == Integer.MAX_VALUE) {
            if (AwsRequest.isDebugMode()) {
                log.info("AWS " + this.urlConnection.getRequestMethod() + " request to " + this.urlConnection.getURL().toString());
            }
            this.httpStatusCode = this.urlConnection.getResponseCode();
            if (this.httpStatusCode == -1) {
                log.warning("Received a -1 httpStatusCode; this indicates that something went very wrong with the request.");
            }
            try {
                this.responseStream = this.urlConnection.getInputStream();
                if (this.httpStatusCode >= 200 && this.httpStatusCode < 300 || this.httpStatusCode == 304) {
                    this.responseLength = this.urlConnection.getContentLengthLong();
                    if ("chunked".equals(this.urlConnection.getHeaderField("transfer-encoding"))) {
                        byte[] responseBytes = IOUtils.inputStreamAsBytes(this.responseStream);
                        this.responseStream = new ByteArrayInputStream(responseBytes);
                        this.responseLength = responseBytes.length;
                    }
                    this.errorCode = null;
                    if (this.responseStyle == ResponseStyle.xml) {
                        if (this.responseLength > 0L) {
                            try {
                                this.xmlDocument = AwsResult.parseXmlResponse(this.responseStream);
                            }
                            catch (SAXException e) {
                                throw new RuntimeException("Invalid XML response received from AWS", e);
                            }
                            finally {
                                this.responseStream.close();
                                this.responseStream = null;
                            }
                        } else {
                            log.warning("The request " + this.request + " specified an XML response style, but the actual response was empty. Change the response style to None.");
                        }
                        this.responseAsString = null;
                        return this;
                    }
                    if (this.responseStyle == ResponseStyle.json) {
                        if (this.responseLength > 0L) {
                            this.xmlDocument = null;
                            try {
                                this.responseAsString = IOUtils.inputStreamAsString(this.responseStream);
                                return this;
                            }
                            finally {
                                this.responseStream.close();
                                this.responseStream = null;
                            }
                        }
                        log.warning("The request " + this.request + " specified a JSON response style, but the actual response was empty. Change the response style to None.");
                        return this;
                    }
                    if (this.responseStyle != ResponseStyle.raw) return this;
                    this.xmlDocument = null;
                    this.responseAsString = null;
                    return this;
                }
                this.throwError();
                return this;
            }
            catch (IOException e) {
                try {
                    this.responseStream = this.urlConnection.getErrorStream();
                }
                catch (Exception e1) {
                    this.responseStream = null;
                }
                this.throwError();
                return this;
            }
            finally {
                if (AwsRequest.isDebugMode()) {
                    if (this.responseStream == null && this.responseAsString == null && this.xmlDocument == null) {
                        log.info("No response received");
                    } else if (this.responseAsString != null) {
                        log.info("Response: " + this.responseAsString);
                    } else if (this.xmlDocument != null) {
                        log.info("Response XML: " + DomUtils.dumpXml(this.xmlDocument, null));
                    } else if (this.urlConnection.getContentLengthLong() > 0L) {
                        log.info("Received a raw stream with content-length: " + this.urlConnection.getContentLengthLong());
                    }
                }
            }
        }
        log.finest("Already downloaded");
        return this;
    }

    private void throwError() throws IOException, AWSRuntimeException {
        this.responseLength = this.urlConnection.getContentLengthLong();
        URL failedUrl = this.urlConnection.getURL();
        if (this.responseStream == null) {
            this.xmlDocument = null;
            this.errorCode = String.valueOf(this.httpStatusCode);
            this.responseAsString = null;
        } else if (this.errorStyle == ResponseStyle.xml) {
            byte[] bytes = IOUtils.inputStreamAsBytes(this.responseStream);
            this.responseAsString = new String(bytes, StandardCharsets.UTF_8);
            this.responseStream = new ByteArrayInputStream(bytes);
            try {
                this.xmlDocument = AwsResult.parseXmlResponse(this.responseStream);
                this.errorCode = DomUtils.getRecursiveChildText(this.xmlDocument.getDocumentElement(), "Code");
            }
            catch (SAXException e1) {
                if (bytes.length == 0) {
                    throw new IOException("This URL request failed:" + failedUrl + " with a " + this.httpStatusCode + " response code. In addition, An XML error was was expected but it was empty.");
                }
                throw new IOException("This URL request failed:" + failedUrl + " with a " + this.httpStatusCode + " response code. A SAX parsing exception occurred while trying to parse this error message: " + new String(bytes, StandardCharsets.UTF_8));
            }
            this.responseStream.close();
            this.responseStream = null;
        } else if (this.errorStyle == ResponseStyle.json || this.errorStyle == ResponseStyle.raw) {
            this.xmlDocument = null;
            this.errorCode = this.responseAsString = IOUtils.inputStreamAsString(this.responseStream);
            this.responseStream.close();
            this.responseStream = null;
            try {
                JsonNode parsedResponse = objectMapper.readTree(this.responseAsString);
                this.errorCode = parsedResponse.at("/message").asText(this.errorCode);
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        AWSThrottlingException.check(this, this.request);
        ServiceUnavailableException.check(this, this.request);
        OperationAbortedException.check(this, this.request);
        AwsInternalException.check(this, this.request);
        if (AwsRequest.isDebugMode() || this.httpStatusCode == -1) {
            log.warning("Failed URL: " + failedUrl + "\nRequest: " + this.request.toString());
        }
    }

    public static Document parseXmlResponse(InputStream response) throws IOException, SAXException {
        try {
            if (response.markSupported()) {
                response.mark(4096);
            }
            Document document = AwsResult.parseXmlResponse(new InputStreamReader(response, StandardCharsets.UTF_8));
            return document;
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        catch (SAXException e) {
            try {
                response.reset();
            }
            catch (IOException e1) {
                log.severe("Could not reset InputStream from Amazon; XML has been read and is not available");
            }
            throw e;
        }
        finally {
            response.close();
        }
    }

    public static Document parseXmlResponse(Reader response) throws SAXException, IOException {
        try {
            DocumentBuilder documentBuilder = DOCUMENT_BUILDER_FACTORY.newDocumentBuilder();
            Document document = documentBuilder.parse(new InputSource(response));
            return document;
        }
        catch (IOException e) {
            String msg = "IOException while parsing XML error response from Amazon";
            log.log(Level.WARNING, "IOException while parsing XML error response from Amazon");
            throw new IOException("IOException while parsing XML error response from Amazon", e);
        }
        catch (ParserConfigurationException e) {
            throw new RuntimeException(e);
        }
        finally {
            response.close();
        }
    }

    public AwsResult validate() {
        if (!this.didValidate) {
            AccessDeniedException.check(this, this.request);
            SignatureDoesNotMatchException.check(this, this.request);
            InvalidParameterValueException.check(this, this.request);
            RedirectException.check(this, this.request);
            ServerClockMismatchException.check(this, this.request);
            PermanentRedirectException.check(this, this.request);
            if (this.httpStatusCode != 304 && (this.httpStatusCode < 200 || this.httpStatusCode >= 300)) {
                if (this.errorStyle == ResponseStyle.xml && this.xmlDocument != null) {
                    throw new AWSRuntimeException(this.httpStatusCode, this.xmlDocument, this.request);
                }
                if (this.errorStyle == ResponseStyle.json) {
                    throw new AWSJsonException(this.httpStatusCode, this.responseAsString, (AwsRequest)this.request);
                }
                throw new AWSRuntimeException(this.httpStatusCode, this.errorCode, this.responseAsString, this.request);
            }
        }
        this.didValidate = true;
        return this;
    }

    public InputStream getResponseStream(boolean validate) {
        String stringValue;
        if (validate && !this.didValidate) {
            log.log(Level.WARNING, "Call to getResponseStream() without first calling validate()");
        }
        if (this.responseStream == null && (stringValue = this.getResponseAsString()) != null) {
            this.responseStream = new ByteArrayInputStream(stringValue.getBytes(StandardCharsets.UTF_8));
        }
        return this.responseStream;
    }

    public InputStream getResponseStream() {
        return this.getResponseStream(true);
    }

    public Document getDocument() {
        String stringValue;
        if (this.xmlDocument == null && "application/xml".equals(this.urlConnection.getHeaderField("content-type")) && (stringValue = this.getResponseAsString()) != null) {
            try {
                this.xmlDocument = AwsResult.parseXmlResponse(new StringReader(stringValue));
            }
            catch (Exception e) {
                log.log(Level.WARNING, "Response does not contain valid XML. Raw response: " + stringValue);
            }
        }
        return this.xmlDocument;
    }

    public String getResponseAsString() {
        if (this.responseAsString == null) {
            if (this.responseStream != null) {
                try {
                    this.responseAsString = IOUtils.inputStreamAsString(this.responseStream);
                }
                catch (IOException e) {
                    log.log(Level.WARNING, "Could not read any data from response stream", e);
                }
            } else if (this.xmlDocument != null) {
                this.responseAsString = DomUtils.dumpXml(this.xmlDocument, this.xmlDocument.getDocumentElement());
            }
        }
        return this.responseAsString;
    }

    public HttpURLConnection getUrlConnection() {
        return this.urlConnection;
    }

    public int getHttpStatusCode() {
        return this.httpStatusCode;
    }

    public String getErrorCode() {
        return this.errorCode;
    }

    public String toString() {
        if (this.xmlDocument == null && this.responseAsString == null && this.responseStream == null) {
            return null;
        }
        if (this.responseAsString != null) {
            return this.responseAsString;
        }
        if (this.xmlDocument != null) {
            return this.getResponseAsString();
        }
        return "Raw stream with content-length " + this.responseLength;
    }

    public ResponseStyle getResponseStyle() {
        return this.responseStyle;
    }

    public ResponseStyle getErrorStyle() {
        return this.errorStyle;
    }
}

