<?php
require_once('Game.php');
require_once('Player.php');
require_once('PlayerToTeam.php');
require_once('Result.php');
require_once('Saison.php');
require_once('Team.php');

class DatabaseOperations
{
	const GameTable = "spielplan";
	private static $GFields = array("id", "our_team","opponent", "date", "home");
	private static $GFieldsType = array(
		"id" => "I4 PRIMARY AUTOINCREMENT NOTNULL",
		"our_team" => "I2 NOTNULL",
		"opponent" => "C(50) NOTNULL",
		"date" => "datetime NOTNULL",
		"home" => "I1 NOTNULL",
	);
	const PlayerTable = "spieler";
	private static $PFields = array("id", "name");
	private static $PFieldsType = array(
		"id" => "I2 PRIMARY AUTOINCREMENT NOTNULL",
		"name" => "C(50) NOTNULL"
	);
	const PlayerToTeamTable = "spieler_zu_mannschaft";
	private static $PTFields = array("id", "spieler", "saison", "mannschaft");
	private static $PTFieldsType = array(
		"id" => "I4 PRIMARY AUTOINCREMENT NOTNULL", 
		"spieler" => "I2 NOTNULL", 
		"saison" => "I2 NOTNULL", 
		"mannschaft" => "I2 NOTNULL");
	const ResultTable = "einzelergebnisse";
	private static $RFields = array("id", "spieler", "spielplan", "p_ges", "p_fehl", "p_abr", "p_count", "opponent", "o_ges", "o_abr", "o_fehl", "o_count");
	private static $RFieldsType = array(
		"id" => "I4 PRIMARY AUTOINCREMENT NOTNULL", 
		"spieler" => "I2 NOTNULL",
		"spielplan" => "I4 NOTNULL",
		"p_ges" => "I2 NOTNULL",
		"p_fehl" => "I2 NOTNULL",
		"p_abr" => "I2 NOTNULL",
		"p_count" => "I1 NOTNULL",
		"opponent" => "C(50) NOTNULL",
		"o_ges" => "I2 NOTNULL",
		"o_abr" => "I2 NOTNULL",
		"o_fehl" => "I2 NOTNULL",
		"o_count" => "I1 NOTNULL"
	);
	const SaisonTable = "saison";
	private static $SFields = array("id", "name", "begin", "end");
	private static $SFieldsType = array(
		"id" => "I2 PRIMARY AUTOINCREMENT NOTNULL",
		"name" => "C(50) NOTNULL",
		"begin" => "datetime NOTNULL",
		"end" => "datetime NOTNULL"
	);
	const TeamTable = "mannschaft";
	private static $TFields = array("id", "name");
	private static $TFieldsType = array(
		"id" => "I2 PRIMARY AUTOINCREMENT NOTNULL",
		"name" => "C(50) NOTNULL"
	);
	
	public static function update_m0m0p2_m0m0p3()
	{
		$gCms = $GLOBALS['gCms'];
		if (gCms) {
			$db = $gCms->GetDb();
			$db->Execute("RENAME TABLE ".self::ResultTable." TO ".cms_db_prefix().self::ResultTable);
			$db->Execute("RENAME TABLE ".self::GameTable." TO ".cms_db_prefix().self::GameTable);
			$db->Execute("RENAME TABLE ".self::PlayerToTeamTable." TO ".cms_db_prefix().self::PlayerToTeamTable);
			$db->Execute("RENAME TABLE ".self::TeamTable." TO ".cms_db_prefix().self::TeamTable);
			$db->Execute("RENAME TABLE ".self::SaisonTable." TO ".cms_db_prefix().self::SaisonTable);
			$db->Execute("RENAME TABLE ".self::PlayerTable." TO ".cms_db_prefix().self::PlayerTable);
		}
	} 
	
	public static function createDatabase()
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms) {
			$db = $gCms->GetDb();
			
			/* configuration zum anlegen erzeugen */
			$configs = array(
				cms_db_prefix().self::PlayerTable => array("fields" => self::$PFields, 'descs' => self::$PFieldsType),
				cms_db_prefix().self::SaisonTable => array("fields" => self::$SFields, 'descs' => self::$SFieldsType),
				cms_db_prefix().self::TeamTable => array("fields" => self::$TFields, 'descs' => self::$TFieldsType),
				cms_db_prefix().self::PlayerToTeamTable => array("fields" => self::$PTFields, 'descs' => self::$PTFieldsType),
				cms_db_prefix().self::GameTable => array("fields" => self::$GFields, 'descs' => self::$GFieldsType),
				cms_db_prefix().self::ResultTable => array("fields" => self::$RFields, 'descs' => self::$RFieldsType),
			);
			
			/* fremdschlüsselinformationen einbringen */
			$configs[cms_db_prefix().self::GameTable]["mysql_constraints"] = ', FOREIGN KEY('.self::$GFields[1].') REFERENCES '.cms_db_prefix().self::TeamTable.'('.self::$TFields[0].') ON DELETE CASCADE ON UPDATE CASCADE';
			$configs[cms_db_prefix().self::PlayerToTeamTable]["mysql_constraints"] = 
				', FOREIGN KEY('.self::$PTFields[1].') REFERENCES '.cms_db_prefix().self::PlayerTable.'('.self::$PFields[0].') ON DELETE CASCADE ON UPDATE CASCADE'.
				', FOREIGN KEY('.self::$PTFields[2].') REFERENCES '.cms_db_prefix().self::SaisonTable.'('.self::$SFields[0].') ON DELETE CASCADE ON UPDATE CASCADE'.
				', FOREIGN KEY('.self::$PTFields[3].') REFERENCES '.cms_db_prefix().self::TeamTable.'('.self::$TFields[0].') ON DELETE CASCADE ON UPDATE CASCADE';
			$configs[cms_db_prefix().self::ResultTable]["mysql_constraints"] = 
				', FOREIGN KEY('.self::$RFields[1].') REFERENCES '.cms_db_prefix().self::PlayerTable.'('.self::$PFields[0].') ON DELETE CASCADE ON UPDATE CASCADE'.
				', FOREIGN KEY('.self::$RFields[2].') REFERENCES '.cms_db_prefix().self::GameTable.'('.self::$GFields[0].') ON DELETE CASCADE ON UPDATE CASCADE';
				
			/* tabellen anlegen */
			foreach ($configs as $config => $dummy) {
				echo $config;
				$tab = NewDataDictionary($db);
				$tabopt = array('mysql' => 'ENGINE=InnoDB DEFAULT CHARSET=latin1');
				$tabfields = "";
				foreach($configs[$config]['fields'] as $field) {
					$tabfields .= $field." ".$configs[$config]['descs'][$field].",";
				}
				if (isset($configs[$config]["mysql_constraints"])) {
					$tabopt['constraints'] = $configs[$config]["mysql_constraints"];
				}
				$tabfields = substr($tabfields,0,strlen($tabfields)-1);
				$sqlarray = $tab->CreateTableSQL($config,$tabfields,$tabopt);
				$tab->ExecuteSQLArray($sqlarray);
			}
		}
	}
	
	public static function deleteDatabase()
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms) {
			$db = $gCms->GetDb();
		}
		
		/* define tables to drop */
		$configs = array(cms_db_prefix().self::PlayerToTeamTable, cms_db_prefix().self::ResultTable, cms_db_prefix().self::GameTable, cms_db_prefix().self::SaisonTable, cms_db_prefix().self::PlayerTable, cms_db_prefix().self::TeamTable);
		
		foreach ($configs as $config) {
			$dict = NewDataDictionary( $db );
			$sqlarray = $dict->DropTableSQL( $config );
			$dict->ExecuteSQLArray($sqlarray);
		}
	}
	
	/**
	* Diese Methode bearbeitet einen String so, dass er nicht für SQL Injections
	* verwendet werden kann.
	*
	* @param string $string
	* @param string $dbtype
	* @return string Der String der enstprechend der Datenbanksyntax escaped wurde.
	*/
	private static function escapeSQLString($string, $dbtype = "mysql")
	{
		if ($dbtype == "mysql") {
			return mysql_escape_string($string);
		}

		return "";
	}
	
	/**
	 * Diese Methode liefert den Spieler zurück, der in der Datenbank die Nummer id besitzt.
	 *
	 * @param id Die Nummer des Spielers
	 * @return Player Der gewünschte Spieler oder NULL wenn der Spieler nicht existiert
	 */
	public static function getPlayer($id)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && is_numeric($id)) {

			$db = $gCms->GetDb();
			$result = $db->Execute("SELECT * FROM ".cms_db_prefix().self::PlayerTable." where id=".$id);
			if ($result) {
				if ($result->RecordCount() == 1) {
					$row = $result->FetchRow();
					return new Player($row[self::$PFields[0]], $row[self::$PFields[1]]);
				}
			}
		}
		
		return NULL;
	}
	
	/**
	 * Diese Methode liefert alle in der Datenbankgespeicherten Benutzer zurück.
	 *
	 * @return Ein Array aller in der Datenbank vorhandenen Spieler
	 */
	public static function getAllPlayers()
	{
		$gCms = $GLOBALS['gCms'];
		if (isset($gCms)) {
			$db = $gCms->GetDb();
			$result = $db->Execute("SELECT * FROM ".cms_db_prefix().self::PlayerTable." ORDER BY ".self::$PFields[1]);
			if ($result) {
				if ($result->RecordCount() >= 1) {
					$players = array();
					for ($i = 0; $i < $result->RecordCount(); $i = $i + 1) {
						$row = $result->FetchRow();
						$players[$i] = new Player($row[self::$PFields[0]], $row[self::$PFields[1]]);
					}
					return $players;
				}
			}
		}
		
		return NULL;
	}
	
	/**
	 * Diese Methode legt einen neuen Spieler mit vorgegebenem Namen an.
	 *
	 * @param string name Der Name des zu erstellenden Spielers
	 */
	public static function createPlayer($name)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && is_string($name)) {
			$name = self::escapeSQLString($name);
			$db = $gCms->GetDb();
			$result = $db->Execute("INSERT INTO ".cms_db_prefix().self::PlayerTable." (".self::$PFields[1].") VALUES ('".$name."')");
		}
	}
	
	/**
	 * Diese Methode ändert einen Spieler. Geändert werden die Daten nur,
	 * wenn der Datensatz des alten Spielers mit dem Datensatz in der
	 * Datenbank übereinstimmt und wenn der alte Spieler und der neue
	 * Spieler die gleiche id besitzen (vergleich über Player->getId()).
	 *
	 * @param Player oldplayer Die alte version des Spielers
	 * @param Player newplayer Die neue version des Spielers
	 * @return bool true, wenn der spieler geändert wurde, false sonst
	 */
	public static function changePlayer($oldplayer, $newplayer)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && ($oldplayer instanceof Player) && ($newplayer instanceof Player)) {
			$savedplayer = self::getPlayer($oldplayer->getId());
			if (($savedplayer != NULL) && ($savedplayer == $oldplayer) && ($oldplayer->getId() == $newplayer->getId())) {
				$db = $gCms->GetDb();
				$result = $db->Execute(
					"UPDATE ".cms_db_prefix().self::PlayerTable.
					" SET ".self::$PFields[1]."='".self::escapeSQLString($newplayer->getName()).
					"' WHERE ".self::$PFields[0]."=".$oldplayer->getId());
				if ($result) {
					return true;
				}
			}
		}
		
		return false;
	}
	
	/**
	 * Diese Methode liefert das Team zurück, das in der Datenbank die Nummer id besitzt.
	 *
	 * @param id Die Nummer des Teams
	 * @return Team Das gewünschte Team oder NULL wenn das Team nicht existiert
	 */
	public static function getTeam($id)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && is_numeric($id)) {

			$db = $gCms->GetDb();
			$result = $db->Execute("SELECT * FROM ".cms_db_prefix().self::TeamTable." where id=".$id);
			if ($result) {
				if ($result->RecordCount() == 1) {
					$row = $result->FetchRow();
					return new Team($row[self::$TFields[0]], $row[self::$TFields[1]]);
				}
			}
		}
		
		return NULL;
	}
	
	/**
	 * Diese Methode liefert alle in der Datenbank gespeicherten Teams zurück.
	 *
	 * @return Ein Array aller in der Datenbank vorhandenen Teams
	 */
	public static function getAllTeams()
	{
		$gCms = $GLOBALS['gCms'];
		if (isset($gCms)) {
			$db = $gCms->GetDb();
			$result = $db->Execute("SELECT * FROM ".cms_db_prefix().self::TeamTable." ORDER BY ".self::$TFields[1]);
			if ($result) {
				if ($result->RecordCount() >= 1) {
					$teams = array();
					for ($i = 0; $i < $result->RecordCount(); $i = $i + 1) {
						$row = $result->FetchRow();
						$teams[$i] = new Team($row[self::$TFields[0]], $row[self::$TFields[1]]);
					}
					return $teams;
				}
			}
		}
		
		return NULL;
	}
	
	/**
	 * Diese Methode legt eine neues Team mit vorgegebenem Namen an.
	 *
	 * @param string name Der Name des zu erstellenden Teams
	 */
	public static function createTeam($name)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && is_string($name)) {
			$name = self::escapeSQLString($name);
			$db = $gCms->GetDb();
			$result = $db->Execute("INSERT INTO ".cms_db_prefix().self::TeamTable." (".self::$TFields[1].") VALUES ('".$name."')");
		}
	}
	
	/**
	 * Diese Methode ändert ein Team. Geändert werden die Daten nur,
	 * wenn der Datensatz des alten Teams mit dem Datensatz in der
	 * Datenbank übereinstimmt und wenn das alte Team und das neue
	 * Team die gleiche id besitzen (vergleich über Team->getId()).
	 *
	 * @param Team oldteam Die alte version des Teams
	 * @param Team newteam Die neue version des Teams
	 * @return bool true, wenn das Team geändert wurde, false sonst
	 */
	public static function changeTeam($oldteam, $newteam)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && ($oldteam instanceof Team) && ($newteam instanceof Team)) {
			$savedteam = self::getTeam($oldteam->getId());
			if (($savedteam != NULL) && ($savedteam == $oldteam) && ($oldteam->getId() == $newteam->getId())) {
				$db = $gCms->GetDb();
				$result = $db->Execute(
					"UPDATE ".cms_db_prefix().self::TeamTable.
					" SET ".self::$TFields[1]."='".self::escapeSQLString($newteam->getName()).
					"' WHERE ".self::$TFields[0]."=".$oldteam->getId());
				if ($result) {
					return true;
				}
			}
		}
		
		return false;
	}
	
	/**
	 * Diese Methode liefert die Saison zurück, die in der Datenbank die Nummer id besitzt.
	 *
	 * @param int id Die id der gewünschten Saison
	 * @return Saison Die gewünschte Saison
	 * @throws SQLException Wenn was schief geht ;-)
	 */
	public static function getSaison($id) 
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && is_numeric($id)) {

			$db = $gCms->GetDb();
			$result = $db->Execute("SELECT * FROM ".cms_db_prefix().self::SaisonTable." where id=".$id);
			if ($result) {
				if ($result->RecordCount() == 1) {
					$row = $result->FetchRow();
					return new Saison($row[self::$SFields[0]], $row[self::$SFields[1]], new DateTime($row[self::$SFields[2]]), new DateTime($row[self::$SFields[3]]));
				}
			}
		}

		return NULL;
	}
	
	/**
	 * Diese Methode liefert alle Saison zurück, die in der Datenbank vorhanden sind.
	 *
	 * @return Ein array, das alle in der Datenbank vorhandenen Saisons enthält
	 * @throws SQLException Wenn was schief geht ;-)
	 */
	public static function getAllSaisons()
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms) {
			$saisons = array();
			$db = $gCms->GetDb();
			$result = $db->Execute("SELECT * FROM ".cms_db_prefix().self::SaisonTable." ORDER BY ".self::$SFields[1]);
			if ($result) {
				for ($i = 0; $i < $result->RecordCount(); $i = $i + 1) {
					$row = $result->FetchRow();
					array_push($saisons, new Saison($row[self::$SFields[0]], $row[self::$SFields[1]], new DateTime($row[self::$SFields[2]]), new DateTime($row[self::$SFields[3]])));
				}
				return $saisons;
			}
		}

		return NULL;
	}
	
	/**
	 * Diese Methode erzeugt eine neue Saison in der Datenbank.
	 *
	 * @param Saison saison Die einzufügende Saison (die id innerhalb dieser Saison wird ignoriert)
	 */
	public static function createSaison($saison)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && ($saison instanceof Saison)) {
			$name = self::escapeSQLString($saison->getName());
			$begin = $saison->getBegin()->format("Y-m-d H:i:s");
			$end = $saison->getEnd()->format("Y-m-d H:i:s");
			$db = $gCms->GetDb();
			$result = $db->Execute("INSERT INTO ".cms_db_prefix().self::SaisonTable." (".self::$SFields[1].",".self::$SFields[2].",".self::$SFields[3].") VALUES ('".$name."','".$begin."','".$end."')");
		}
	}
	
	/**
	 * Diese Methode ändert die Saison oldSaison->getId() auf die Werte, die in newSaison stehen (bis auf id, diese wird
	 * bei newSaison ignoriert).
	 *
	 * @param Saison oldSaison Die alte Saison (nur id ist wichtig, muss mit der bisherigen Datenbankversion übereinstimmen)
	 * @param Saison newSaison Die neuen Werte dieser Saison (alle bis auf id it wichtig)
	 */
	public static function changeSaison($oldSaison, $newSaison)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && ($oldSaison instanceof Saison) && ($newSaison instanceof Saison)) {
			$savedsaison = self::getSaison($oldSaison->getId());
			if (($savedsaison != NULL) && ($savedsaison == $oldSaison) && ($oldSaison->getId() == $newSaison->getId())) {
				$db = $gCms->GetDb();
				$result = $db->Execute(
					"UPDATE ".cms_db_prefix().self::SaisonTable.
					" SET ".self::$SFields[1]."='".self::escapeSQLString($newSaison->getName()).
					"', ".self::$SFields[2]."='".$newSaison->getBegin()->format("Y-m-d H:i:s").
					"', ".self::$SFields[3]."='".$newSaison->getEnd()->format("Y-m-d H:i:s").
					"' WHERE ".self::$SFields[0]."=".$oldSaison->getId());
				if ($result) {
					return true;
				}
			}
		}
		
		return false;
	}
	
	/**
	 * Diese Methode erzeugt einen neuen Eintrag im Spielplan.
	 *
	 * @param Game game Das einzufügende Spiel wird innerhalb der 
	 * Datenbank erzeugt. Hierbei wird der Inhalt von $game->getId() ignoriert
	 */
	public static function createGame($game)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && ($game instanceof Game)) {
			$opponent = self::escapeSQLString($game->getOpponent());
			$date = $game->getDate()->format("Y-m-d H:i:s");
			$home = ($game->getHome() == true) ? 1:0;
			$ourteam = $game->getOurTeam()->getId();
			$db = $gCms->GetDb();
			$result = $db->Execute("INSERT INTO ".cms_db_prefix().self::GameTable." (".
			self::$GFields[1].",".self::$GFields[2].",".self::$GFields[3].",".self::$GFields[4].")".
			"VALUES (".$ourteam.",'".$opponent."','".$date."',".$home.")");
		}
	}
	
	/**
	 * Diese Liefert das spiel mit der gewünschten Id zurück
	 *
	 * @param int game Die Id der $game Instanz innerhalb der Datenbank
	 */
	public static function getGame($game)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && is_numeric($game)) {
			$db = $gCms->GetDb();
			$result = $db->Execute("SELECT * FROM ".cms_db_prefix().self::GameTable." WHERE id=".$game);
			if ($result) {
				if ($result->RecordCount() == 1) {
					$row = $result->FetchRow();
					$team = DatabaseOperations::getTeam($row[self::$GFields[1]]);
					return new Game($row[self::$GFields[0]], $team, $row[self::$GFields[2]], new DateTime($row[self::$GFields[3]]),$row[self::$GFields[4]]);
				}
			}
		}
		
		return NULL;
	}
	
	/**
	 * Diese Methode ändert ein bereits erstelltes Spiel innerhalb der Datenbank.
	 *
	 * @param Game oldgame Das ursprüngliche spiel. Dieses muss mit dem in der
	 * Datenbank hinterlegten Spiel übereinstimmen
	 * @param Game oldgame Die neuen Daten des spiels, die übernommen werden sollen
	 */
	public static function changeGame($oldgame, $newgame)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && ($oldgame instanceof Game) && ($newgame instanceof Game)) {
			$savedgame = self::getGame($oldgame->getId());
			if (($oldgame == $savedgame) && ($oldgame->getId() == $newgame->getId())) {
				$opponent = self::escapeSQLString($newgame->getOpponent());
				$date = $newgame->getDate()->format("Y-m-d H:i:s");
				$home = $newgame->getHome() ? 1:0;
				$ourteam = $newgame->getOurTeam()->getId();
				$db = $gCms->GetDb();
				$result = $db->Execute("UPDATE ".cms_db_prefix().self::GameTable." SET ".
				self::$GFields[1]."=".$ourteam.", ".
				self::$GFields[2]."='".$opponent."', ".
				self::$GFields[3]."='".$date."', ".
				self::$GFields[4]."=".$home." WHERE id=".$oldgame->getId());
				if ($result) {
					return true;
				}
			}
		}
		
		return false;
	}
	
	/**
	 * Diese Methode liefert den Spielplan (alle Spiele einer Mannschaft als Array)
	 * einer Mannschaft zu einer Saison zurück. Sollte keine Mannschaft übergeben
	 * werden, werden alle Spiele der Saison zurückgeliefert.
	 *
	 * @param saison Die Saison, für die der Spielplan ermittelt werden soll
	 * @param team Die Mannschaft, für die der Spielplan ermittelt werden soll
	 * @return Ein Array von Spielen, die o.g. Kriterien erfüllen
	 */
	public static function getTimeTable($saison, $team=NULL)
	{
		$gCms = $GLOBALS['gCms'];
		$games = array();
		if ($gCms && ($saison instanceof Saison) && (($team instanceof Team) || ($team == NULL))) {
			$queryteam = true;
			$saisonstart = $saison->getBegin()->format("Y-m-d H:i:s");
			$saisonend = $saison->getEnd()->format("Y-m-d H:i:s");
			$query = "SELECT * FROM ".cms_db_prefix().self::GameTable." WHERE ".
				self::$GFields[3].">='".$saisonstart."' AND ".
				self::$GFields[3]."<='".$saisonend."'";
			if ($team != NULL) {
				$query .= " AND ".self::$GFields[1]."=".$team->getId();
				$queryteam = false;
			}
			$query .= " ORDER BY ".self::$GFields[3];
			$db = $gCms->GetDb();
			$result = $db->Execute($query);
			for ($i = 0; $i < $result->RecordCount(); $i = $i + 1) {
				$row = $result->FetchRow();
				if ($queryteam) {
					$team = self::getTeam($row[self::$GFields[1]]);
				}
				$game = new Game($row[self::$GFields[0]], $team, $row[self::$GFields[2]], new DateTime($row[self::$GFields[3]]), $row[self::$GFields[4]]);
				array_push($games, $game);				
			}
		}
		
		return $games;
	}
	
	/**
	 * Diese Methode liefert den Spielplan (alle Spiele einer Mannschaft als Array)
	 * einer Mannschaft zurück, die in der Zukunft liegen (days > 0) oder in der 
	 * Vergangenheit liegen (days < 0). Der betrag gibt von days gibt dabei den
	 * gibt dabei die Anzahl der Tage in der Zukunft oder in der Vergangenheit an.
	 *
	 * @param days Anzahl der Tage in der Zukunft (days > 0) oder Anzahl der Tage in der Vergangenheit (days < 0)
	 * @param team Die Mannschaft, für die der Spielplan ermittelt werden soll (NULL für alle Teams)
	 * @return Ein Array von Spielen, die o.g. Kriterien erfüllen
	 */
        public static function getTimeTableTimediff($days, $team=NULL)
	{
		$gCms = $GLOBALS['gCms'];
		$games = array();
	
		if ($gCms && ($days != 0) && (($team instanceof Team) || ($team == NULL))) {
			$queryteam = true;
			$dayperiode = new DateInterval("P".abs($days)."D");
			$now = new DateTime;
			if ($days > 0) {
				$saisonstart = $now->format("Y-m-d H:i:s");
				$now->add($dayperiode);
				$saisonend = $now->format("Y-m-d H:i:s");
			} else {
				$saisonend = $now->format("Y-m-d H:i:s");
				$now->sub($dayperiode);
				$saisonstart = $now->format("Y-m-d H:i:s");
                        }
			$query = "SELECT * FROM ".cms_db_prefix().self::GameTable." WHERE ".
				self::$GFields[3].">='".$saisonstart."' AND ".
				self::$GFields[3]."<='".$saisonend."'";
			if ($team != NULL) {
				$query .= " AND ".self::$GFields[1]."=".$team->getId();
				$queryteam = false;
			}
			$query .= " ORDER BY ".self::$GFields[3];
			if ($days < 0) {
				$query .= " DESC";
			}
			$db = $gCms->GetDb();
			$result = $db->Execute($query);
			for ($i = 0; $i < $result->RecordCount(); $i = $i + 1) {
				$row = $result->FetchRow();
				if ($queryteam) {
					$team = self::getTeam($row[self::$GFields[1]]);
				}
				$game = new Game($row[self::$GFields[0]], $team, $row[self::$GFields[2]], new DateTime($row[self::$GFields[3]]), $row[self::$GFields[4]]);
				array_push($games, $game);				
			}
		}
	
		return $games;
	}
	
	/**
	 * Diese Methode erzeugt einen neuen Eintrag in der Ergebnisliste.
	 *
	 * @param Result result Das einzufügende Ergebnis wird innerhalb der 
	 * Datenbank erzeugt. Hierbei wird der Inhalt von $result->getId() ignoriert
	 */
	public static function createResult($result)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && ($result instanceof Result)) {
			$opponent = self::escapeSQLString($result->getOpponent());
			$player = $result->getPlayer()->getId();
			$spielplan = $result->getGame()->getId();
			$ges = $result->getGes();
			$abr = $result->getAbr();
			$fehl = $result->getFehl();
			$count = $result->getCount() ? 1:0;
			$oges = $result->getOGes();
			$oabr = $result->getOAbr();
			$ofehl = $result->getOFehl();
			$ocount = $result->getOCount() ? 1:0;
			$db = $gCms->GetDb();
			$result = $db->Execute("INSERT INTO ".cms_db_prefix().self::ResultTable." (".
				self::$RFields[1].",".
				self::$RFields[2].",".
				self::$RFields[3].",".
				self::$RFields[4].",".
				self::$RFields[5].",".
				self::$RFields[6].",".
				self::$RFields[7].",".
				self::$RFields[8].",".
				self::$RFields[9].",".
				self::$RFields[10].",".
				self::$RFields[11].")".
			" VALUES (".
				$player.",".
				$spielplan.",".
				$ges.",".
				$fehl.",".
				$abr.",".
				$count.",'".
				$opponent."',".
				$oges.",".
				$oabr.",".
				$ofehl.",".
				$ocount.")");
		}
	}
	
	/**
	 * Diese Methode liest einen bestehenden Eintrag aus der Ergebnisliste.
	 *
	 * @param int result Das zu lesende Ergebnis wird innerhalb der 
	 * Datenbank abgefragt. Der Parameter entspricht der id des
	 * Eintrags
	 * @return Result Das Resultat aus der Datenbank oder NULL wenn
	 * das Resultat nicht existiert
	 */
	public static function getResult($result)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && is_numeric($result)) {
			$db = $gCms->GetDb();
			$dbresult = $db->Execute("SELECT * FROM ".cms_db_prefix().self::ResultTable." WHERE ".
			self::$RFields[0]."=".$result);
			if ($dbresult->RecordCount() == 1) {
				$row = $dbresult->FetchRow();
				$result = new Result(
					$row[self::$RFields[0]],
					self::getPlayer($row[self::$RFields[1]]),
					self::getGame($row[self::$RFields[2]]),
					$row[self::$RFields[7]],
					$row[self::$RFields[3]],
					$row[self::$RFields[5]],
					$row[self::$RFields[4]],
					$row[self::$RFields[6]],
					$row[self::$RFields[8]],
					$row[self::$RFields[9]],
					$row[self::$RFields[10]],
					$row[self::$RFields[11]]);
				return $result;
			}
		}
		return NULL;
	}
	
	/**
	 * Diese Methode ändert einen bestehenden Eintrag aus der Ergebnisliste.
	 *
	 * @param Result oldresult Das zu ändernde Ergebnis (es muss dabei gelten,
	 * dass $oldresult dem alten Eintrag in er Datenbank entsprechen muss und
	 * $newresult->getId() = $oldresult->getId() erfüllt sein muss)
	 * @param Result newresult Das geändernde Ergebnis
	 * @return boolean true, wenn die Änderung erfolgreich war, sonst false
	 */
	public static function changeResult($oldresult, $newresult)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && ($oldresult instanceof Result) && ($newresult instanceof Result)) {
			$savedresult = self::getResult($oldresult->getId());
			if (($oldresult == $savedresult) && ($oldresult->getId() == $newresult->getId())) {
				$id = $newresult->getId();
				$opponent = self::escapeSQLString($newresult->getOpponent());
				$player = $newresult->getPlayer()->getId();
				$spielplan = $newresult->getGame()->getId();
				$ges = $newresult->getGes();
				$abr = $newresult->getAbr();
				$fehl = $newresult->getFehl();
				$count = $newresult->getCount() ? 1:0;
				$oges = $newresult->getOGes();
				$oabr = $newresult->getOAbr();
				$ofehl = $newresult->getOFehl();
				$ocount = $newresult->getOCount() ? 1:0;
				$db = $gCms->GetDb();
				$result = $db->Execute("UPDATE ".cms_db_prefix().self::ResultTable." SET ".
					self::$RFields[1]."=".$player.", ".
					self::$RFields[2]."=".$spielplan.", ".
					self::$RFields[3]."=".$ges.", ".
					self::$RFields[4]."=".$fehl.", ".
					self::$RFields[5]."=".$abr.", ".
					self::$RFields[6]."=".$count.", ".
					self::$RFields[7]."='".$opponent."', ".
					self::$RFields[8]."=".$oges.", ".
					self::$RFields[9]."=".$oabr.", ".
					self::$RFields[10]."=".$ofehl.", ".
					self::$RFields[11]."=".$ocount." WHERE ".
					self::$RFields[0]."=".$id
				);
				if ($result) {
					return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * Diese Methode liest die bestehenden Einträge aus der Ergebnisliste
	 * und gibt diese als array zurück.
	 *
	 * @param Game game Das Spiel innerhalb der Datenbank, zu dem die
	 * Ergebnisse ausgelesen werden sollen.
	 * @return array of Result Die Einzelergebnisse aus der Datenbank.
	 */
	public static function getResultOfGame($game)
	{
		$gCms = $GLOBALS['gCms'];
		$results = array();
		if ($gCms && ($game instanceof Game)) {
			$db = $gCms->GetDb();
			$dbresult = $db->Execute("SELECT * FROM ".cms_db_prefix().self::ResultTable." WHERE ".
			self::$RFields[2]."=".$game->getId());
			for ($i = 0; $i < $dbresult->RecordCount(); $i = $i + 1) {
				$row = $dbresult->FetchRow();
				$result = new Result(
				$row[self::$RFields[0]],
				self::getPlayer($row[self::$RFields[1]]),
				self::getGame($row[self::$RFields[2]]),
				$row[self::$RFields[7]],
				$row[self::$RFields[3]],
				$row[self::$RFields[5]],
				$row[self::$RFields[4]],
				$row[self::$RFields[6]],
				$row[self::$RFields[8]],
				$row[self::$RFields[9]],
				$row[self::$RFields[10]],
				$row[self::$RFields[11]]);
				array_push($results, $result);
			}
		}
		return $results;
	}
	
	/**
	 * Diese Methode legt in der Datenbank eine neue Zuordnung für
	 * den übergebenen Spieler, die übergebene Mannschaft und die
	 * übergebene Saison an.
	 * 
	 * @param Player $player
	 * @param Team $team
	 * @param Saison $saison
	 * @return true, falls die Zuordnung erfolgreich war, false sonst
	 */
	public static function createPlayerToTeam($player, $team, $saison)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && ($player instanceof Player) && ($team instanceof Team) && ($saison instanceof Saison)) {
			$db = $gCms->GetDb();
			$result = $db->Execute("INSERT INTO ".cms_db_prefix().self::PlayerToTeamTable." (".
			self::$PTFields[1].", ".self::$PTFields[2].", ".self::$PTFields[3].") VALUES (".
			$player->getId().", ".$saison->getId().", ".$team->getId().")");
			if ($result) {
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * Diese Function liest aus der Datenbank eine Zurodnung für Spieler,
	 * Saison und Mannschaft, deren id bekannt ist.
	 * 
	 * @param int $id
	 * @return PlayerToTeam Die zugehörige Zuordnung
	 */
	public static function getPlayerToTeam($id)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && is_numeric($id)) {
			$db = $gCms->GetDb();
			$result = $db->Execute("SELECT * FROM ".cms_db_prefix().self::PlayerToTeamTable." WHERE ".
			self::$PTFields[0]."=".$id);
			if ($result->RecordCount() == 1) {
				$row = $result->FetchRow();
				$player = self::getPlayer($row[self::$PTFields[1]]);
				$saison = self::getSaison($row[self::$PTFields[2]]);
				$team = self::getTeam($row[self::$PTFields[3]]);
				return new PlayerToTeam($id,$player,$saison,$team);
			}
		}
		
		return NULL;
	}
	
	/**
	 * Diese Methode ändert eine exitierende zuordnung von Spieler,
	 * Saison und Mannschaft.
	 * 
	 * @param PlayerToTeam $old Die Originalverknüpfung. Sie muss mit dem
	 * Inhalt in der Datenbank übereinstimmen.
	 * @param PlayerToTeam $new Die neue Verknüpfung. Es muss gelten
	 * $old->getId() == $new->getId()
	 * @return boolean true, if database could be changed, false otherwise
	 */
	public static function changePlayerToTeam($old, $new)
	{
		$gCms = $GLOBALS['gCms'];
		if ($gCms && ($old instanceof PlayerToTeam) && ($new instanceof PlayerToTeam)) {
			$saved = self::getPlayerToTeam($old->getId());
			if (($saved == $old) && ($old->getId() == $new->getId())) {
				$db = $gCms->GetDb();
				$result = $db->Execute("UPDATE ".cms_db_prefix().self::PlayerToTeamTable." SET ".
				self::$PTFields[1]."=".$new->getPlayer()->getId().", ".
				self::$PTFields[2]."=".$new->getSaison()->getId().", ".
				self::$PTFields[3]."=".$new->getTeam()->getId()." ".
				" WHERE ".self::$PTFields[0]."=".$old->getId());
				if ($result) {
					return true;
				}
			}
		}
		
		return false;
	}
	
	/**
	 * Diese Methode liefert aus der Datenbank alle zu einer Saison
	 * gehörenden Zuordnungen zwischen Spielern und Mannschaften.
	 * 
	 * @param Saison $saison Die Saison, für die die Zuordnungen der
	 * Datenbank ermittelt werden sollen
	 * @return array of PlayerToTeam, die gefundenen Zuordnungen
	 */
	public static function getPlayersToTeam($saison)
	{
		$gCms = $GLOBALS['gCms'];
		$playerstosaison = array();
		if ($gCms && ($saison instanceof Saison)) {
			$db = $gCms->GetDb();
			$result = $db->Execute("SELECT * FROM ".cms_db_prefix().self::PlayerToTeamTable." WHERE ".
			self::$PTFields[2]."=".$saison->getId());
			for ($i = 0; $i < $result->RecordCount(); $i = $i + 1) {
				$row = $result->FetchRow();
				$player = self::getPlayer($row[self::$PTFields[1]]);
				$team = self::getTeam($row[self::$PTFields[3]]);
				$ptt = new PlayerToTeam($row[self::$PTFields[0]],$player,$saison,$team);
				array_push($playerstosaison, $ptt);
			}
		}
		
		return $playerstosaison;
	}
}
?>
