<?php
#BEGIN_LICENSE
#-------------------------------------------------------------------------
# Module: CGJobMgr (c) 2011 by Robert Campbell 
#         (calguy1000@cmsmadesimple.org)
#  An addon module for CMS Made Simple to manage jobs that take considerable
#  amounts of time to process.
# 
#-------------------------------------------------------------------------
# CMS - CMS Made Simple is (c) 2005 by Ted Kulp (wishy@cmsmadesimple.org)
# This project's homepage is: http://www.cmsmadesimple.org
#
#-------------------------------------------------------------------------
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# However, as a special exception to the GPL, this software is distributed
# as an addon module to CMS Made Simple.  You may not use this software
# in any Non GPL version of CMS Made simple, or in any version of CMS
# Made simple that does not indicate clearly and obviously in its admin 
# section that the site was built with CMS Made simple.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
# Or read it online: http://www.gnu.org/licenses/licenses.html#GPL
#
#-------------------------------------------------------------------------
#END_LICENSE

class cgjobmgr_job
{
  const STATUS_UNSTARTED   = 'unstarted';
  const STATUS_INPROGRESS  = 'in_progress';
  const STATUS_ERROR       = 'error';
  const STATUS_COMPLETING  = 'completing';
  const STATUS_FINISHED    = 'finished';
  const STATUS_ABORTED     = 'aborted';
  const STATUS_PAUSED      = 'paused';
  const PRIORITY_HIGH      = 300;
  const PRIORITY_NORMAL    = 200;
  const PRIORITY_LOW       = 100;

  private $_id;
  private $_owner;
  private $_name;
  private $_priority;
  private $_status;
  private $_persistent;
  private $_recur_interval = 0;
  private $_start_at;
  private $_started_at;
  private $_finished_at;
  private $_locked;

  private $_start_task;
  private $_tasks;
  private $_end_task;


  /**
   * Constructor
   *
   * Autmatically creates a new job with a normal priority, and a status of UNSTARTED
   *
   * @param string The name for the job (optional)
   * @param integer The job owner's uid
   */
  public function __construct($name = '',$owner = -1,$priority = self::PRIORITY_NORMAL)
  {
    $this->_name = $name;
    $this->_priority = $priority;
    if( $owner == -1 ) {
      $owner = get_userid(FALSE);
    }
    $this->_owner = $owner;
    $this->set_status(self::STATUS_UNSTARTED);
    $this->set_persistent_value('__ORIG_PRIORITY__',$priority);
    $this->set_persistent_value('__ALLLOW_ITERATIVE_REDIRECT__',1);
  }


  public function __get($key)
  {
    switch( $key ) {
    case 'id':
    case 'owner':
    case 'name':
    case 'priority':
    case 'status':
    case 'persistent':
    case 'recur_interval':
    case 'start_at':
    case 'started_at':
    case 'finished_at':
      $key = '_'.$key;
      return $this->$key;
      break;

    default:
      throw new CGJobMgrException('Attempt to retrieve invalid property: '.$key.' from a job');
      
    }
  }

  /**
   * Return the job id (if the job has already been saved)
   *
   * @retrun mixed  integer job id if the job has been saved, or null
   */
  public function get_id()
  {
    return $this->_id;
  }


  /**
   * Set an optional start task (to be done before regular tasks)
   *
   * @param cgjobmgr_task The task object
   */
  public function set_start_task(cgjobmgr_task& $start_task)
  {
    $this->_start_task = $start_task;
  }


  /**
   * Retrieve the start task (if any)
   *
   * @returns cgjobmgr_task object, or null
   */
  public function get_start_task()
  {
    return $this->_start_task;
  }


  /**
   * Set an optional end task (to be done after regular tasks)
   *
   * @param cgjobmgr_task The task object
   */
  public function set_end_task(cgjobmgr_task& $end_task)
  {
    $this->_end_task = $end_task;
  }


  /**
   * Retrieve the end task (if any)
   *
   * @returns cgjobmgr_task object, or null
   */
  public function get_end_task()
  {
    return $this->_end_task;
  }


  /**
   * Add a task into the processing loop
   *
   *
   * @param cgjobmgr_task The task object
   */
  public function add_task(cgjobmgr_task& $task)
  {
    if( !is_array($this->_tasks) ) $this->_tasks = array();
    $this->_tasks[] = $task;
  }


  /**
   * Get the current status of the job
   *
   */
  public function get_status()
  {
    return $this->_status;
  }


  /**
   * Set the current status of the job
   * Most developers will not need this method.
   *
   * @internal
   */
  public function set_status($status,$message = null)
  {
    switch( $status )
      {
      case self::STATUS_FINISHED:
	$this->_finished_at = time();
	if( $this->_recur_interval > 0 ) {
	  $this->_start_at = $this->_finished_at + $this->_recur_interval * 60;
	  // its a recurring task, mark all of the tasks as incomplete now.
	  foreach( $this->_tasks as &$onetask ) {
	    $onetask->set_status(cgjobmgr_task::STATUS_NONE);
	  }
	}
	$runcount = $this->get_persistent_value('__RUNCOUNT__',0);
	$runcount++;
	$this->set_persistent_value('__RUNCOUNT__',$runcount);
	$this->_status = $status;
	break;

      case self::STATUS_UNSTARTED:
	$this->_started_at = null;
	$this->_finished_at = null;
	$this->_status = $status;
	break;
	
      case self::STATUS_INPROGRESS:
	$this->_started_at = time();
	$this->_status = $status;
	break;

      case self::STATUS_COMPLETING:
      case self::STATUS_ABORTED:
      case self::STATUS_ERROR:
	if( $message ) {
	  $this->set_persistent_value('__MESSAGE__',$message);
	}
	$this->_finished_at = time();
      case self::STATUS_PAUSED:
	$this->_status = $status;
	break;

      default:
	throw new CGJobMgrException('Attempt to set job to invalid status: '.$status);
      }
  }


  /**
   * Set the recur interval (in minutes).  may also adjust the priority.
   *
   * @param integer the start interval, in minutes, 0 is a valid value.
   * @return void
   */
  public function set_recur_interval($n)
  {
    $n = max(0,min($n,10000000));
    $this->_recur_interval = $n;
  }

  /**
   * Return the recur interval (if any)
   *
   * @return integer minutes between repetitions.
   */
  public function get_recur_interval()
  {
    return $this->_recur_interval;
  }

  /** 
   * Set the start time for this event.
   *
   * @param integer Unix timestamp representing the nearest start time.
   */
  public function set_start_time($t)
  {
    $t = min(time(),$t);
    $this->_start_at = $t;
  }


  /**
   * Get the start time for this event.  if any.
   *
   * @return integer Unix timestamp representing the nearest start time.
   */
  public function get_start_time()
  {
    return $this->_start_at;
  }

  /**
   * Retrieve the time that this job was started (if any)
   *
   * @return integer Unix timestamp representing the time the job was started, or null.
   */
  public function get_started_at()
  {
    return $this->_started_at;
  }

  /**
   * Retrieve the time that this job was finished (if any)
   *
   * @return integer Unix timestamp representing the time the job was started, or null.
   */
  public function get_finished_at()
  {
    return $this->_finished_at;
  }

  /**
   * Return the current job priority
   *
   * @returns integer
   */
  public function get_priority()
  {
    return $this->_priority;
  }


  /**
   * Set the job priority
   *
   * @param integer priority
   */
  public function set_priority($priority)
  {
    $priority = max($priority,0);
    $priority = min($priority,self::PRIORITY_HIGH);
    $this->_priority = $priority;
  }


  /**
   * Retrieve a piece of data attached to the job
   *
   * @param string The key for the data
   * @param mixed the default value for the data
   * @returns mixed
   */
  public function get_persistent_value($key,$dflt = '')
  {
    if( !is_array($this->_persistent) ) return $dflt;
    if( !isset($this->_persistent[$key]) ) return $dflt;
    return $this->_persistent[$key];
  }


  /**
   * Attach a piece of data to the job.
   * The data is serialized and stored with the job when it is saved.
   *
   * @param string The key for the data
   * @param mixed  The data to save.
   */
  public function set_persistent_value($key,$val)
  {
    if( !is_array($this->_persistent) ) $this->_persistent = array();
    $this->_persistent[$key] = $val;
  }


  /**
   * Return the current task that to process
   *
   * @internal
   * @returns cgjobmgr_task derived object
   */
  public function &get_current_task()
  {
    for( $i = 0; $i < count($this->_tasks); $i++ )
      {
	if( $this->_tasks[$i]->get_status() == cgjobmgr_task::STATUS_COMPLETE ) continue;
	
	return $this->_tasks[$i];
      }
    $res = null;
    return $res;
  }


  /**
   * Retrieve a flag indicating wether or not this job can take advantage of iterative redirects
   *
   * @return boolean
   */
  public function get_iterate_flag()
  {
    return $this->get_persistent_value('__ALLOW_ITERATIVE_REDIRECT__',1);
  }

  /**
   * Set a flag indicating wether or not this job can take advantage of iterative redirects.
   *
   * @param boolean
   */
  public function set_iterate_flag($flag = TRUE)
  {
    if( $flag )
      $flag = 1;
    else 
      $flag = 0;
    $this->set_persistent_value('__ALLOW_ITERATIVE_REDIRECT__',$flag);
  }

  /**
   * Retrieve a message associated with this job.
   *
   * Messages may be associated with a job when it terminates.  This method allows retrieving that message.
   *
   * @return string The message, if any has been set.
   */
  public function get_message()
  {
    return $this->get_persistent_value('__MESSAGE__');
  }

  /**
   * Set a message associated with this job.
   *
   * This may be used when a job terminates.
   *
   * @internal
   * @param string The message
   */
  public function set_message($msg)
  {
    $this->set_persistent_value('__MESSAGE__',$msg);
  }

  /**
   * Create a job object from database data
   *
   * @internal
   * @returns cgjobmgr_job object
   */
  public static function &load_from_data($row)
  {
    $obj = new cgjobmgr_job();
    foreach( $row as $key => $value )
      {
	switch( $key )
	  {
	  case 'id':
	  case 'name':
	  case 'owner':
	  case 'priority':
	  case 'status':
	  case 'recur_interval':
	  case 'start_at':
	  case 'started_at':
	  case 'finished_at':
	  case 'locked':
	    $fld = '_'.$key;
	    $obj->$fld = $value;
	    break;
	    
	  case 'assoc':
	    $obj->_persistent = unserialize($value);
	    break;
	    
	  case 'tasks':
	    $obj->_tasks = unserialize($value);
	    break;
	    
	  case 'created':
	  case 'modified':
	    break;

	  default:
	    throw new CGJobMgrException('Unknown field '.$key.' in row data');
	  }
      } // foreach

    return $obj;
  }

  public static function &load_by_id($jobid)
  {
    $db = cmsms()->GetDb();
    $query = 'SELECT * FROM '.cms_db_prefix().'module_cgjobmgr WHERE id = ?';
    $row = $db->GetRow($query,array($jobid));
    if( is_array($row) ) {
      return self::load_from_data($row);
    }

    $res = null;
    return $res;
  }

  private function _insert()
  {
    $db = cmsms()->GetDb();
    $query = 'INSERT INTO '.cms_db_prefix()."module_cgjobmgr
              (name,owner,priority,status,assoc,tasks,recur_interval,start_at,
               started_at,finished_at,created,modified)
              VALUES (?,?,?,?,?,?,?,?,?,?,NOW(),NOW())";
    $dbr = $db->Execute($query,
			array($this->_name,$this->_owner,$this->_priority,
			      $this->_status,serialize($this->_persistent),
			      serialize($this->_tasks),
			      $this->_recur_interval,$this->_start_at,
			      $this->_started_at,$this->_finished_at));
    if( !$dbr ) {
      throw new CGJobMgrException('Error Inserting job record ');
    }

    $this->_id = $db->Insert_Id();
    audit($this->_id,'CGJobMgr','New job created: '.$this->_name.' with priority '.$this->_priority);

    $mod = cms_utils::get_module('CGJobMgr');
    $mod->SendEvent('JobCreated',array('job'=>$this));
  }


  private function _update()
  {
    $db = cmsms()->GetDb();
    $query = 'UPDATE '.cms_db_prefix()."module_cgjobmgr
              SET name = ?, owner = ?, priority = ?, status = ?,
                  assoc = ?, tasks = ?, recur_interval = ?, start_at = ?,
                  started_at = ?, finished_at = ?,
                  modified = NOW()
              WHERE id = ?";
    $dbr = $db->Execute($query,
			array($this->_name,$this->_owner,$this->_priority,
			      $this->_status,serialize($this->_persistent),
			      serialize($this->_tasks),$this->_recur_interval,$this->_start_at,
			      $this->_started_at,$this->_finished_at,
			      $this->_id));
    if( !$dbr ) {
      throw new CGJobMgrException('Error Updating job record ');
    }
  }


  /**
   * Delete the current object from the database
   */
  public function delete()
  {
    if( $this->_id <= 0 ) {
      throw new CGJobMgrException('Cannot delete a job with no id.');
    }

    $db = cmsms()->GetDb();
    $query = 'DELETE FROM '.cms_db_prefix().'module_cgjobmgr WHERE id = ?';
    $dbr = $db->Execute($query,array($this->_id));
    $this->_id = null;

    $mod = cms_utils::get_module('CGJobMgr');
    $mod->SendEvent('JobDeleted',array('job'=>$this));
  }


  /**
   * Save the jobs current state to the database
   */
  public function save()
  {
    if( $this->_id <= 0 ) {
      $this->_insert();
    }
    else {
      $this->_update();
    }

    cgjobmgr_utils::notifier($this);
  }


  /**
   * Lock the current job so that it cannot be handled by the automatic process handling
   */
  public function lock()
  {
    if( !$this->_id )
      throw new CGJobMgrException('Attempt to lock a job that has not been saved');

    $db = cmsms()->GetDb();
    $query = 'UPDATE '.cms_db_prefix().'module_cgjobmgr SET locked = 1 WHERE id = ?';
    $dbr = $db->Execute($query,array($this->_id));
    if( $db->ErrorMsg() )
      throw new CGJobMgrException('Problem locking job '.$this->_id);
  }


  /**
   * Unlock the current job to allow for automatic processing.
   */
  public function unlock()
  {
    if( !$this->_id ) {
      throw new CGJobMgrException('Attempt to unlock a job that has not been saved');
    }

    $db = cmsms()->GetDb();
    $query = 'UPDATE '.cms_db_prefix().'module_cgjobmgr SET locked = 0 WHERE id = ?';
    $dbr = $db->Execute($query,array($this->_id));
    if( $db->ErrorMsg() )
      throw new CGJobMgrException('Problem unlocking job '.$this->_id);
  }
} // end of class

#
# EOF
#
?>