<?php
#BEGIN_LICENSE
#-------------------------------------------------------------------------
# Module: Promotions (c) 2008 by Robert Campbell 
#         (calguy1000@cmsmadesimple.org)
#  An addon module for CMS Made Simple to allow creating, interfacing with
#  and managing promotions or sales in the E-comemrce suite.
# 
#-------------------------------------------------------------------------
# CMS - CMS Made Simple is (c) 2005 by Ted Kulp (wishy@cmsmadesimple.org)
# This project's homepage is: http://www.cmsmadesimple.org
#
#-------------------------------------------------------------------------
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# However, as a special exception to the GPL, this software is distributed
# as an addon module to CMS Made Simple.  You may not use this software
# in any Non GPL version of CMS Made simple, or in any version of CMS
# Made simple that does not indicate clearly and obviously in its admin 
# section that the site was built with CMS Made simple.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
# Or read it online: http://www.gnu.org/licenses/licenses.html#GPL
#
#-------------------------------------------------------------------------
#END_LICENSE

$cgextensions = cms_join_path($gCms->config['root_path'],'modules',
			      'CGExtensions','CGExtensions.module.php');
if( !is_readable( $cgextensions ) )
{
  echo '<h1><font color="red">ERROR: The CGExtensions module could not be found.</font></h1>';
  return;
}
require_once($cgextensions);

define('PROMOTIONS_TABLE',cms_db_prefix().'module_promotions');
define('PROMOTIONS_COND_TABLE',cms_db_prefix().'module_promotions_cond');

class Promotions extends CGExtensions
{

  const NEWCOUPONFORM_TEMPLATE  = 'NEWCOUPONFORM_TEMPLATE';
  const DFLTCOUPONFORM_TEMPLATE = 'DFLTCOUPONFORM_TEMPLATE';
  const MAIN_PERM               = 'Manage Promotions';

  public function __construct()
  {
    parent::__construct();

    global $gCms;
    $smarty = $gCms->GetSmarty();
    
    $smarty->register_function('promo_get_prod_discount',
			       array('Promotions','smarty_function_promo_get_prod_discount'));
  }


  /*---------------------------------------------------------
   GetName()
   ---------------------------------------------------------*/
  public function GetName()
  {
    return 'Promotions';
  }

  /*---------------------------------------------------------
   GetFriendlyName()
   ---------------------------------------------------------*/
  public function GetFriendlyName()
  {
    return $this->Lang('friendlyname');
  }

	
  /*---------------------------------------------------------
   GetVersion()
   ---------------------------------------------------------*/
  public function GetVersion()
  {
    return '1.0.16';
  }


  /*---------------------------------------------------------
   GetHelp()
   ---------------------------------------------------------*/
  public function GetHelp()
  {
    return $this->Lang('help');
  }


  /*---------------------------------------------------------
   GetAuthor()
   ---------------------------------------------------------*/
  public function GetAuthor()
  {
    return 'calguy1000';
  }


  /*---------------------------------------------------------
   GetAuthorEmail()
   ---------------------------------------------------------*/
  public function GetAuthorEmail()
  {
    return 'calguy1000@cmsmadesimple.org';
  }


  /*---------------------------------------------------------
   GetChangeLog()
   ---------------------------------------------------------*/
  public function GetChangeLog()
  {
    return @file_get_contents(dirname(__FILE__).'/changelog.inc');
  }
  

  /*---------------------------------------------------------
   IsPluginModule()
   ---------------------------------------------------------*/
  public function IsPluginModule()
  {
	return true;
  }


  /*---------------------------------------------------------
   HasAdmin()
   ---------------------------------------------------------*/
  public function HasAdmin()
  {
    return true;
  }


  /*---------------------------------------------------------
   GetAdminSection()
   ---------------------------------------------------------*/
  public function GetAdminSection()
  {
    global $CMS_VERSION;
    if( version_compare($CMS_VERSION,'1.8','<') ) return 'extensions';
    return 'ecommerce';
  }


  /*---------------------------------------------------------
   GetAdminDescription()
   ---------------------------------------------------------*/
  public function GetAdminDescription()
  {
    return $this->Lang('moddescription');
  }


  /*---------------------------------------------------------
   VisibleToAdminUser()
   ---------------------------------------------------------*/
  public function VisibleToAdminUser()
  {
    return $this->CheckPermission(Promotions::MAIN_PERM) ||
      $this->CheckPermission('Modify Templates') ||
      $this->CheckPermission('Modify Site Preferences');
  }


  /*---------------------------------------------------------
   GetDependencies()
   ---------------------------------------------------------*/
  public function GetDependencies()
  {
    return array('CGExtensions'=>'1.19',
		 'Orders'=>'1.9.1',
		 'JQueryTools'=>'1.0.5',
		 'Products'=>'2.5.1');
  }


  /*---------------------------------------------------------
   MinimumCMSVersion()
   ---------------------------------------------------------*/
  public function MinimumCMSVersion()
  {
    return "1.9.4.3";
  }
	
	
  /*---------------------------------------------------------
   InstallPostMessage()
   ---------------------------------------------------------*/
  public function InstallPostMessage()
  {
    return $this->Lang('postinstall');
  }


  /*---------------------------------------------------------
   UninstallPostMessage()
   ---------------------------------------------------------*/
  public function UninstallPostMessage()
  {
    return $this->Lang('postuninstall');
  }


  /*---------------------------------------------------------
   UninstallPreMessage()
   ---------------------------------------------------------*/
  public function UninstallPreMessage()
  {
    return $this->Lang('really_uninstall');
  }	


  /*---------------------------------------------------------
   AllowAutoInstall()
   ---------------------------------------------------------*/
  public function AllowAutoInstall()
  {
    return FALSE;
  }	


  /*---------------------------------------------------------
   AllowAutoUpgrade()
   ---------------------------------------------------------*/
  public function AllowAutoUpgrade()
  {
    return FALSE;
  }	


  /*---------------------------------------------------------
   GetHeaderHTML()
   ---------------------------------------------------------*/
  public function GetHeaderHTML()
  {
    $txt = <<<EOT
<style type="text/css">
fieldset.cgfieldset {
  margin-left: 20%;
}
legend.cglegend {
  font-size: 1em;
  font-weight: normal;
}
.nooffset {
  margin-left: 0;
}
</style>
EOT;
    $obj = $this->GetModuleInstance('JQueryTools');
    if( is_object($obj) )
      {
$tmpl = <<<EOT
{JQueryTools action='incjs' exclude='form'}
{JQueryTools action='ready'}
EOT;
        $txt .= $this->ProcessTemplateFromData($tmpl);
      }
    return $txt;
  }	


  /*---------------------------------------------------------
   HasCapability()
   ---------------------------------------------------------*/
  public function HasCapability($capability, $params=array())
  {
    if( $capability == 'promotions_calculation' )
      return TRUE;

    return parent::HasCapability($capability,$params);
  }


  /*---------------------------------------------------------
   SetParameters()
   ---------------------------------------------------------*/
  public function SetParameters()
  {
    $this->AddImageDir('icons');
    $this->RegisterModulePlugin();
    $this->RestrictUnknownParams();

    $this->CreateParameter('inline','default',$this->Lang('param_inline'));
    $this->SetParameterType('inline',CLEAN_INT);
    
    $this->CreateParameter('template','',$this->Lang('param_template'));
    $this->SetParameterType('template',CLEAN_STRING);

    $this->SetParameterType(CLEAN_REGEXP.'/promo_.*/',CLEAN_STRING);
  }

  /**
   * Find applicable promotions given basket details, and other information
   *
   * @param int UserID from FEU module
   * @param array Array of items in the basket (array of objects)
   * @param float subtotal of order (before taxes or shipping)
   * @param float weight of order (products has the units)
   * @param int   unix format date for which order is applicable
   * @returns hash of promotion name, and amount of promotion.
   */
  public function FindPromotions($feu_uid,$items,$subtotal,$weight,$date = '')
  {
    if( empty($date) ) $date = time();
    $sess = new cge_session('Promotions');
    $db = $this->GetDb();
    $date = $db->DbTimeStamp($date);

    $query = 'SELECT id FROM '.PROMOTIONS_TABLE."
               WHERE (start_date <= $date) AND (end_date >= $date)
               ORDER BY offer_type ASC";
    $promotion_ids = $db->GetCol($query);
    if( !$promotion_ids )
      {
	return FALSE;
      }

    // now we have a list of promotion id's load each promotion
    // and test it.
    // calcualte the discounts applied for all line items
    // and calculate the value of them.
    $discount_amt = 10000000;
    $discount_items = '';
    foreach( $promotion_ids as $promo_id )
      {
	$match_type = '';
	$promo = promo_ops::load_promo_by_id($promo_id);
	if( !$promo ) continue;

	// check all of the conditions in the promotion
	$match = true;
	for( $i = 0; $i < $promo->count_conditions(); $i++ )
	  {
	    $condition = $promo->get_condition($i);
	    
	    switch( $condition->get_cond_type() )
	      {
	      case PROMOTIONS_COND_FEU:
		if( !promo_tests::test_feu_uid($feu_uid,$promo) )
		  {
		    $match = false;
		    break;
		  }
		break;

	      case PROMOTIONS_COND_SUBTOTAL:
		if( !promo_tests::test_order_subtotal($subtotal,$promo) )
		  {
		    $match = false;
		    break;
		  }
		break;

	      case PROMOTIONS_COND_COUPON:
		$coupons = $sess->get('coupons','');
		if( $coupons )
		  {
		    // test all of our coupons.
		    $test = false;
		    foreach( $coupons as $coupon )
		      {
			if( promo_tests::test_coupon($coupon,$promo) )
			  {
			    $test = true;
			  }
		      }
		    if( !$test )
		      {
			$match = false;
			break;
		      }
		  }
		else
		  {
		    $match = false;
		  }
		break;

	      case PROMOTIONS_COND_WEIGHT:
		if( !promo_tests::test_order_weight($weight,$promo) )
		  {
		    $match = false;
		    break;
		  }
		break;

	      case PROMOTIONS_COND_PRODID:
		if( !promo_tests::test_single_product($items,$promo) )
		  {
		    $match = false;
		    break;
		  }
		break;

	      case PROMOTIONS_COND_PRODCAT:
		if( !promo_tests::test_product_from_category($items,$promo) )
		  {
		    $match = false;
		    break;
		  }
		break;

	      case PROMOTIONS_COND_PRODHIER:
		if( !promo_tests::test_product_from_hierarchy($items,$promo) )
		  {
		    $match = false;
		    break;
		  }
		break;
	      }
	  }

	// check the value of the discount	
	if( $match )
	  {
	    $adj_items = promo_ops::calculate_discount($promo,$subtotal,$items);
	    $adj_amt   = promo_ops::calc_total_discount($adj_items);
	    if( $discount_amt > $adj_amt )
	      {
		$discount_amt = $adj_amt;
		$discount_items = $adj_items;
	      }
	    continue;
	  }
      }

    return $discount_items;
  }


  /**
   * Calculate the best discount for a product given an FEU user id, and a product id.
   * 
   * @returns a hash of results for the best discount found or FALSE.
   */
  static public function get_discount_for_product($feu_uid,$product_id)
  {
    $sess = new cge_session('Promotions');
    global $gCms;
    $db = $gCms->GetDb();
    $date = $db->DbTimeStamp(time());

    // get a list of the applicable promotions.
    $query = 'SELECT id FROM '.PROMOTIONS_TABLE."
               WHERE (start_date <= $date) AND (end_date >= $date)
               ORDER BY offer_type ASC";
    $promotion_ids = $db->GetCol($query);
    if( !$promotion_ids )
      {
	return FALSE;
      }

    // now we have a list of promotions applicable for this date
    // load each one and test it.
    $best_discount = '';
    foreach( $promotion_ids as $promo_id )
      {
	$promo = promo_ops::load_promo_by_id($promo_id);
	if( !$promo ) continue;

	$match = true;
	for( $i = 0; $i < $promo->count_conditions(); $i++ )
	  {
	    $condition = $promo->get_condition($i);
	    
	    switch( $condition->get_cond_type() )
	      {
	      case PROMOTIONS_COND_FEU:
		if( !promo_tests::test_feu_uid($feu_uid,$promo) )
		  {
		    $match = false;
		    break;
		  }
		break;

	      case PROMOTIONS_COND_COUPON:
		$coupons = $sess->get('coupons','');
		if( $coupons )
		  {
		    // test all of our coupons.
		    $test = false;
		    foreach( $coupons as $coupon )
		      {
			if( promo_tests::test_coupon($coupon,$promo) )
			  {
			    $test = true;
			  }
		      }
		    if( !$test )
		      {
			$match = false;
			break;
		      }
		  }
		else
		  {
		    $match = false;
		  }
		break;

	      case PROMOTIONS_COND_PRODID:
		if( !promo_tests::single_product_matches($promo,$product_id) )
		  {
		    $match = false;
		    break;
		  }
		break;

	      case PROMOTIONS_COND_PRODCAT:
		if( !promo_tests::single_product_from_category($promo,$product_id) )
		  {
		    $match = false;
		    break;
		  }
		break;

	      case PROMOTIONS_COND_PRODHIER:
		if( !promo_tests::single_product_from_hierarchy($promo,$product_id) )
		  {
		    $match = false;
		    break;
		  }
		break;
	      }
	  }

	if( $match )
	  {
	    // this promotion matches the product id
	    $res = promo_ops::calculate_product_discount($promo,$product_id);
	    if( is_array($res) )
	      {
		if( !is_array($best_discount) )
		  {
		    $best_discount = $res;
		    $best_discount['promo_id'] = $promo->get_id();
		  }
		else if( $res['percentage'] < $best_discount['percentage'] )
		  {
		    $best_discount = $res;
		    $best_discount['promo_id'] = $promo->get_id();
		  }
	      }
	  }
      }

    if( is_array($best_discount) )
      {
	return $best_discount;
      }
    return FALSE;
  }


  static public function clear_coupons()
  {
    $sess = new cge_session('Promotions');
    $sess->clear();
  }


  static public function add_coupon($code)
  {
    $sess = new cge_session('Promotions');
    $tmp = $sess->get('coupons','');
    if( is_array($tmp) )
      {
	if( !in_array($code,$tmp) )
	  $tmp[] = $code;
      }
    else
      {
	$tmp = array();
	$tmp[] = $code;
      }
    $sess->put('coupons',$tmp);
  }


  /**
   * A smarty plugin for finding the discount for a product
   */
  public function smarty_function_promo_get_prod_discount($params,&$smarty)
  {
    $feu_uid = -1;
    $product_id = -1;
    if( isset($params['feu_uid']) )
      {
	$feu_uid = (int)$params['feu_uid'];
      }
    else
      {
	$feu_module = module_helper::get_instance('FrontEndUsers');
	$feu_uid = $feu_module->LoggedInId(); 
      }

    if( isset($params['product_id']) )
      {
	$product_id = (int)$params['product_id'];
      }

    if( $product_id < 1 || $feu_uid < 1 ) 
      {
	// invalid/insufficient params 
	return;
      }

    $res = self::get_discount_for_product($feu_uid,$product_id);
    if( $res )
      {
	$res['decimal'] = $res['percentage'];
	$res['percentage'] *= 100.0;
      }
    if( isset($params['assign']) )
      {
	$smarty->assign($params['assign'],$res);
	return;
      }
    return $res;
  }
} // class

#
# EOF
#
?>
