<?php

/*
 *      Copyright 2010 Henrik Lennartsson
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

global $gCms;
$config = $gCms->GetConfig();

if (OpenID::GetConfigBool('openid.buggy.gmp')) {
    OpenID::Log('OpenID.module.php : Adapting to buggy GMP implementation');
    // See README for info, required because of buggy GMP
    define('Auth_OpenID_BUGGY_GMP', TRUE);
}

if (OpenID::GetConfigBool('openid.insecure.rand')) {
    OpenID::Log('OpenID.module.php : Using insecure rand function');
    // Should use insecure random source on windows
    define('Auth_OpenID_RAND_SOURCE', null);
}

if (OpenID::GetConfigBool('openid.ca.override')) {
    OpenID::Log('OpenID.module.php : Overriding default CA file');
    // Added to support curl with old root cert file
    // Download cacert.pem from http://curl.haxx.se/docs/caextract.html and place it in your OpenID directory

    define('Auth_OpenID_CURLOPT_CAINFO', $config['root_path'] . '/modules/OpenID/cacert.pem');
}

define('OpenIDModule_OPENID_DIR', 'openid-php-openid-782224d/');

$path_extra = dirname(dirname(dirname(__FILE__)));
$path = ini_get('include_path');
$path = $path_extra . PATH_SEPARATOR . dirname(__FILE__) . PATH_SEPARATOR . $path . PATH_SEPARATOR . dirname(__FILE__) . '/' . OpenIDModule_OPENID_DIR;
ini_set('include_path', $path);

require_once OpenIDModule_OPENID_DIR . "Auth/OpenID/Consumer.php";

$dirname = dirname(__FILE__);
require_once($dirname . DIRECTORY_SEPARATOR . 'lib/class.CMSMSDatabaseConnection.php');
require_once OpenIDModule_OPENID_DIR . "Auth/OpenID/MySQLStore.php";

require_once OpenIDModule_OPENID_DIR . "Auth/OpenID/SReg.php";
require_once OpenIDModule_OPENID_DIR . "Auth/OpenID/AX.php";
require_once OpenIDModule_OPENID_DIR . "Auth/OpenID/PAPE.php";

global $pape_policy_uris;
$pape_policy_uris = array(
    PAPE_AUTH_MULTI_FACTOR_PHYSICAL,
    PAPE_AUTH_MULTI_FACTOR,
    PAPE_AUTH_PHISHING_RESISTANT
);

// Check if Facebook classes already loaded - user might have installed FeuFacebook module
if (!class_exists('FacebookApiException')) {
    require_once 'facebook_2_1_2/facebook.php';
}

class OpenID extends CMSModule {

    // BEGIN Module Interface

    function GetName() {
        return 'OpenID';
    }

    function GetFriendlyName() {
        return $this->Lang('friendlyname');
    }

    function GetVersion() {
        return '1.2.1';
    }

    function GetHelp() {
        return $this->Lang('help');
    }

    function GetAuthor() {
        return 'Henrik Lennartsson';
    }

    function GetAuthorEmail() {
        return 'openid@lennartssons.net';
    }

    function GetChangeLog() {
        return $this->Lang('changelog');
    }

    function IsPluginModule() {
        return true;
    }

    function HasAdmin() {
        return true;
    }

    function GetAdminSection() {
        return 'usersgroups';
    }

    function GetAdminDescription() {
        return $this->Lang('moddescription');
    }

    function VisibleToAdminUser() {
        return TRUE;
    }

    function GetDashboardOutput() {
        return $this->Lang('dashboard_output');
    }

    function GetNotificationOutput($priority=2) {
        return '';
    }

    function GetDependencies() {
        return array();
    }

    function MinimumCMSVersion() {
        return "1.0";
    }

    function SetParameters() {
        $this->RegisterModulePlugin();
        $this->SetParameterType('openid_username', CLEAN_STRING);
        $this->SetParameterType('openid_identifier', CLEAN_STRING);
        $this->SetParameterType('mact', CLEAN_STRING);
        $this->SetParameterType('page', CLEAN_STRING);
        $this->SetParameterType('provider', CLEAN_STRING);
    }

    function GetEventDescription($eventname) {
        return $this->Lang('event_info_' . $eventname);
    }

    function GetEventHelp($eventname) {
        return $this->Lang('event_help_' . $eventname);
    }

    function InstallPostMessage() {
        return $this->Lang('postinstall');
    }

    function UninstallPostMessage() {
        return $this->Lang('postuninstall');
    }

    function UninstallPreMessage() {
        return $this->Lang('really_uninstall');
    }

    function DoAction($action, $id, $params, $returnid=-1) {
        global $gCms;
        $smarty = & $gCms->GetSmarty();
        $smarty->assign('openidactionid', $id);
        $smarty->assign('openidactionparams', $params);
        $smarty->assign('openidjqueryplugin_path', $gCms->config['root_url'] . '/modules/OpenID/jQueryOpenIdPlugin');
        $smarty->assign('openid_module_path', $gCms->config['root_url'] . '/modules/OpenID');
        $smarty->assign('openid_selector_path', $gCms->config['root_url'] . '/modules/OpenID/openid-selector_1_2');
        $smarty->assign('mod', $this);
        $smarty->assign($this->GetName(), $this);
        parent::DoAction($action, $id, $params, $returnid);
    }

    // END Module Interface

    function _GetAssociationsTableName() {
        return cms_db_prefix() . "module_openid_associations";
    }

    function _GetNoncesTableName() {
        return cms_db_prefix() . "module_openid_nonces";
    }

    function _GetStoreMysql() {

        // Create database connection based on CMSMS connection
        $db = new CMSMSDatabaseConnection();

        // create MySQL database storage area for OpenID data
        $store = new Auth_OpenID_MySQLStore($db, $this->_GetAssociationsTableName(), $this->_GetNoncesTableName());

        return $store;
    }

    /**
     * Creates an OpenID consumer
     * @return Auth_OpenID_Consumer
     */
    function GetConsumer() {
        $store = $this->_GetStoreMysql();
        $consumer = new Auth_OpenID_Consumer($store);
        return $consumer;
    }

    function getScheme() {
        $scheme = 'http';
        if (isset($_SERVER['HTTPS']) and $_SERVER['HTTPS'] == 'on') {
            $scheme .= 's';
        }
        return $scheme;
    }

    /**
     *
     * @param string $identity_url
     * @param string $provider
     * @return int the uid of the user, false if not found
     */
    function GetUserId($identity_url) {
        global $gCms;
        /* @var $db ADOConnection */
        $db = & $this->GetDb();
        $provider = $this->_GetProviderId($identity_url);
        $query = "SELECT id FROM " . cms_db_prefix() . "module_openid_users where openid_identity = ? and provider = ?";
        $this->Log($query . " " . $identity_url . " " . $provider);
        /* @var $result pear_ResultSet */
        $result = $db->Execute($query, array($identity_url, $provider));
        if (!$result || $result->NumRows() == 0) {
            $this->Log("Could not find user for identity " . $identity_url . " and provider " . $provider);
        }
        $rs = $result->FetchRow();
        return $rs ? $rs['id'] : FALSE;
    }

    /**
     * Returns array containing one array for each OpenID provider the user
     * has registered. Array values are
     *   provider : The id of the OpenID provider
     *   openid_identity : The identifier supplied by the provider upon successful authentication
     *   email : The user email if sent by the OpenID provider
     *
     * Returns empty array if user is not logged in
     * 
     * @return array 
     */
    function GetRegisteredOpenIDs() {
        $openids = array();
        global $gCms;
        /* @var $feu FrontEndUsers */
        $feu = $this->GetModuleInstance('FrontEndUsers');
        if (!$feu->LoggedIn()) {
            return $openids;
        }
        $uid = $feu->LoggedInId();
        /* @var $db ADOConnection */
        $db = & $this->GetDb();
        $query = "SELECT openid_identity, provider, email FROM " . cms_db_prefix() . "module_openid_users where id = ?";
        $this->Log($query . " " . $uid);
        /* @var $result pear_ResultSet */
        $result = $db->Execute($query, array($uid));
        if (!$result || $result->NumRows() == 0) {
            $this->Log("No providers registered for user " . $uid);
            return $openids;
        }
        for ($index = 0; $index < $result->NumRows(); $index++) {
            $row = $result->FetchRow();
            $openids[] = $row;
        }
        return $openids;
    }

    /**
     * Register an OpenID identity served by a provider with a user.
     *
     * @param <type> $uid The unique user identifier
     * @param <type> $identity_url The identity URL provided by OpenID
     * @param string email The users email, can be empty
     */
    function RegisterOpenID($uid, $identity_url, $email = '') {
        global $gCms;
        /* @var $db ADOConnection */
        $db = & $this->GetDb();
        $provider = $this->_GetProviderId($identity_url);
        $this->Log("RegisterOpenID : uid = $uid url = $identity_url email = $email provider = $provider");
        $query = "SELECT COUNT(*) AS count FROM " . cms_db_prefix() . "module_openid_users where id = ? and provider = ?";
        $rs = $db->GetRow($query, array($uid, $provider));
        OpenID::Log("result : " . print_r($rs, TRUE));
        $registered = $rs && $rs['count'] > 0;
        OpenID::Log("registered : " . print_r($registered, TRUE));
        if ($registered) {
            $query = "UPDATE " . cms_db_prefix() . "module_openid_users
                 SET openid_identity = ?, email = ? where id = ? and provider = ?";
        } else {
            $query = "INSERT INTO " . cms_db_prefix() . "module_openid_users
                (openid_identity, email, id, provider) VALUES (?, ?, ?, ?)";
        }
        OpenID::Log("query : " . $query);
        $result = $db->Execute($query, array($identity_url, $email, $uid, $provider));
        if (!$result) {
            echo "DEBUG: SQL = " . $db->sql . "<br/>";
            die($db->ErrorMsg());
        }
    }

    /**
     *
     * @param long $uid The unique user identifier
     * @param String $provider The name of the OpenID provider
     * @return boolean TRUE if succesful, FALSE otherwise
     */
    function RemoveOpenID($uid, $provider) {
        global $gCms;
        /* @var $db ADOConnection */
        $db = & $this->GetDb();
        $this->Log("RemoveOpenID : uid = $uid provider = $provider");
        $query = "DELETE FROM " . cms_db_prefix() . "module_openid_users where id = ? and provider = ?";
        OpenID::Log("query : " . $query);
        $result = $db->Execute($query, array($uid, $provider));
        if (!$result) {
            echo "DEBUG: SQL = " . $db->sql . "<br/>";
            $this->Log("RemoveOpenID : Could not remove provider $provider for user $uid. Cause : " . $db->sql);
            return FALSE;
        }
        return TRUE;
    }

    /**
     * @param string $identity the openID identity the user has provided
     * @return string a string identifying the OpenID provider like 'google' or 'openid'
     */
    function _GetProviderId($identity) {
        if (substr_count($identity, '.yahoo.com/')) {
            return 'yahoo';
        } else if (substr_count($identity, '.google.com/')) {
            return 'google';
        } else if (substr_count($identity, '.myopenid.com')) {
            return 'myopenid';
        } else if (substr_count($identity, '.aol.com/')) {
            return 'aol';
        } else if (substr_count($identity, 'myspace.com')) {
            return 'myspace';
        } else if (substr_count($identity, 'facebook:')) {
            return 'facebook';
        } else if (substr_count($identity, 'twitter:')) {
            return 'twitter';
        } else {
            return 'openid';
        }
    }

    /**
     *
     * @param string $provider The id for the provider
     * @return <type>
     */
    function DisplayProvider($provider) {
        $preference = $this->GetPreference('enable_provider_' . $provider, '0');
        return!empty($preference);
    }

    /**
     * Log message to error log if logging is enabled in module settings
     * @param String $message
     */
    function Log($message) {
        global $gCms;
        $config = $gCms->GetConfig();
        if (OpenID::GetConfigBool('openid.debug')) {
            error_log($message);
            // debug_display($message);
        }
    }

    /**
     * Wrapper that removes any admin directory when referencing the moduleinterface.php.
     * Will happen in case the returnid variable is empty.
     * See http://forum.cmsmadesimple.org/viewtopic.php?f=7&t=56088
     * (User could only add provider if he/she was already logged on to the admin panel)
     */
    function CreateLink($id, $action, $returnid = '', $contents = '', $params = array(), $warn_message = '', $onlyhref = false, $inline = false, $addttext = '', $targetcontentonly = false, $prettyurl = '') {
        global $gCms;
        $link = parent::CreateLink($id, $action, $returnid, $contents, $params, $warn_message, $onlyhref, $inline, $addttext, $targetcontentonly, $prettyurl);
        // Remove admin directory from path, we want to use the default (non-admin) moduleinterface.php
        // See http://forum.cmsmadesimple.org/viewtopic.php?f=7&t=56088
        $config = $gCms->GetConfig();
        $adminDir = $config['admin_dir'];
        $link = str_replace("/$adminDir/moduleinterface.php", "/moduleinterface.php", $link);
        $link = str_replace(urlencode("/$adminDir/moduleinterface.php"), "/moduleinterface.php", $link);
        return $link;
    }
    
    /**
     *
     * @return String the Facebook URL that will trigger the remote login and redirect
     * back to action.facebook.php 
     */
    function FacebookLoginUrl($id, $action, $returnid) {
        /** @var this OpenID */
        // Create link to action.facebook.php. If there's a redirect configured we want a link to moduleinterface.php
        $redirectId = $this->GetRedirectLoginPageId();
        if ($redirectId) {
            $next = $this->CreateLink($id, $action, null, null, null, null, true, false, null, null);
        } else {
            $next = $this->CreateLink($id, $action, $returnid, null, null, null, true, true, null, null);
        }
        // Use current page as cancel-URL
        global $gCms;
        $currentUrl = $gCms->GetHierarchyManager()->getNodeById($gCms->variables['content_id'])->GetContent()->GetURL();
        // Generate login URL
        /* @var facebook Facebook */
        $facebook = new Facebook(array(
                    'appId' => $this->GetPreference('facebook_app_id', ''),
                    'secret' => $this->GetPreference('facebook_app_secret', ''),
                    'cookie' => true,
                ));
        // Fix: Remove xml encoded amps before passing to facebook library
        $loginUrl = $facebook->getLoginUrl(array(
                    'cancel_url' => str_replace('amp;', '', $currentUrl),
                    'next' => str_replace('amp;', '', $next)));
        // Fix: Replace amps in order to be xhtml compliant
        $loginUrl = str_replace('&', '&amp;', $loginUrl);
        $this->Log('FacebookLoginUrl : ' . $loginUrl);
        return $loginUrl;
    }

    /**
     * Custom method for getting a bool value from the CMS configuration.
     * In CMSMS 1.9 the array has been replaced with an object implementing
     * ArrayAccess. If CMSMS is in debug mode the class will throw an exception
     * if the key does not exist. This method will check if the key exists before
     * returning the value (if any) to work around the exception.
     * @param String $key
     * @return Boolean true of the key is set in the config array and set to non-false value
     */
    public static function GetConfigBool($key) {
        global $gCms;
        $config = $gCms->GetConfig();
        if ($config instanceof ArrayAccess) {
            return $config->offsetExists($key) && $config[$key];
        } else {
            return array_key_exists($key, $config) && $config[$key];
        }
    }

    public function GetRedirectLoginPageId() {
        $redirectLogin = $this->GetPreference('login_redirect', 'none');
        $redirectLoginPage = $this->GetPreference('login_redirect_pageid');
        $feuRedirect = $this->GetModuleInstance('FrontEndUsers')->GetPreference('pageid_login');
        $this->Log("Redirect after login is set to $redirectLogin (custom page/id = $redirectLoginPage and FrontEndUser redirect = $feuRedirect");
        if ('none' == $redirectLogin or !$redirectLogin) {
            $this->Log("Not using any redirect after login");
            return '';
        } else if ('feu' == $redirectLogin) {
            $id = ContentManager::GetPageIDFromAlias($feuRedirect);
            $this->Log("Using redirect setting $feuRedirect from FrontEndUses. Resolved this to ID $id");
            if (!$id) {
                die("Illegal setting in FrontEndUsers, page with id/alias $feuRedirect not found");
            }
            return $id;
        } else {
            $id = ContentManager::GetPageIDFromAlias($redirectLoginPage);
            $this->Log("Using custom OpenID module redirect $redirectLoginPage. Resolved this to ID $id");
            if (!$id) {
                die("Illegal setting in OpenID module, page with id/alias $redirectLoginPage not found");
            }
            return $id;
        }
    }
    
}

?>
