<?php
#BEGIN_LICENSE
#-------------------------------------------------------------------------
# Module: Availability (c) 2008 by Robert Campbell 
#         (calguy1000@cmsmadesimple.org)
#  An addon module for CMS Made Simple to provide full resource management
#  capabilities and reservation support.  It is designed to be a resource
#  manager for hotels, or cars, or other complex items that are reserved
#  on a daily basis.
#
#-------------------------------------------------------------------------
# CMS - CMS Made Simple is (c) 2005 by Ted Kulp (wishy@cmsmadesimple.org)
# This project's homepage is: http://www.cmsmadesimple.org
#
#-------------------------------------------------------------------------
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# However, as a special exception to the GPL, this software is distributed
# as an addon module to CMS Made Simple.  You may not use this software
# in any Non GPL version of CMS Made simple, or in any version of CMS
# Made simple that does not indicate clearly and obviously in its admin 
# section that the site was built with CMS Made simple.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
# Or read it online: http://www.gnu.org/licenses/licenses.html#GPL
#
#-------------------------------------------------------------------------
#END_LICENSE

function availability_AdjustPrice(&$mod,$price,$adjustment)
{
    if( empty($adjustment) ) return $price;

    $op = '+';
    $tmp = $adjustment[0];
    if( $tmp == '*' || $tmp == '+' || $tmp == '-' )
      {
	$op = $tmp;
	$adjustment = substr($adjustment,1);
      }
    if( !is_numeric($adjustment) ) return $price;
    $adjustment = floatval($adjustment);
    switch( $op )
      {
      case '+':
	$price += $adjustment;
	break;
      case '-':
	$price -= $adjustment;
	break;
      case '*':
	$price *= $adjustment;
	break;

      default:
	return false;
      }

    return $price;
}


function availability_CalculateReservationParts(&$mod,&$reservation)
{
    // I know this routine could probably be a helluva lot more efficient
    // but I'll re-tackle that at a later time.
    $db =& $mod->GetDb();

    // get the unique resource ids
    $resources = array();
    for( $i = 0; $i < count($reservation['parts']); $i++ )
      {
	$resources[] = $reservation['parts'][$i]['rsrc_id'];
      }
    $parts = cge_array::to_hash($reservation['parts'],'rsrc_id');

    // 1.  Get base price and other info for selected resources
    $q = 'SELECT id,base_price,capacity,res_interval FROM '.AVAILABILITY_TABLE_RSRCS.' WHERE active =1 AND id IN('.implode(',',$resources).')';
    $tmp = $db->GetArray($q);
    $rsrcs = cge_array::to_hash($tmp,'id');

    // 2.  Get weekday price adjustments and other adjustments for selected resources.
    //     and put them together.
    $q = 'SELECT * FROM '.AVAILABILITY_TABLE_RSRC_PRICES.' WHERE rsrc_id IN ('.implode(',',$resources).')';
    $tmp = $db->GetArray($q);
    foreach( $tmp as $one )
      {
	$tid = $one['rsrc_id'];
	$tmp2 = array();
	$tmp2[0] = $one['sunday'];
	$tmp2[1] = $one['monday'];
	$tmp2[2] = $one['tuesday'];
	$tmp2[3] = $one['wednesday'];
	$tmp2[4] = $one['thursday'];
	$tmp2[5] = $one['friday'];
	$tmp2[6] = $one['saturday'];
	$rsrcs[$tid]['weekdays'] = $tmp2;
	$rsrcs[$tid]['overcap'] = $one['overcap'];
	$rsrcs[$tid]['discount_days'] = $one['discount_days'];
	$rsrcs[$tid]['discount_adjust'] = $one['discount_adjust'];
	$rsrcs[$tid]['discount_type'] = $one['discount_type'];
      }

    // 4.  for each reservation part
    for( $pn = 0; $pn < count($reservation['parts']); $pn++ )
      {
	$thepart =& $reservation['parts'][$pn];
	$onersrc =& $rsrcs[$thepart['rsrc_id']];
	$part_start_date = $db->DbTimeStamp($parts[$onersrc['id']]['start_date']);
	$part_end_date = $db->DbTimeStamp($parts[$onersrc['id']]['end_date']);

	// Get events in effect for the selected dates, for this resource
	$q = 'SELECT DISTINCT id,name,bookable,price_adjust,TO_DAYS(start_date) AS start_day,TO_DAYS(end_date) as end_day 
          FROM '.AVAILABILITY_TABLE_EVENTS.' E
          LEFT JOIN '.AVAILABILITY_TABLE_RSRC_EVENTS." RE
          ON E.id = RE.event_id
          WHERE RE.rsrc_id = ?
            AND NOT ($part_start_date > E.end_date OR $part_end_date < E.start_date)";
	$tmp = $db->GetArray($q,array($thepart['rsrc_id']));
	$q = 'SELECT rsrc_id FROM '.AVAILABILITY_TABLE_RSRC_EVENTS.' WHERE event_id = ?';
	$events = array();
	foreach( $tmp as $oneevent )
	  {
	    $tmp2 = $db->GetArray($q,array($oneevent['id']));
	    $oneevent['rsrcs'] = cge_array::extract_field($tmp2,'rsrc_id');
	    $events[] = $oneevent;
	  }

	// get some date information
	$q = "SELECT TO_DAYS($part_start_date) AS start_day, TO_DAYS($part_end_date) AS end_day";
	$daynumbers = $db->GetRow($q);
	$weekdays = array();
	for( $d = $daynumbers['start_day']; $d <= $daynumbers['end_day']; $d++ )
	  {
	    $q = "SELECT DAYOFWEEK(FROM_DAYS($d))-1";
	    $weekdays[$d] = $db->GetOne($q);
	  }

	// adjust for daily resources
	if( $onersrc['res_interval'] == 'daily' )
	  {
	    $daynumbers['end_day'] -= 1;
	  }

	//  a. for each day
	$pricedays = array();
	for( $daynum = $daynumbers['start_day']; $daynum <= $daynumbers['end_day']; $daynum ++ )
	  {
	    // don't do calculations for same day twice.
	    if( isset($pricedays[$daynum]) ) continue;

	    // i. set the price for that day to the base price
	    $tmp2 = array('base_price'=>$onersrc['base_price'],
			  'price'=>$onersrc['base_price'],
			  'adjustments'=>array(),
			  'overcap_amt'=>0);

	    // ii. adjust the price based on the weekday
	    $dow = $weekdays[$daynum];
	    if( !empty($onersrc['weekdays'][$dow]) )
	      {
		$tmp2['price'] = $mod->AdjustPrice($tmp2['base_price'],$onersrc['weekdays'][$dow]);
		$tmp2['adjustments'][] = $mod->Lang('adjust_weekday_price',$dow,$onersrc['weekdays'][$dow]);
	      }

	    // iii. adjust the price based on overcapacity
	    if( $onersrc['capacity'] > 0 && $onersrc['overcap'] > 0 &&
		$thepart['num_people'] > $onersrc['capacity'])
	      {
		$num = ($thepart['num_people'] - $onersrc['capacity']);
		$overcap =  $num * $onersrc['overcap'];
		$tmp2['price'] += $overcap;
		$tmp2['adjustments'][] = $mod->Lang('adjust_overcapacity',$num,$overcap);
	      }

	    // iv. for each event
	    foreach( $events as $oneevent )
	      {
		// 1. test if event in effect for this resource
		if( !in_array($onersrc['id'],$oneevent['rsrcs']) ) continue;

		// 2. if the event is not bookable, return error
		if( !$oneevent['bookable'] ) 
		  {
		    // shouldn't get here due to query, but just in case.
		    continue;
		  }

		// 3.  test if the event is in effect for this day
		if( $daynum < $oneevent['start_day'] || $daynum > $oneevent['end_day'] )
		  {
		    // shouldn't get here due to query, but just in case
		    continue;
		  }
	
		if( !empty($oneevent['price_adjust']) )
		  {
		    // a. mark price as adjusted
		    $tmp2['adjustments'][] = $mod->Lang('adjust_event_price',$oneevent['id'],$oneevent['price_adjust']);

		    // b. adjust price for that day
		    $tmp2['price'] = $mod->AdjustPrice($tmp2['price'],$oneevent['price_adjust']);
		  }
	      }

	    $pricedays[$daynum] = $tmp2;
	  } // for daynum

	/* bulk discounts
	// now appy bulk discounts if any.
	if( $onersrc['discount_days'] > 0 && 
	    $onersrc['discount_type'] != AVAILABILITY_DISCOUNT_NONE &&
	    !empty($onersrc['discount_adjust']) &&
	    count($pricedays) > $onersrc['discount_days'] )
	  {
	    // woot, a discount of some type applies.
	    switch($onersrc['discount_type'])
	      {
	      case AVAILABILITY_DISCOUNT_ALLADJUST:
		// the simplest of the adjustment types
		// we just adjust the price of each day by the amount specified
		foreach( $pricedays as $daynum => $tmp2 )
		  {
		    $tmp2['price'] = $mod->AdjustPrice($tmp2['price'],$onersrc['discount_adjust']);
		    $tmp2['adjustments'][] = $mod->Lang('adjust_discount_alladjust',$onersrc['discount_adjust']);
		    $pricedays[$daynum] = $tmp2;
		  }
		break;

	      case AVAILABILITY_DISCOUNT_ADJUSTDAYS:
		// adjust only the matching days by the adjustment
		$n = count($pricedays) - (count($pricedays) % $onersrc['discount_days']);
		foreach( $pricedays as $daynum => $tmp2 )
		  {
		    if( $n <= 0 ) break;
		    $n--;

		    $tmp2['price'] = $mod->AdjustPrice($tmp2['price'],$onersrc['discount_adjust']);
		    $tmp2['adjustments'][] = $mod->Lang('adjust_discount_adjustdays',$onersrc['discount_adjust']);
		    $pricedays[$daynum] = $tmp2;
		  }
		break;

	      case AVAILABILITY_DISCOUNT_FREEBLOCK:
		// the most complex of the discounting
		// for each applicable block of N days... find the cheapest
		// and mark those days as free..
		$adjust = (int)$onersrc['discount_adjust'];
		$adjust = max(1,$adjust);
		$adjust = min($onersrc['discount_days'],$adjust);
		$n = ((int)(count($pricedays)/$onersrc['discount_days'])) * $adjust;
		for( $i = 0; $i < $n; $i++ )
		  {
		    $val = 9999999;
		    $vi = -1;
		    foreach( $pricedays as $daynum => $tmp2 )
		      {
			if( $tmp2['price'] < $val && !isset($tmp2['discounted']))
			  {
			    $val = $tmp2['price'];
			    $vi = $daynum;
			  }
		      }
		    if( $vi > 0 )
		      {
			$pricedays[$vi]['price'] = 0;
			$pricedays[$vi]['discounted'] = 1;
			$pricedays[$vi]['adjustments'][] = $mod->Lang('adjust_discount_freeblock');
		      }
		  }
		  break;
		}
	      } // bulk discounts 
         */

	// we have calculated data for this day
	// now merge it into the master.
	$total = 0;
	foreach( $pricedays as $tmpday => $tmprec )
	  {
	    foreach( $tmprec as $name => $value )
	      {
		switch($name)
		  {
		  case 'price':
		    $total += $value;
		    break;

		  case 'adjustments':
		    if( isset($thepart['adjustments']) && is_array($thepart['adjustments']) )
		      {
			$thepart['adjustments'] = array_merge($thepart['adjustments'],$value);
		      }
		    else
		      {
			$thepart['adjustments'] = $value;
		      }
		    break;
		  }
	      }
	  }
	$thepart['price'] = $total;
      } // foreach part

    $reservation['calculated'] = 1;
}


function availability_CalculateReservationPrice(&$mod,&$reservation)
{
  if( !isset($reservation['calculated']) || $reservation['calculated'] == 0 )
    {
      $mod->CalculateReservationParts($reservation);
    }

  // set total to 0
  $total = 0;
  if( !count($reservation['parts']) ) return FALSE;

  // or each PART
  for( $i = 0; $i < count($reservation['parts']); $i++ )
    {
      $total = $total + $reservation['parts'][$i]['price'];
    }
  
  // return the total price
  return $total;
}


function availability_CheckReservationConflicts(&$mod,$reservation,$allconflicts = 0)
{
  // Check for conflicts with other reservations
  $gCms = cmsms();
  $db = $gCms->GetDb();

  for( $i = 0; $i < count($reservation['parts']); $i++ )
    {
      $part =& $reservation['parts'][$i];
      $start_date = $db->DbTimeStamp($part['start_date']);
      $end_date = $db->DbTimeStamp($part['end_date']);
	
      $qparms = array();
      $query = 'SELECT a.resv_id,b.status FROM '.AVAILABILITY_TABLE_RESERVATION_PARTS." a
                  LEFT JOIN ".AVAILABILITY_TABLE_RESERVATIONS." b
                    ON a.resv_id = b.id
                 WHERE (( (a.start_date BETWEEN $start_date AND $end_date) OR (a.end_date BETWEEN $start_date AND $end_date))
                    OR (a.start_date < $start_date AND a.end_date > $end_date))
                   AND a.rsrc_id = ?";
      $qparms[] = $part['rsrc_id'];

      if( !$allconflicts )
	{
	  // only check against confirmed reservations.
	  $query .= ' AND b.status = ?';
	  $qparms[] = AVAILABILITY_STATUS_CONFIRMED;
	}
      if( isset($reservation['id']) )
	{
	  $query .= " AND a.resv_id != ?";
	  $qparms[] = $reservation['id'];
	}
      $query .= " LIMIT 1";
      $tmp = $db->GetOne($query,$qparms);
      if( $tmp )
	{
	  return true;
	}

      // Check for conflicts with events
      $qparms = array();
      $query = 'SELECT re.rsrc_id FROM '.AVAILABILITY_TABLE_EVENTS.' ev
                  LEFT JOIN '.AVAILABILITY_TABLE_RSRC_EVENTS." re
                    ON ev.id = re.event_id
                 WHERE ((ev.start_date BETWEEN $start_date AND $end_date)
                    OR (ev.end_date BETWEEN $start_date AND $end_date))
                   AND rsrc_id IN = ?
                   AND ev.bookable = 0
                 LIMIT 1";
      $qparms[] = $part['rsrc_id'];
      $tmp = $db->GetOne($query,$qparms);
      if( $tmp )
	{
	  return true;
	}
    }

  return false;
}


function availability_AdjustReservationPartInterval2(&$mod,&$rsrc,$start_date,$end_date,$meridian = '')
{
  switch( $rsrc['res_interval'] )
    {
    case 'daily':
      $start_date = cge_date_utils::ts_set_time_from_str($start_date,$rsrc['checkintime']);
      $end_date = cge_date_utils::ts_set_time_from_str($end_date,$rsrc['checkouttime']);
      // ensure start_date is less than end date
      // if not, add 1 day to end date
      if( $start_date >= $end_date )
	{
	  $end_date = strtotime('+1 day',$end_date);
	}
      break;

    case 'hourly':
      $end_tmp = new cge_date($start_date);
      $end_tmp->set_hour($end_tmp->hour()+1);
      $end_tmp->set_seconds(0);
      $end_date = $end_tmp->to_timestamp();
      break;

    case 'half_day':
      switch($meridian)
	{
	case 'am':
	  $start_date = cge_date_utils::ts_set_time_from_str($start_date,$rsrc['starttime1']);
	  $end_date   = cge_date_utils::ts_set_time_from_str($start_date,$rsrc['endtime1']);
	  break;
	case 'pm':
	  $start_date = cge_date_utils::ts_set_time_from_str($start_date,$rsrc['starttime2']);
	  $end_date   = cge_date_utils::ts_set_time_from_str($start_date,$rsrc['endtime2']);
	  break;
	default:
	  $res['result'] = 'error';
	  $res['message'] = $mod->Lang('error_invalid_dates');
	  break;
	}
      break;
    }

  return array($start_date,$end_date);
}


function availability_GetResourceSummary(&$mod,&$rsrc)
{
  // display a summary report about mod resource... should tell
  // the type of resource, how available times and days.
  $tmp = array();
  //$tmp[] = $rsrc['res_inverval'];
  if( $rsrc['res_interval'] == 'hourly' )
    {
      $tmp[] = $mod->Lang('resource_summary_1',
			  $mod->Lang($rsrc['res_interval']),
			  $mod->Lang($rsrc['granularity']));
      $tmp[] = $mod->Lang('resource_summary_3',
			   $rsrc['starttime1'],
			   $rsrc['endtime1'],
			   $rsrc['starttime2'],
			   $rsrc['endtime2']);
    }
  else if( $rsrc['res_interval'] == 'half_day' )
    {
      $tmp[] = $mod->Lang('resource_summary_2',$mod->Lang($rsrc['res_interval']));
      $tmp[] = $mod->Lang('resource_summary_3',
			   $rsrc['starttime1'],
			   $rsrc['endtime1'],
			   $rsrc['starttime2'],
			   $rsrc['endtime2']);
    }
  else
    {
      $tmp[] = $mod->Lang('resource_summary_2',$mod->Lang($rsrc['res_interval']));
      $tmp[] = $mod->Lang('resource_summary_4',
			   $rsrc['checkintime'],
			   $rsrc['checkouttime']);
    }

  return implode(', ',$tmp);
}


function availability_AdjustReservationPartInterval(&$mod,$rsrc_id,$start_ut,$end_ut,$meridian)
{
  $row = $mod->GetResource($rsrc_id);
  list($start_time,$end_time) = availability_AdjustReservationPartInterval2($mod,$row,$start_ut,$end_ut,$meridian);
  $part = array();
  $part['rsrc_id'] = $rsrc_id;
  $part['start_date'] = $start_time;
  $part['end_date'] = $end_time;
  $part['price'] = 0;

  // done
  return $part;
}


function availability_GetAdminCCList(&$mod)
{
  $db = cmsms()->GetDb();
  $tmp = $mod->GetPreference('reservation_confirm_email_cc');
  $origlist = explode(',',$tmp);
  $uidlist = array();
  foreach( $origlist as $one )
    {
      if( $one < 0 )
	{
	  $query = 'SELECT user_id FROM '.cms_db_prefix().'user_groups 
                     WHERE group_id = ?';
	  $tmp = $db->GetArray($query,array($one * -1));
	  $tmp = cge_array::extract_field($tmp,'user_id');
	  $uidlist = array_merge($uidlist,$tmp);
	}
      else
	{
	  $uidlist[] = $one;
	}
    }

  $uidlist = array_unique($uidlist);
  $ccemaillist = array();
  if( count($uidlist) > 0 )
    {
      $userops = cmsms()->GetUserOperations();
      $allusers = $userops->LoadUsers();
      foreach( $uidlist as $uid )
	{
	  for( $i = 0; $i < count($allusers); $i++ )
	    {
	      if( $allusers[$i]->active == 1 &&
		  $allusers[$i]->id == $uid && 
		  !empty($allusers[$i]->email) )
		{
		  $ccemaillist[] = $allusers[$i]->email;
		}
	    }
	}
      $ccemallist = array_unique($ccemaillist);
    }

  if( count($ccemaillist) )
    {
      return $ccemaillist;
    }
  return false;
}


function availability_SendReservationDetailsEmail(&$mod,$resv_id,$template,$logit,&$error)
{
  $gCms = cmsms();
  $db = $gCms->GetDb();
  $smarty = $gCms->GetSmarty();
  $feu = $mod->GetModuleInstance('FrontEndUsers');

  $resv_obj = reservation_ops::load_by_id($resv_id);
  if( !is_object($resv_obj) )
    {
      $error = $mod->DisplayErrorMessage($mod->Lang('error_notfound'));
      return false;
    }

  $reservation = $resv_obj->to_array();
  $query = 'SELECT DISTINCT rsrc_id FROM '.AVAILABILITY_TABLE_RESERVATION_PARTS.'
            WHERE resv_id = ?';
  $tmp = $db->GetCol($query,array($resv_id));
  $query = 'SELECT * FROM '.AVAILABILITY_TABLE_RSRCS.'
           WHERE active = 1
             AND id IN ('.implode(',',$tmp).')';
  $tmp = $db->GetArray($query);
  $reservation['resources'] = cge_array::to_hash($tmp,'id');
  $custid = $reservation['customer_id'];
  $userobj = availability_GetUserObject($mod,$custid);

  // Gather email addresses
  if( empty($userobj->email) )
    {
      $error = $mod->Lang('error_no_customer_email');
      return false;
    }
  $ccemaillist = $mod->GetAdminCCList();

  //
  // Give data to smarty
  //
  $detailpage = $mod->GetPreference('reservation_default_page',-1);
  if( $detailpage == -1 )
    {
      $contentops = $gCms->GetContentOperations();
      $detailpage = $contentops->GetDefaultContent();
    }
  $pretty_url = $mod->GetResourceType()."/viewresvn/$detailpage/{$reservation['id']}";
  $smarty->assign('reservation',$reservation);
  $smarty->assign('resv_obj',$resv_obj);
  if( count($reservation['notes']) )
    {
      $smarty->assign('notes',$reservation['notes']);
    }
  $smarty->assign('userinfo',$userobj);
  $smarty->assign('edit_url',
		  $mod->CreateURL('cntnt01','addresvn',$detailpage,
				   array('resv_id'=>$reservation['id']),
				   false,$pretty_url));
  $smarty->assign('email_url',
		  $mod->CreateURL('cntnt01','sendemail',$detailpage,
				  array('resv_id'=>$reservation['id'])));
  
  // Process the template
  $body = $mod->ProcessTemplateFromDatabase('resvncfmemail_'.$template);
  $subject = $mod->GetPreference('reservation_confirm_email_subject',
				$mod->Lang('dflt_reservation_confirm_email_subject'));
  $subject = $mod->ProcessTemplateFromData($subject);
  $attachment = $mod->GetPreference('reservation_confirm_email_attachment');

  $mailer =& $mod->GetModuleInstance('CMSMailer');
  $mailer->reset();
  $mailer->SetSubject($subject);
  $mailer->SetBody($body);
  $mailer->IsHTML(true);
  if( $attachment != -1 && !empty($attachment) )
    {
      $config = $gCms->GetConfig();
      $fn = cms_join_path($config['uploads_path'],$attachment);
      if( @file_exists($fn) )
	{
	  $mailer->AddAttachment($fn,$attachment);
	}
    }
  $mailer->AddAddress($userobj->email);
  if( is_array($ccemaillist) )
    {
      foreach($ccemaillist as $oneaddr)
	{
	  $mailer->AddBCC($oneaddr);
	}
    }
  $mailer->Send();

  if( $logit )
    {
      $now = $db->DbTimeStamp(time());
      $query = 'INSERT INTO '.AVAILABILITY_TABLE_RESERVATION_NOTES."
                 (resv_id, subject, text, create_date, admin_note)
                VALUES (?,?,?,$now,1)";
      $dbr = $db->Execute($query,array($reservation['id'],$subject,$body));
      if( !$dbr )
	{
	  echo "FATAL DB ERROR:<br/>";
	  echo "&nbsp;&nbsp;QUERY: ".$db->sql."<br/>";
	  echo "&nbsp;&nbsp;ERROR: ".$db->ErrorMsg()."<br/>";
	}
    }

  return true;
}


function availability_GetReservationAttribute(&$mod,$resvn_id,$attrib,$dflt)
{
  $gCms = cmsms();
  $db = $gCms->GetDb();
  
  if( empty($attrib) ) return $dflt;
  $query = 'SELECT value FROM '.AVAILABILITY_TABLE_RESERVATION_ATTRIBS.'
             WHERE resv_id = ? AND name = ?';
  $tmp = $db->GetOne($query,array($resvn_id,$attrib));
  if( !$tmp ) return $dflt;
  return $tmp;
}


function availability_SetReservationAttribute(&$mod,$resvn_id,$attrib,$value)
{
  $gCms = cmsms();
  $db = $gCms->GetDb();
  
  if( empty($attrib) ) return false;
  $query = 'SELECT id FROM '.AVAILABILITY_TABLE_RESERVATION_ATTRIBS.'
             WHERE resvn_id = ? AND name = ?';
  $id = $db->GetOne($query,array($resvn_id,$attrib));
  if( $id )
    {
      // update
      $query = 'UPDATE '.AVAILABILITY_TABLE_RESERVATION_ATTRIBS.'
                  SET value = ? WHERE id = ?';
      $db->Execute($query,array($value,$id));
    }
  else
    {
      // insert
      $query = 'INSERT INTO '.AVAILABILITY_TABLE_RESERVATION_ATTRIBS.'
                  (resv_id,name,value)
                VALUES (?,?,?)';
      $db->Execute($query,array($resv_id,$attrib,$value));
    }
  return true;
}


function &availability_GetUserObject(&$mod,$custid)
{
  $feu =& $mod->GetModuleInstance('FrontEndUsers');

  $data = array();
  $tmp = $feu->GetUserProperties($custid);
  $userprops = cge_array::to_hash($tmp,'title');
  $membergroups = $feu->GetMemberGroupsArray($custid);
  $properties = array();
  foreach( $membergroups as $onegroup )
    {
      $proprelations = $feu->GetGroupPropertyRelations($onegroup['groupid']);
      $properties = RRUtils::array_merge_by_name_required($properties,$proprelations);
      uasort( $properties, array('RRUtils','compare_elements_by_sortorder_key') );
    }
  foreach( $properties as $oneprop )
    {
      $defn = $feu->GetPropertyDefn( $oneprop['name'] );
      $obj = new StdClass();
      $obj->name = $defn['name'];
      $obj->prompt = $defn['prompt'];
      if( isset($userprops[$oneprop['name']]) )
	$obj->value = $userprops[$oneprop['name']]['data'];
      $data[$oneprop['name']] = $obj;
    }
  $userobj = new StdClass();
  $userobj->username = $feu->GetUserName($custid);
  $userobj->uid = $custid;
  $userobj->email = $feu->GetEmail($custid);
  $userobj->properties = $data;

  $gCms = cmsms();
  $smarty = $gCms->GetSmarty();
  $smarty->assign('username',$userobj->username);
  $smarty->assign('_email_address_',$userobj->email);
  $smarty->assign('userid',$custid);

  $str = $mod->ProcessTemplateFromData(
	 $mod->GetPreference('customer_display_template','{$username}'));
  $userobj->display = $str;
  
  return $userobj;
}


function availability_CalcNumSlots($resources = '',$starttime = '',$endtime = '',$include_events = 1)
{
  $gCms = cmsms();
  $db = $gCms->GetDb();
  $rsrc_data = '';

  if( is_array($resources) )
    {
      // list of resources
      $query = 'SELECT * FROM '.AVAILABILITY_TABLE_RSRCS.'
                 WHERE id IN ('.implode(',',$resources).')';
      $rsrc_data = $db->GetArray($query);
      if( !$rsrc_data ) return FALSE;
      
    }
  else if( $resources != '' )
    {
      // single resource
      $query = 'SELECT * FROM '.AVAILABILITY_TABLE_RSRCS.'
                 WHERE id = ?';
      $rsrc = $db->GetRow($query);
      if( !$rsrc ) return FALSE;

      $rsrc_data = array();
      $rsrc_data[] = $rsrc;
    }
  else
    {
      // all resources.
      $query = 'SELECT * FROM '.AVAILABILITY_TABLE_RSRC;
      $rsrc_data = $db->GetArray($query);
      if( !$rsrc_data ) return FALSE;
    }

  if( $include_events )
    {
      // get the list of resource ids
      $rsrc_ids = cge_array::extract_field($rsrc_data,'id');

      // get the list of non bookable events for those resourses.
      // between those dates.
    }

  // calculate the number of days
  if( $startdate = '' )
    {
      // calculate first day of month.
      $startdate = mktime(0,0,0,date('n'),1,date('Y'));
    }
  if( $enddate = '' )
    {
      // calculate last day of month.
      $days_in_month = array(31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
      $days_in_month[1] += date('L');
      $enddate = mktime(23,59,55,date('n'),$days_in_month[date('n')],date('Y'));
    }
  if( $enddate < $startdate )
    {
      swap($startdate,$enddate);
    }
  

  // calculate it up.
  $nslots = 0;
  foreach( $rsrc_data as $rsrc )
    {
      switch( $rsrc['res_interval'] )
	{
	case 'daily':
	  $nslots += 1;

	case 'half_day':
	  $nslots += 2;

	case 'hourly':
	  $nslots += (int)((strtotime($tmp2['endtime1']) - strtotime($tmp2['starttime1'])) / 3600);
	  $nslots += (int)((strtotime($tmp2['endtime2']) - strtotime($tmp2['starttime2'])) / 3600);
	}
    }
  return $nslots;
}


#
# EOF
#
?>
