<?php

	/*
		This class aim to handle CMS Forms very differently.
		
		Author: Jean-Christophe Cuvelier <cybertotophe@gmail.com>
		Copyrights: Jean-Christophe Cuvelier - Morris & Chapman Belgium - 2010
		Licence: GPL		
	
	*/

class CMSForm	extends	CmsObject
{
	const VERSION = '0.0.6';
	
	protected $module_name;
	protected $id; // The form ID
	protected $returnid;
	protected $action;	
	protected $method = 'get';
	protected $enctype;
	
	protected $widgets = array(); // The form widgets
	protected $hidden_widgets = array(); // Specific widgets shown on the begin;
	
	protected $labels = array(
		'submit' => 'Submit',
		'apply' => 'Apply',
		'cancel' => 'Cancel',
		'next' => 'Next',
		'previous' => 'Previous'
		);
		
	protected $active_buttons = array('cancel','submit');
	protected $form_errors = array(); 
	
	public function __construct($module_name, $id, $action, $returnid)
	{
		$this->module_name = $module_name;
		$this->id = $id;
		$this->returnid = $returnid;
		$this->action = $action;
	}
	
	public function setMultipartForm()
	{
		$this->method = 'post';
		$this->enctype = 'multipart/form-data';
	}
	
	public function process()
	{
		foreach($this->widgets as $widget)
		{
			$widget->process();
		}
	}
	
	public function getHeaders()
	{
		global $gCms;
		if (isset($gCms->modules[$this->module_name]['object']))
		{			
			$html = $gCms->modules[$this->module_name]['object']->CreateFormStart($this->id, $this->action, $this->returnid, $this->method, $this->enctype);
			foreach($this->hidden_widgets as $widget)
			{
				$html .= $widget;
			}			
			return $html;
		}
		return null;
	}
	
	public function getFooters()
	{
		global $gCms;
		if (isset($gCms->modules[$this->module_name]['object']))
		{			
			$html = $gCms->modules[$this->module_name]['object']->CreateFormEnd();
			return $html;
		}
		return null;
	}
	
	public function setLabel($label, $title)
	{
		$this->labels[$label] = $title;
	}
	
	
	public function setButtons($buttons = array())
	{
		if (is_array($buttons))	$this->active_buttons = $buttons;
	}
	
	public function getButtons()
	{
		global $gCms;
		$html = '';
		foreach($this->active_buttons as $button)
		{
			$html .= $gCms->modules[$this->module_name]['object']->CreateInputSubmit($this->id, $button, $this->labels[$button]);
		}
		
		return $html;
	}
	
	public function setWidget($name,$type,$settings = array())
	{
		$widget = new CMSFormWidget(&$this,$this->id,$this->module_name,$name,$type,$settings);
		if ($type == 'hidden')
		{
			$this->hidden_widgets[$name] = $widget;
		}
		elseif ($type == 'file')
		{
			$this->setMultipartForm(); // If we add files, we have to be posted multipart forms...
			$this->widgets[$name] = $widget;
		}
		else
		{
			$this->widgets[$name] = $widget;
		}
	}
	
	public function getWidgets()
	{
		return $this->widgets;
	}
	
	public function getWidget($name)
	{
		if (isset($this->widgets[$name]))
		{
			return $this->widgets[$name];
		}
		elseif (isset($this->hidden_widgets[$name]))
		{
			return $this->hidden_widgets[$name];
		}
		else
		{
			return null;
		}
	}
	
	public function showWidgets($template=null)
	{
		$html = '';
		foreach($this->widgets as $widget)
		{
			$html .= $widget->show($template);
		}
		return $html;
	}
	
	public function hasErrors()
	{
		$has_errors = false;
		
		foreach($this->widgets as $widget)
		{
			if($widget->hasErrors())
			{
				$has_errors = true;
			}
		}
		
		
		foreach($this->hidden_widgets as $widget)
		{
			if($widget->hasErrors())
			{
				$has_errors = true;
			}
		}
		
		if (count($this->form_errors) != 0)
		{
			$has_errors = true;
		}
		
		return $has_errors; 
	}
	
	public function noError()
	{
		return !$this->hasErrors();
	}
	
	public function getErrors()
	{
		return $this->form_errors;
	}
	
	public function showErrors()
	{
		$html = '';
		if(count($this->form_errors) > 0)
		{
			$html .= '<ul class="errors">';
			foreach($this->form_errors as $priority => $errors)
			{
				$html .= '<li><em class="form_error_priority">'.$priority.'</em><ul>';
				foreach($errors as $error)
				{
					$html .= '<li>'.$error.'</li>';
				}				
				$html .= '</ul></li>';
			}
			$html .= '</ul>';
		}
		return $html;
	}
	
	public function setError($message,$priority='default')
	{
		$this->form_errors[$priority][] = $message;
	}
	
	public function isSubmitted()
	{
		if (isset($_REQUEST[$this->id.'submit']))
		{
			return true;
		}
		return false;
	}
	
	public function isApplied()
	{
		if (isset($_REQUEST[$this->id.'apply']))
		{
			return true;
		}
		return false;
	}	
	
	public function isCancelled()
	{
		if (isset($_REQUEST[$this->id.'cancel']))
		{
			return true;
		}
		return false;
	}

	public function isPosted()
	{
		// Add check for next and previous
		if ($this->isSubmitted() || $this->isApplied())
		{
			return true;
		}
		return false;
	}
}

class CMSFormWidget	extends CmsObject
{
	protected $id; // The form ID
	protected $module_name;
	protected $name; // The input name
	protected $type;
	protected $input;
	protected $values = array();
	protected $settings = array();
	protected $form_errors;
	protected $validations = array();
	protected $showned = false;
	protected $form;
	
	public function __construct(&$form, $id, $module_name, $name, $type,$settings=array())
	{
		$this->form = &$form;
		$this->id = $id;
		$this->module_name = $module_name;
		$this->name = $name;
		$this->type = $type;
		$this->settings = $settings;
		$this->initValues();
	}
		
	public function process()
	{
		// Validate
		$valid = true;
		if (isset($this->settings['validators']))
		{
			foreach ($this->settings['validators'] as $validator => $value)
			{
				$validate = new CMSFormValidator(&$this,$validator,$value);
				try
				{
					if($validate->check() === false) $valid = false;
				}
				catch(Exception $e)
				{
					$this->setError($e->getMessage(), 'form error');
				}
				
			}
		}
		
		// Save value
		if( $valid == true
			&&
			((isset($this->settings['object']))
			||
			(isset($this->settings['preference'])))
		)
		{
			$this->saveValues();
		}
	}
		
	public function __toString()
	{
		if ($this->type == 'hidden')
		{
			return $this->getInput();
		}
		else
		{
			$html = '
			<div class="form_widget">
				<div class="form_label"><label for="'.$this->id.$this->name.'">'.$this->getLabel().'</label></div>';						
			if (count($this->form_errors))
			{
				$html .= '<div class="form_errors">' . $this->showErrors() . '</div>';
			}				
			$html .= '
				<div class="form_input">'.$this->getInput().'</div>
			</div>';
			return $html;
		}
	}
	
	public function getName()
	{
		return $this->name;
	}
	
	public function getForm()
	{
		if(!is_object($this->form))
		{
			throw new Exception('An error occured retrieving the form object.');
		}
		return $this->form;
	}
	
	public function show($template = null, $force = false)
	{
		if (!$this->showned || $force)
		{
			$html = '';
			if (!is_null($template))
			{
				$text = str_replace('%LABEL%', $this->getLabel(), $template);
				$text = str_replace('%INPUT%', $this->getInput(), $text);
				$text = str_replace('%ERRORS%', $this->showErrors(), $text);
				$text = str_replace('%TIPS%', $this->getTips(), $text);
				$html .= $text;
			}
			else
			{
				$html .= $this;				
			}		
			$this->showned = true;
			return $html;
		}
		return null;
	}
	
	public function initValues()
	{	
		global $gCms;
		switch($this->type)
		{
			case 'checkbox':
				if(
					!isset($_REQUEST[$this->id.$this->name])
					&&
					(
						isset($_REQUEST[$this->id.'submit'])
						||
						isset($_REQUEST[$this->id.'apply'])
					)
					
				)
				{
					// Case when checkbox is unchecked and form is submitted, we should empty the value
					$this->values[]=array();
					break;
				}
			default:
				if (isset($_REQUEST[$this->id.$this->name]))
				{
					$values = html_entity_decode($_REQUEST[$this->id.$this->name]);
				}
				elseif(isset($this->settings['value']))
				{
					$values = $this->settings['value'];
				}
				elseif(
					isset($this->settings['object']) && !isset($_REQUEST[$this->id.'submit']) && !isset($_REQUEST[$this->id.'apply']))
				{
					$values = $this->fetchValues();
				}
				elseif(isset($this->settings['preference']) && $gCms->modules[$this->module_name]['object']->getPreference($this->settings['preference']) != '')
				{
							$values = explode('|',$gCms->modules[$this->module_name]['object']->getPreference($this->settings['preference']));
				}
				
				if(isset($this->settings['default_value']) && !$this->form->isSubmitted()) 
				{
					if (
						(is_array($values) && count($values) == 0)
						||
						(empty($values))
						)
					{
						$values = $this->settings['default_value'];
					}
					
				}
				
				if (!empty($values))
				{
					if (is_array($values)) 
					{
						$this->values = $values;
					}	
					else
					{
					$this->values[] = $values;
					}
				}
				break;
		}
		
	}
	
	protected function saveValues()
	{
		if(isset($this->settings['object']) && is_object($this->settings['object']) && $this->type != 'file')
		{
			// This do not save the object state, so we have to do it outside the form
			
			if ($this->type == 'date')
			{
				$values = $this->values;
			}
			else
			{
				$values = implode('|', $this->values);
			}		
		
			if (isset($this->settings['set_method']))
			{
				$this->settings['object']->{$this->settings['set_method']}($values);
			}
			else
			{
				if (isset($this->settings['field_name']))
				{
					$name = $this->settings['field_name'];
				}
				else
				{
					$name = $this->name;
				}
				
				if(method_exists($this->settings['object'], 'set'))
				{		
					$this->settings['object']->set($name, $values);
				}
				else
				{
					try
					{
						$this->settings['object']->$name = $values;
					}
					catch(Exception $e)
					{
						die('unable to do');
					}
				}
				
				
			}
		}
		
		if(isset($this->settings['preference']) && !isset($_REQUEST[$this->id.'cancel']))
		{
			// Check if there is no cancel button first because we save the value directly !
			global $gCms;
			$gCms->modules[$this->module_name]['object']->setPreference($this->settings['preference'], implode('|', $this->values));
		}
	}
	
	protected function fetchValues()
	{
		if(isset($this->settings['object']) && is_object($this->settings['object']))
		{
			if (isset($this->settings['get_method']))
			{
				return explode('|',$this->settings['object']->{$this->settings['get_method']}());
			}
			else
			{
				if (isset($this->settings['field_name']))
				{
					$name = $this->settings['field_name'];
				}
				else
				{
					$name = $this->name;
				}
				
				if(method_exists($this->settings['object'], 'get'))
				{
					return explode('|',$this->settings['object']->get($name));
				}
				else
				{
						try
						{
							return explode('|', $this->settings['object']->$name);
						}
						catch(Exception $e)
						{
						//	die('unable to do');
						}
				}
			}			
		}
	}
	
	public function setValues($values)
	{
		if (is_array($values))
		{
			$this->values = $values;
		}
		else
		{
			$this->values = array($values);
		}
	}
	
	public function getValues()
	{
		return $this->values;
	}
	
	public function getStringValue()
	{
		return implode('|',$this->values);
	}
	
	public function isEmptyValues()
	{
		if (is_array($this->values) && count($this->values) == 0)
		{
			return true;
		}
		elseif (is_array($this->values) && count($this->values) == 1 && empty($this->values[0]))
		{
			return true;
		}
		elseif(empty($this->values))
		{
			return true;
		}
		return false;
	}
	
	public function getValuesToString()
	{
		if (is_array($this->values))
		{
			return (string)implode('|', $this->values);
		}
		else
		{
			return (string)$this->values;
		}
	}
	
	public function getLabel()
	{
		if (isset($this->settings['label']))
		{
			return $this->settings['label'];
		}
		// Try to get it from language file
		global $gCms;
		if (isset($gCms->modules[$this->module_name]['object']))
		{
			return $gCms->modules[$this->module_name]['object']->lang('form_'.$this->name);
		}
		return null;
	}
	
	public function getTips()
	{
		if (isset($this->settings['tips']))
		{
			return $this->settings['tips'];
		}
		// Try to get it from language file
		global $gCms;
		if (isset($gCms->modules[$this->module_name]['object']))
		{
			// TODO: This should be shown only it the lang key exists...
			//return $gCms->modules[$this->module_name]['object']->lang('tips_'.$this->name);
		}
		return null;
	}
	
	public function getInput()
	{	
		if (!empty($this->input))
		{
			return $this->input;
		}
		
		global $gCms;
		if (isset($gCms->modules[$this->module_name]['object']))
		{
			switch($this->type)
			{
				case 'text':
					return $gCms->modules[$this->module_name]['object']->CreateInputText($this->id, $this->name, implode('|',$this->values), isset($this->settings['size'])?$this->settings['size']:80);				
				case 'hidden':
					return $gCms->modules[$this->module_name]['object']->CreateInputHidden($this->id, $this->name, implode('|',$this->values));
				case 'select':
					return self::CreateInputSelectList($this->id, $this->name, isset($this->settings['values'])?$this->settings['values']:array(), $this->values, 1, '', false);
				case 'checkbox':
					return $gCms->modules[$this->module_name]['object']->CreateInputCheckbox($this->id, $this->name, '1', (integer)implode('|',$this->values));
				case 'textarea':
					if (isset($this->settings['show_wysiwyg']) && $this->settings['show_wysiwyg'] == true)
					{
						return $gCms->modules[$this->module_name]['object']->CreateTextArea(true, $this->id, implode('|',$this->values), $this->name);						
					}
					else
					{
						return $gCms->modules[$this->module_name]['object']->CreateTextArea(false, $this->id, implode('|',$this->values), $this->name);
					}
				case 'time':
					return self::CreateTimeSelect($this->id,$this->name,$this->values);
				case 'date':
					return self::CreateDateSelect($this->id,$this->name,$this->values, $this->settings);
				case 'pages':
					return $gCms->GetContentOperations()->CreateHierarchyDropdown('', implode('|',$this->values), $this->id . $this->name);
						//$this->getPreference('default_page',)
				case 'static':
					return implode('|',$this->values);
				case 'file':
					return $this->getUploadField();
				case 'password':
					return $gCms->modules[$this->module_name]['object']->CreateInputPassword($this->id, $this->name, implode('|',$this->values));
				default:
					return null;
			}
		}
		return null;
		
	}
	
	public function getUploadField()
	{
		global $gCms;
		$field = $gCms->modules[$this->module_name]['object']->CreateInputFile($this->id, $this->name, '', isset($this->settings['size'])?$this->settings['size']:30);
		$hml = '<span>';
		if (count($this->values))
		{
			$file_url = isset($this->settings['base_url'])?$this->settings['base_url']:'';
			if ((substr($file_url, -1) != '/') && (substr($this->values[0],0,1) != '/')) $file_url .'/';
			$file_url .= $this->values[0];
			
			if (self::isImage($this->values[0]))
			{
				$text = '<img src="'.$file_url.'" />';
			}
			else
			{
				$text = basename($this->values[0]);
			}
			
			$html .= '<span style="display:block; margin-bottom: 7px;"><a href="'.$file_url. '" rel="external" >'. $text .'</a></span> ';
		}
		
		if (isset($this->settings['delete_checkbox']))
		{
			$field .= ' ' .  $gCms->modules[$this->module_name]['object']->CreateInputCheckbox($this->id, $this->settings['delete_checkbox'], '1') . ' ' .  $gCms->modules[$this->module_name]['object']->lang('delete');
		}
		
		$html .= $field . '</span>';
		return $html;
	}
	
	public static function isImage($filename)
	{
		$valid_extensions = array('jpeg','jpg','gif','png');
		if (in_array(self::getFileExtension($filename),$valid_extensions))
		{
			return true;
		}
		return false;
	}
	
	public static function getFileExtension($filename)
	{
		$file = explode('.', $filename);
  		if (count($file) > 1)
	  	{
	  		return end($file);
	  	}
	  	else 
	  	{
	  		return null;
	  	}
	}
	

	// Tools
	
	public static function createDateSelect($id,$name,$values,$settings)
	{
		if (count($values) == 1)
		{
			if (strpos($values[0], '-') !== false) $values = explode('-',$values[0]);
		}
		$start_year = isset($settings['start_year'])?$settings['start_year']:date('Y');
		$number_years = isset($settings['number_years'])?$settings['number_years']:20;
		$end_year = $start_year + $number_years;
		
		$year = self::CreateInputSelectList($id,$name.'[0]',self::CreateNumberList($end_year,$start_year),array($values[0]),1,'',false);
		$month = self::CreateInputSelectList($id,$name.'[1]',self::CreateMonthsList(),array($values[1]),1,'',false);
		$day = self::CreateInputSelectList($id,$name.'[2]',self::CreateNumberList(31,1),array($values[2]),1,'',false);
		if (isset($settings['european_date']))
		{
			return $day . $month . $year;
		}
		return $year . $month . $day;
	}
	
	public static function CreateTimeSelect($id,$name,$values)
	{
		$hours = self::CreateInputSelectList($id,$name.'[0]',self::CreateNumberList(23),array($values[0]),1,'',false);
		$minutes = self::CreateInputSelectList($id,$name.'[1]',self::CreateNumberList(59),array($values[1]),1,'',false);
		$seconds = self::CreateInputSelectList($id,$name.'[2]',self::CreateNumberList(59),array($values[2]),1,'',false);
		return $hours . ' : ' . $minutes . ' : ' . $seconds;
	}

	public static function CreateMonthsList()
	{
		$months = array();
		for ($i = 1; $i <= 12; ++$i) {
			$t = mktime(0, 0, 0, $i, 1, 2000);
			$months[$i] = date('M', $t);
		}
		return $months;
	}

	public static function CreateNumberList($end, $start=0)
	{
		if (($end < 0)||(!is_numeric($end))) $end = 1;
		$list = array();
		for ($i = $start; $i <= $end; $i++)
		{
			$list[$i] = (string)$i;
		}
		return $list;
	}
	
	public static function CreateInputSelectList($id, $name, $items, $selecteditems=array(), $size=3, $addttext='', $multiple = true)
	{
	  $id = cms_htmlentities($id);
	  $name = cms_htmlentities($name);
	  $size = cms_htmlentities($size);
	  $multiple = cms_htmlentities($multiple);
	
		$text = '<select name="'.$id.$name.'"';
		if ($addttext != '')
		{
			$text .= ' ' . $addttext;
		}
		if( $multiple )
		  {
			$text .= ' multiple="multiple" ';
		  }
		  
		  if ($size > 1)
		  {
		  	$text .= ' size="'.$size.'"';
		  }
		
		$text .= '>';
		
		$count = 0;
		foreach ($items as $key=>$value)
		{
		  	if (is_array($value))
		 	{
				$array = array_shift($value);
				$key = implode('|',array_keys($array));
				$value = implode('|', $array);
			}
			
			//$value = cms_htmlentities($value);
			$value = $value;
	
			$text .= '<option value="'.$key.'"';
			if (in_array($key, $selecteditems))
			{
				$text .= ' ' . 'selected="selected"';
			}
			$text .= '>';
			$text .= $value;
			$text .= '</option>';
			$count++;
		}
		$text .= '</select>'."\n";
	
		return $text;
	}
	
	function temp()
	{
		$widget['label'] = $this->lang('form_'.$name);
		$widget['value'] = '';
		
		switch($type)
		{
			case 'text':
				$widget['value'] = isset($params[$name])?$params[$name]:isset($settings['default_value'])?$settings['default_value']:'';
				$widget['input'] = $this->CreateInputText($form_id, $name, $widget['value'], isset($settings['size'])?$settings['size']:80);
				break;
		}
		
		$form[$name] = $widget;
		return $form;
	}
		
	public function hasErrors()
	{
		if (count($this->form_errors) == 0)
		{
			return false;
		}
		return true;
	}
	
	public function noError()
	{
		return !$this->hasErrors();
	}
	
	public function getErrors()
	{
		return $this->form_errors;
	}
	
	public function showErrors()
	{
		$html = '';
		if(count($this->form_errors) > 0)
		{
			$html .= '<ul class="form_widget_errors">';
			foreach($this->form_errors as $priority => $errors)
			{
				$html .= '<li><em class="form_widget_error_priority">'.$priority.'</em><ul>';
				foreach($errors as $error)
				{
					$html .= '<li class="form_widget_error_message">'.$error.'</li>';
				}				
				$html .= '</ul></li>';
			}
			$html .= '</ul>';
		}
		return $html;
	}
	
	public function setError($message,$priority='default')
	{
		$this->form_errors[$priority][] = $message;
	}
	
	public function addValidator($validator, $params = array())
	{
		$this->settings['validators'][$validator] = $params;
	}
	
	public function removeValidator($validator)
	{
		unset($this->settings['validators'][$validator]);
	}
	
}

class CMSFormValidator
{
	protected $widget;
	protected $validator;
	protected $params;
	
	public function __construct($widget,$validator,$params)
	{
		$this->widget = $widget;
		$this->validator = $validator;
		$this->params = $params;
	}
	
	public function check()
	{
		switch($this->validator)
		{
			case 'not_empty':
				return $this->CheckNotEmpty();
			case 'equal_field':
				return $this->CheckEqualField();
			case 'email':
				return $this->CheckEmail();
			case 'unique':
				return $this->CheckUnique();
			default:
				throw new Exception('Unknown form validator "'.$this->validator .'".');
		}
		return true;
	}
	
	protected function CheckUnique()
	{
		$value = implode('|',$this->widget->getValues());
		if (call_user_func($this->params, $value) !== null)
		{
			global $gCms;
			throw new Exception($gCms->modules['CMSForms']['object']->lang('field not unique', $value));
		}
		return true;
	}
	
	protected function CheckNotEmpty()
	{	
		global $gCms;
		$values = $this->widget->getValues();
		if (is_array($values) && count($values) == 0)
		{
			throw new Exception($gCms->modules['CMSForms']['object']->lang('field_cannot_be_empty', $this->widget->getName()));
		}
		elseif(empty($values))
		{
			throw new Exception($gCms->modules['CMSForms']['object']->lang('field_cannot_be_empty', $this->widget->getName()));
		}
		
		return true;
	}
	
	protected function CheckEqualField()
	{
		global $gCms;
			$value1 = serialize($this->widget->getValues());
			
			try
			{
				$value2 = serialize($this->widget->getForm()->getWidget($this->params)->getValues());
			}
			catch(Exception $e)
			{
				throw new Exception($gCms->modules['CMSForms']['object']->lang('unknown field', $this->params));
				return false;
			}
			
			if ($value1 != $value2)
			{
				throw new Exception($gCms->modules['CMSForms']['object']->lang('fields not equal', $this->widget->getName(), $this->params));
			}
			
			return true;
		
	}
	
	protected function CheckEmail()
	{
		$email = implode('|', $this->widget->getValues());
		if (!self::validEmail($email))
		{
			global $gCms;
			throw new Exception($gCms->modules['CMSForms']['object']->lang('invalid email', $email));
		}
		return true;
	}
	// External functions
	/**
	Validate an email address.
	Provide email address (raw input)
	Returns true if the email address has the email 
	address format and the domain exists.
	*/
	public static function validEmail($email)
	{
	   $isValid = true;
	   $atIndex = strrpos($email, "@");
	   if (is_bool($atIndex) && !$atIndex)
	   {
	      $isValid = false;
	   }
	   else
	   {
	      $domain = substr($email, $atIndex+1);
	      $local = substr($email, 0, $atIndex);
	      $localLen = strlen($local);
	      $domainLen = strlen($domain);
	      if ($localLen < 1 || $localLen > 64)
	      {
	         // local part length exceeded
	         $isValid = false;
	      }
	      else if ($domainLen < 1 || $domainLen > 255)
	      {
	         // domain part length exceeded
	         $isValid = false;
	      }
	      else if ($local[0] == '.' || $local[$localLen-1] == '.')
	      {
	         // local part starts or ends with '.'
	         $isValid = false;
	      }
	      else if (preg_match('/\\.\\./', $local))
	      {
	         // local part has two consecutive dots
	         $isValid = false;
	      }
	      else if (!preg_match('/^[A-Za-z0-9\\-\\.]+$/', $domain))
	      {
	         // character not valid in domain part
	         $isValid = false;
	      }
	      else if (preg_match('/\\.\\./', $domain))
	      {
	         // domain part has two consecutive dots
	         $isValid = false;
	      }
	      else if
	(!preg_match('/^(\\\\.|[A-Za-z0-9!#%&`_=\\/$\'*+?^{}|~.-])+$/',
	                 str_replace("\\\\","",$local)))
	      {
	         // character not valid in local part unless 
	         // local part is quoted
	         if (!preg_match('/^"(\\\\"|[^"])+"$/',
	             str_replace("\\\\","",$local)))
	         {
	            $isValid = false;
	         }
	      }
	      if ($isValid && !(checkdnsrr($domain,"MX") || 
	 ↪checkdnsrr($domain,"A")))
	      {
	         // domain not found in DNS
	         $isValid = false;
	      }
	   }
	   return $isValid;
	}
}