<?php
#-------------------------------------------------------------------------------
#
# Module : AdvancedContent (c) 2010 by Georg Busch (georg.busch@gmx.net)
#          a content management tool for CMS Made Simple
#          The projects homepage is dev.cmsmadesimple.org/projects/content2/
#          CMS Made Simple is (c) 2004-2010 by Ted Kulp
#          The projects homepage is: cmsmadesimple.org
# Version: 0.5
# File   : class.filePicker.php
#          this file contains some functions of the TineMCE filepicker
# Purpose: functions of the filepicker
# License: GPL
#
#-------------------------------------------------------------------------------

class AdvancedContentFilePicker {
	
	/**
	 * @access private
	 * @var object
	 */
	var $module;
	
	
	/**
	 * Constructor
	 */
	function AdvancedContentFilePicker(&$module) {
		$this->module = $module;
	}
	
	
	/**
	 * function GetMimeType($file)
	 * not part of the module api
	 * gets the mime type of a file
	 * 
	 * @param string $file - the path to a file to check the mime type
	 *
	 * @return string - the mime-type or file extension
	 */
	function GetFileType($file, $useMime = false) {
		
		$fileType = '';
		if($useMime) {
			if (version_compare(PHP_VERSION, '5.3.0') >= 0 && function_exists('finfo_open')) {
				$finfo    = finfo_open(FILEINFO_MIME_TYPE); 
				$fileType = strtolower(finfo_file($finfo, $file));
				finfo_close($finfo);
			}
			else if(function_exists('mime_content_type')) {
				$fileType = strtolower(mime_content_type($file));
			}
		}
		else {
			$fileType = strtolower(substr(strrchr($file, "."), 1));
		}
		return $fileType;
	}
	
	
	/**
	 * Create a dropdown form element containing a list of files that match certain conditions
	 *
	 * @param string $name - the name of the input field
	 * @param string $dir - the directory to list the files of (may be relative to uploads dir or absolute path)
	 * @param string $selected - the preselected value
	 * @param array|string $excludePrefix - fileprefixes to exclude (array('foo','bar',...) or csv)
	 * @param array|string $includePrefix - fileprefixes to include (array('foo','bar',...) or csv)
	 * @param array|string $excludeSufix - filesufixes to exclude (array('foo','bar',...) or csv)
	 * @param array|string $includeSufix - filesufixes to include (array('foo','bar',...) or csv)
	 * @param array|string $fileExt - filesufixes to include (array('foo','bar',...) or csv)
	 * @param string $type
	 * @param boolean $allowNone - set to false to hide the option 'none'
	 *
	 * @return string - the HTML output of a select element with options
	 */
	function CreateFileDropdown($name, $dir, $selected = '', $excludePrefix = '', $includePrefix = '', 
		$excludeSufix = '', $includeSufix = '', $fileExt = '', $type = '', $allowNone = true, $defaultMode = false) 
	{
		
		$files = $this->GetFiles($dir, $excludePrefix, $includePrefix, 
			$excludeSufix, $includeSufix, $fileExt, $type, true, $defaultMode);
		
		if(empty($files))
			return;
		
		$dropdown = '<select name="'.$name.'" id="'.$name.'_AdvancedContentFilePickerInput"';
		if($type == 'image')
			$dropdown .= ' onchange="toggleImage(\'#' . $name . '_AdvancedContentFilePickerPreviewImage\', jQuery(\'#' . $name . '_AdvancedContentFilePickerInput option:selected:first\').attr(\'thumbnail\'),\'\',this.value)"';
		
		$dropdown .= '>';
		if( $allowNone )
		{
			$dropdown .= '<option value=""';
			if($selected == '')
				$dropdown .= ' selected="selected"';
			$dropdown .= ' thumbnail="">--- '.lang('none').' ---</option>';
		}
		
		foreach( $files as $file )
		{
			$dropdown .= '<option value="'.$file['rel_url'].'"';
			if($file['rel_url'] == $selected)
				$dropdown .= ' selected="selected"';
			if($file['isimage'] && $type == 'image')
				$dropdown .= ' thumbnail="'.(isset($file['thumburl']) && $file['thumburl']!=''?$file['thumburl']:$file['fullurl']).'"';
			$dropdown .= '>'.$file['name'].'</option>';
		}
		$dropdown .= '</select>';
		return $dropdown;
	}
	
	
	/**
	 * Return an array containing a list of files in a directory
	 * performs a non recursive search
	 *
	 * @param string $dir - the directory to list the files of (may be relative to uploads dir or absolute path)
	 * @param array|string $excludePrefix - fileprefixes to exclude (array('foo','bar',...) or csv)
	 * @param array|string $includePrefix - fileprefixes to include (array('foo','bar',...) or csv)
	 * @param array|string $excludeSufix - filesufixes to exclude (array('foo','bar',...) or csv)
	 * @param array|string $includeSufix - filesufixes to include (array('foo','bar',...) or csv)
	 * @param array|string $fileExt - filesufixes to include (array('foo','bar',...) or csv)
	 * @param string $type
	 * @param boolean $excludeDirs - set to true to exclude directories
	 *
	 * @return array
	 */
	function GetFiles($dir, $excludePrefix = '', $includePrefix = '', $excludeSufix = '', 
		$includeSufix = '', $fileExt = '', $type = '', $excludeDirs = false, $defaultMode = false) 
	{
		
		global $gCms;
		$files = Array();
		
		if($gCms->config['debug']) {
			debug_buffer($dir,'AdvancedContent FilePicker: dir (before CleanPath)');
		}
		
		$dir = $this->CleanPath($dir);
		$url = $this->CleanURL($dir);
		
		if($gCms->config['debug']) {
			debug_buffer($dir,'AdvancedContent FilePicker: dir (after CleanPath)');
		}
		
		if(!file_exists($dir))
			return $files;
		
		$filePickerstyle = $this->module->GetPreference('filepickerstyle','both');
		
		if(!is_array($excludePrefix))
			$excludePrefix = $this->CleanArray(explode(',',$excludePrefix));
		
		if(!is_array($includePrefix))
			$includePrefix = $this->CleanArray(explode(',',$includePrefix));
		
		if(!is_array($excludeSufix))
			$excludeSufix = $this->CleanArray(explode(',',$excludeSufix));
		
		if(!is_array($includeSufix))
			$includeSufix = $this->CleanArray(explode(',',$includeSufix));
		
		if(!is_array($fileExt))
			$fileExt = $this->CleanArray(explode(',',$fileExt));
		
		$excludePrefix = $this->CleanArray(array_diff($excludePrefix, $includePrefix));
		$excludeSufix  = $this->CleanArray(array_diff($excludeSufix, $includeSufix));
		
		$d = dir($dir);
		while ($entry = $d->read()) {
			
			if ($entry[0] == '.' || (is_dir($dir.$entry) && $excludeDirs))
				continue;
			
			$skip         = false;
			$includeThumb = false;
			
			$isDir = is_dir($dir.$entry);
			$fExt  = strtolower(substr(strrchr($entry, "."), 1));
			$fName = $entry;
			
			if(!$isDir) {
				foreach($excludePrefix as $str) {
					if(startswith($entry,$str)) {
						$skip = true;
						break;
					}
				}
				if($skip) continue;
				
				foreach($includePrefix as $str) {
					if(startswith($str,'thumb_')) {
						$includeThumb = true;
					}
					if(!startswith($entry,$str)) {
						$skip = true;
						break;
					}
				}
				if($skip) continue;
				
				if($fExt != '') {
					$fExtPos = strripos($entry, $fExt);
					if($fExtPos !== FALSE)
						$fName = substr($entry,0,$fExtPos-1);
				}
				
				foreach($excludeSufix as $str) {
					if(endswith($fName,$str)) {
						$skip = true;
						break;
					}
				}
				if($skip) continue;
				
				foreach($includeSufix as $str) {
					if(!endswith($fName,$str)) {
						$skip = true;
						break;
					}
				}
				if($skip) continue;
			}
			
			if(startswith($entry,'thumb_') && !$this->module->GetPreference('showthumbnailfiles',0) && !$includeThumb) {
				continue;
			}
			$fileType = $this->GetFileType($dir.$entry,$this->module->GetPreference('usemimetype',0));
			if (!$isDir && $type=='image' && !in_array(str_replace('image/','',$fileType),$fileExt)) {
				continue;
			}
			else if(!$isDir && $type!='image' && !in_array($fExt,$fileExt) && !empty($fileExt)) {
				continue;
			}
			
			$file             = array();
			$file['name']     = $entry;
			$file['isimage']  = false;
			if (!$isDir && in_array(str_replace('image/','',$fileType),array('jpg','jpeg','gif','png'))) {
				$file['isimage'] = true;
			}
			$file['fullpath']     = $dir.$entry;
			$file['fullurl']      = $url.$entry;
			
			if(!$defaultMode)
				$file['rel_url']  = trim(str_replace($gCms->config['uploads_url'],'',$file['fullurl']),'/');
			else
				$file['rel_url']  = trim(str_replace($gCms->config['root_url'],'',$file['fullurl']),'/');
			
			$file['ext']          = $fExt;
			$file['isdir']        = $isDir;
			$file['type']         = $fileType;
			$file['last_modifed'] = filemtime($dir.$entry);
			
			if (!$file['isdir']) {
				
				if($file['isimage']) {
					$file['thumbnail'] = '';
					$file['thumburl'] = '';
					if ($filePickerstyle != 'filename') {
						if ($this->module->GetPreference('showthumbnailfiles',0) || $includeThumb) {
							$file['thumbnail'] = $this->GetThumbnail($dir.str_replace('thumb_', '', $entry), $this->module->GetPreference('makethumbnail',0));
							$file['thumburl']  = $this->GetThumbnail($dir.str_replace('thumb_', '', $entry), $this->module->GetPreference('makethumbnail',0),true);
						} else {
							$file['thumbnail'] = $this->GetThumbnail($dir.$entry, $this->module->GetPreference('makethumbnail',0));
							$file['thumburl']  = $this->GetThumbnail($dir.$entry, $this->module->GetPreference('makethumbnail',0),true);
						}
					}
					
					$imgsize = @getimagesize($file['fullpath']);
					if ($imgsize) {
						$file['dimensions'] = $imgsize[0] . ' x ' . $imgsize[1];
					} else {
						$file['dimensions'] = '&nbsp;';
					}
				}
			}
			
			// Debug stuff...
			if($gCms->config['debug']) {
				$file['debug'] = "<br /><strong>Debug stuff...</strong><br />";
				$file['debug'] .= "file-type : $fileType<br />";
				$file['debug'] .= "file extension : $fExt<br />";
				$file['debug'] .= "is image : ".($file['isimage']?'yes':'no')."<br />";
				$file['debug'] .= "full path : ".$dir.$entry."<br />";
				$file['debug'] .= "full url : ".$url.$entry."<br />";
				$file['debug'] .= "relative url (stored value): ".$file['rel_url']."<br />";
				$file['debug'] .= "thumbnail path : ".$dir."thumb_".$entry."<br />";
				$file['debug'] .= "thumbnail url : ".$url."thumb_".$entry."<br />";
				$file['debug'] .= "filename without file extension : $fName<br />";
			}
			
			$file['fileicon'] = $this->GetFileIcon($file);
			
			if (!$file['isdir']) {
				$info = @stat($file['fullpath']);
				if ($info) {
					$file['size'] = $info['size'];
				}
			}
			
			$files[] = $file;
		}
		$d->close();
		usort($files, array($this, 'SortFiles'));
		return $files;
	}
	
	
	/**
	 * function CleanPath($path, $fullPath = true)
	 * not part of the module api
	 * replaces all slashes and dots (even multiple ones) and multiple backslashes with one single DIRECTORY_SEPARATOR to make a clean path
	 * 
	 * @param string $path - the path to clean
	 * @param string $fullPath - true to prepend uploads path on return, false to return only the cleaned $path
	 *
	 * @return string - the clean path
	 */
	function CleanPath($path, $fullPath = true) {
		global $gCms;
		$path = trim(str_replace(array($gCms->config['uploads_path'],$gCms->config['uploads_url']), '', $path),'/'.DIRECTORY_SEPARATOR);
		$path = str_replace(DIRECTORY_SEPARATOR,'/',$path);
		$path = trim(preg_replace('/(\/\.)|(\.\/)|(\/?\.\.\/?)/','/',$path),'./');
		if($fullPath) {
			$path = preg_replace('/[\\/]+/',DIRECTORY_SEPARATOR, '/' . $path . '/');
			$path = $gCms->config['uploads_path'] . $path;
			return $path;
		}
		return trim(preg_replace('/[\\/]+/',DIRECTORY_SEPARATOR, $path) , DIRECTORY_SEPARATOR);
	}
	
	
	/**
	 * function CleanURL($url, $fullUrl = true)
	 * not part of the module api
	 * replaces all multiple slashes, dots and multiple backslashes with one single slash to make a clean url
	 * 
	 * @param string $url - the url to clean
	 * @param string $fullUrl - true to prepend uploads url on return, false to return only the cleaned $url
	 *
	 * @return string - the clean url
	 */
	function CleanURL($url, $fullUrl = true) {
		global $gCms;
		$url = trim(str_replace(array($gCms->config['uploads_path'],$gCms->config['uploads_url']), '', $url),'/'.DIRECTORY_SEPARATOR);
		$url = str_replace(DIRECTORY_SEPARATOR,'/',$url);
		$url = trim(preg_replace('/(\/\.)|(\.\/)|(\/?\.\.\/?)/','/',$url),'./');
		if($fullUrl) {
			$url = preg_replace('/\/+/','/', '/' . str_replace('http://','',$url) . '/');
			$url = $gCms->config['uploads_url'] . $url;
			return $url;
		}
		return trim(preg_replace('/\/+/','/',$url),'/');
	}
	
	
	/**
	 * function GetThumbnailFile($path, $makeThumbnail, $urlOnly)
	 * not part of the module api
	 * path can be relative to the uploads dir or absolute
	 */
	function GetThumbnail($path, $makeThumbnail = true, $urlOnly = false) {
		if($path == '')
			return false;
		
		global $gCms;
		$fileName = basename($path);
		$image    = '';
		$path     = $this->CleanPath(str_replace($fileName,'',$path));
		$url      = $this->CleanUrl(str_replace($fileName,'',$path));
		
		if (!file_exists($path . 'thumb_' . $fileName) || filemtime($path . $fileName) > filemtime($path . 'thumb_' . $fileName)) {
			if($makeThumbnail) {
				if(!$this->HandleFileResizing($path . $fileName, $path . 'thumb_' . $fileName, get_site_preference('thumbnail_width', 96), get_site_preference('thumbnail_height',96),true,false,60))
					return;
			}
			else
				return;
		}
		
		if($urlOnly)
			return $url . 'thumb_' . $fileName;
		else
			return '<img src="' . $url . 'thumb_' . $fileName . '" alt="' . str_replace($gCms->config['root_url'],'',$url) . $fileName . '" title="' . str_replace($gCms->config['root_url'],'',$url) . $fileName . '" />';
	}
	
	
	/**
	 * function GetFileIcon($fileInfo)
	 * not part of the module api
	 * gets the icon for a file
	 * if FileManager is installed it uses FileManager icons
	 * otherwise it uses own icons
	 *
	 * @param array $fileInfo - an array with info about the file (ext, isdir, isimage)
	 * @return string - the HTML ouput of an image
	 */
	function GetFileIcon($fileInfo) {
		
		if($fileManager =& $this->module->GetModuleInstance('FileManager')) {
			return $fileManager->GetFileIcon($fileInfo['ext'], $fileInfo['isdir']);
		}
		else {
			if($fileInfo['isdir'])
				return '<img src="../modules/AdvancedContent/images/dir.gif" title="" alt="" />';
			else if ($fileInfo['isimage'])
				return '<img src="../modules/AdvancedContent/images/images.gif" title="" alt="" />';
			else
				return '<img src="../modules/AdvancedContent/images/fileicon.gif" title="" alt="" />';
		}
	}
	
	
	/**
	 * function GetActionIcon($action)
	 * not part of the module api
	 * gets the icon for the file operation
	 * if FileManager is installed it uses FileManager icons
	 * otherwise it uses own icons
	 *
	 * @param string $action - the file operation (e.g. 'delete')
	 * @return string - the HTML ouput of an image
	 */
	function GetActionIcon($action) {
		
		if($fileManager =& $this->module->GetModuleInstance('FileManager')) {
			return $fileManager->GetActionIcon($action);
		}
		else {
			return '<img src="../modules/AdvancedContent/images/'.$action.'.gif" title="'.$action.'" alt="'.$action.'" />';
		}
	}
	
	
	/**
	 * function IsDirEmpty($dir)
	 * not part of the module api
	 * taken from TinyMCE Module (extracted from filepicker)
	 */
	function IsDirEmpty($dir) {
		$d = dir($this->CleanPath($dir));
		while ($entry = $d->read()) {  
			if ($entry == '.') continue;
			if ($entry == '..') continue;
			return false;
		}
		return true;
	}
	
	
	/**
	 * function SortFiles($file1, $file2)
	 * not part of the module api
	 * taken from TinyMCE Module (extracted from filepicker)
	 */
	function SortFiles($file1, $file2) {
		if ($file1['isdir'] && !$file2['isdir']) return -1;
		if (!$file1['isdir'] && $file2['isdir']) return 1;
		return strnatcasecmp($file1['name'], $file2['name']);
	}
	
	
	/**
	 * function HandleFileResizing($source, $output, $newWidth, $newHeight, $keepAspectRatio = true, $allowUpscaling = false, $quality = 100)
	 * not part of the module api
	 */
	function HandleFileResizing($source, $output, $newWidth, $newHeight, $keepAspectRatio = true, $allowUpscaling = false, $quality = 100) {
		// ToDo: keep tmp dir !!!
		//$source = $this->CleanPath(str_replace(basename($source),'',$source)).basename($source);
		$output = $this->CleanPath(str_replace(basename($output),'',$output)).basename($output);
		$imgData = @getimagesize($source);
		
		if (!$imgData) 
			return false;
		
		$origImage = '';
		switch($imgData['mime']) {
			case 'image/jpeg': $origImage = @imagecreatefromjpeg($source); break;
			case 'image/gif' : $origImage = @imagecreatefromgif($source); break;
			case 'image/png' : $origImage = @imagecreatefrompng($source); break;
			default: return false;
		}
		$origWidth   = @imagesx($origImage);
		$origHeight  = @imagesy($origImage);
		$aspectRatio = $origWidth / $origHeight;
		
		$newWidth  = floor($newWidth);
		$newHeight = floor($newHeight);
		
		if($newWidth <= 0 && $newHeight > 0) { // force height
			$newWidth = $origWidth;
			if($newHeight > $origHeight && !$allowUpscaling) {
				$newHeight = $origHeight;
			}
			if($keepAspectRatio) {
				$newWidth = floor($newHeight * $aspectRatio);
			}
		}
		else if($newHeight <= 0 && $newWidth > 0) { // force width
			$newHeight = $origHeight;
			if($newWidth > $origWidth && !$allowUpscaling) {
				$newWidth = $origWidth;
			}
			if($keepAspectRatio) {
				$newHeight = floor($newWidth / $aspectRatio);
			}
		}
		else if($newHeight > 0 && $newWidth > 0) { // both
			if($newWidth > $origWidth && !$allowUpscaling) {
				$newWidth = $origWidth;
			}
			if($newHeight > $origHeight && !$allowUpscaling) {
				$newHeight = $origHeight;
			}
			$newAspectRatio = $newWidth / $newHeight;
			if($keepAspectRatio) {
				
				if($aspectRatio > 1 && $newAspectRatio < 1) { // landscape to portrait
					$_tmp = floor($newWidth / $aspectRatio);
					if($_tmp > 0 && $_tmp <= $newHeight)
						$newHeight = $_tmp;
				}
				else if($aspectRatio < 1 && $newAspectRatio > 1) { // portrait to landscape
					$_tmp = floor($newHeight * $aspectRatio);
					if($_tmp > 0 && $_tmp <= $newWidth)
						$newWidth = $_tmp;
				}
				else {
					if($newAspectRatio < $aspectRatio) {
						$_tmp = floor($newWidth / $aspectRatio);
						if($_tmp > 0 && $_tmp <= $newHeight)
						$newHeight = $_tmp;
					}
					else if($newAspectRatio > $aspectRatio) {
						$_tmp = floor($newHeight * $aspectRatio);
						if($_tmp > 0 && $_tmp <= $newWidth)
						$newWidth = $_tmp;
					}
				}
			}
		}
		else {
			$newHeight = $origHeight;
			$newWidth  = $origWidth;
		}
		
		if($newWidth < 1)
			$newWidth = 1;
		if($newHeight < 1)
			$newHeight = 1;
		
		$newImage = @imagecreatetruecolor(floor($newWidth), floor($newHeight));
		@imagecopyresampled($newImage, $origImage, 0, 0, 0, 0, $newWidth, $newHeight, $origWidth, $origHeight);
		switch($imgData['mime']) {
			case 'image/jpeg': return @imagejpeg($newImage, $output, $quality);
			case 'image/gif' : return @imagegif($newImage, $output);
			case 'image/png' : return @imagepng($newImage, $output);
			default          : return false;
		}
		//todo: enable format change?
		/*
		$format = 'unchanged';
		if ($format=='unchanged') {
			switch($imgData['mime']) {
				case 'image/jpeg': return @imagejpeg($newImage, $output, $quality);
				case 'image/gif' : return @imagegif($newImage, $output);
				case 'image/png' : return @imagepng($newImage, $output);
				default          : return false;
			}
		} else {
			switch ($format) {
				case 'jpg': return @imagejpeg($newImage, $output, $quality);
				case 'gif': return @imagegif($newImage, $output);
				case 'png': return @imagepng($newImage, $output);
				default   : return;
			}
		}
		*/
	}
	
	
	/**
	 * function ContainsIllegalChars($filename)
	 * not part of the module api
	 * taken from TinyMCE Module
	 */
	function ContainsIllegalChars($filename) {
		if (strpos($filename, '\'') !== false) return true;
		if (strpos($filename, '"' ) !== false) return true;
		if (strpos($filename, '/' ) !== false) return true;
		if (strpos($filename, '\\') !== false) return true;
		if (strpos($filename, '&' ) !== false) return true;
		if (strpos($filename, '\$') !== false) return true;
		if (strpos($filename, '+' ) !== false) return true;
		return false;
	}
	
	
	/**
	 * function IsVarEmpty($var, $trim, $unsetEmptyIndexes)
	 * not part of the module api
	 * checks if a var is empty. id var is an array it recursivley checks all elements
	 *
	 * @param mixed $var - the var to check for empty value(s)
	 * @param boolean $trim - true to trim off spaces
	 * @param boolean $unsetEmptyIndexes - true to delete empty elements from array
	 * @return boolean - true if empty, false if not
	 */
	function IsVarEmpty(&$var, $trim = true, $unsetEmptyIndexes = false) {
		if (is_array($var)) {
			foreach ($var as $k=>$v) {
				if (!$this->IsVarEmpty($v)) {
					return false;
				}
				else {
					if($unsetEmptyIndexes)
						unset($var[$k]);
					return true;
				}
			}
		}
		else if($trim && trim($var) == '') {
			return true;
		}
		else if($var == '') {
			return true;
		}
		return false;
	}
	
	
	/**
	 * function CleanArray($array)
	 * not part of the module api
	 * removes empty elements from an array 
	 * (can be useful when using function explode to create the array of a csv)
	 *
	 * @param array $array - the array to clean up
	 * @return array - an array without empty elements or an empty array
	 */
	function CleanArray($array) {
		if (is_array($array)) {
			foreach ($array as $k=>$v) {
				if ($this->IsVarEmpty($v,true,true)) {
					unset($array[$k]);
				}
				else {
					if(is_array($v)) {
						$v = $this->CleanArray($v);
						if($this->IsVarEmpty($v,true,true))
							unset($array[$k]);
						else
							$array[$k] = $v;
					}
				}
			}
			return $array;
		}
		return array();
	}
}

?>
