<?php
# Bookmarks. A plugin for CMS - CMS Made Simple
# Copyright (c) 2004 by Rob Allen <rob@akrabat.com>
#
# CMS- CMS Made Simple is Copyright (c) Ted Kulp (wishy@users.sf.net)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


class Bookmarks extends CMSModule
{
	var $categories_table_name;
	var $bookmarks_to_categories_table_name;
	var $bookmarks_table_name;
	
	function Bookmarks()
	{
		$this->CMSModule();
		
		$this->categories_table_name = cms_db_prefix() . 'module_bookmarks_categories';
		$this->bookmarks_to_categories_table_name = cms_db_prefix().'module_bookmarks_to_categories';
		$this->bookmarks_table_name = cms_db_prefix().'module_bookmarks';	
	}
	
	function GetName()
	{
		return 'Bookmarks';
	}

	function GetFriendlyName()
	{
		return $this->Lang('friendlyname');
	}
		
	function GetAuthor()
	{
		return 'Rob Allen';
	}
	
	function GetAuthorEmail()
	{
		return 'rob@akrabat.com';
	}

	function IsPluginModule()
	{
		return true;
	}

	function HasAdmin()
	{
		return true;
	}
	
	function GetAdminSection()
	{
		return 'content';
	}

	function GetVersion()
	{
		return '2.0';
	}

	function GetDescription($lang = 'en_US')
	{
		return $this->Lang('moddescription');
	}

	function GetHelp($lang = 'en_US')
	{
		return $this->Lang('help'); 
	}

	function GetChangeLog()
	{
		return $this->Lang('changelog'); 
	}

	function Install()
	{
		$db = $this->cms->db;				/* @var $db ADOConnection */
		$dict = NewDataDictionary($db); 	/* @var $dict ADODB_DataDict */
	
		// create categories table
		$fields = "
			category_id I KEY,
			category_name C(255),
			category_order I
		";
	
		$table_opt_array = array('mysql' => 'TYPE=MyISAM');
		$sql_array = $dict->CreateTableSQL(cms_db_prefix()."module_bookmarks_categories",
		$fields, $table_opt_array );
		$dict->ExecuteSQLArray( $sql_array );
	
		$db->CreateSequence( cms_db_prefix()."module_bookmarks_categories_seq" );
	
		// create bookmarks table
		$fields = "
			bookmark_id I KEY,
			bookmark_title C(255),
			bookmark_url C(255),
			bookmark_summary X,
			bookmark_created_by I,
			bookmark_approved I1 DEFAULT 0,
			bookmark_create_date T,
			bookmark_modified_date T
		";
	
		$table_opt_array = array('mysql' => 'TYPE=MyISAM');
		$sql_array = $dict->CreateTableSQL(cms_db_prefix()."module_bookmarks",
		$fields, $table_opt_array );
		$dict->ExecuteSQLArray( $sql_array );
	
		$db->CreateSequence( cms_db_prefix()."module_bookmarks_seq" );
	
		// create bookmarks_to_categories table
		$fields = "
			category_id I KEY,
			bookmark_id I KEY
		";
	
		$table_opt_array = array('mysql' => 'TYPE=MyISAM');
		$sql_array = $dict->CreateTableSQL(cms_db_prefix()."module_bookmarks_to_categories",
		$fields, $table_opt_array );
		$dict->ExecuteSQLArray( $sql_array );
	
		$this->CreatePermission('Modify Bookmarks', 'Modify Bookmarks');
	}

	function InstallPostMessage()
	{
		return $this->Lang('postinstall'); 
	}

	function Upgrade($oldversion, $newversion)
	{
		// No database changes yet!
	}

	function Uninstall()
	{
		$db = $this->cms->db; /* @var $db ADOConnection */
		$dict = NewDataDictionary($db);
	
		$sqlarray = $dict->DropTableSQL(cms_db_prefix()."module_bookmarks");
		$dict->ExecuteSQLArray($sqlarray);
	
		$sqlarray = $dict->DropTableSQL(cms_db_prefix()."module_bookmarks_to_categories");
		$dict->ExecuteSQLArray($sqlarray);
	
		$sqlarray = $dict->DropTableSQL(cms_db_prefix()."module_bookmarks_categories");
		$dict->ExecuteSQLArray($sqlarray);
	
		$db->DropSequence(cms_db_prefix()."module_bookmarks_seq");
		$db->DropSequence(cms_db_prefix()."module_bookmarks_categories_seq");
		$this->RemovePermission('Modify Bookmarks');
	}

  /*---------------------------------------------------------
   UninstallPostMessage()
   ---------------------------------------------------------*/
	function UninstallPostMessage() {
		return $this->Lang('postuninstall');
	}		

	function SetParameters()
	{
		$this->CreateParameter('email_from', 'userid', $this->lang('help_param_emailfrom'));
		$this->CreateParameter('email_to', 'userid', $this->lang('help_param_emailto'));
		$this->CreateParameter('include_back_button', 'true', $this->lang('help_param_backbutton'));
		$this->CreateParameter('addform', 'true', $this->lang('help_param_addform'));
		$this->CreateParameter('target', '_blank', $this->lang('help_param_target'));
		$this->CreateParameter('summaries', 'true', $this->lang('help_param_summaries'));
		$this->CreateParameter('makerssbutton', 'true', $this->lang('help_param_makerssbutton'));
		$this->CreateParameter('auto_detect_link', 'true', $this->lang('help_param_autodetectlink'));
		$this->CreateParameter('order_by_date', 'true', $this->lang('help_param_orderbydate'));
		$this->CreateParameter('type', 'text', $this->lang('help_param_type'));
		$this->CreateParameter('number', '25', $this->lang('help_param_number'));
		$this->CreateParameter('display_unapproved', 'true', $this->lang('help_param_display_unapproved'));
		$this->CreateParameter('show_category_with_title', 'true', $this->lang('help_param_show_catwithtitle'));
		$this->CreateParameter('category', 'retailers', $this->lang('help_param_category'));
		$this->CreateParameter('columns', '2', $this->lang('help_param_columns'));
	}

	function GetBookmark($bookmark_id, $default_to_approved=0) {
		$db = $this->cms->db; /* @var $db ADOConnection */
		$bookmarks_table_name = cms_db_prefix().'module_bookmarks';
		$bookmarks_to_categories_table_name = cms_db_prefix().'module_bookmarks_to_categories';
	
		$sql = "SELECT * from $bookmarks_table_name WHERE bookmark_id = $bookmark_id";
		$rs = $db->Execute($sql);
		if($rs->RowCount() > 0)
		{
			$result = $rs->FetchRow();
			$result['categories'] = array();
			// now pick up categories
			$sql = "SELECT category_id from $bookmarks_to_categories_table_name WHERE bookmark_id = $bookmark_id";
			$rs = $db->Execute($sql);
			while($row = $rs->FetchRow())
			{
				$result['categories'][] = $row['category_id'];
			}
		}
		else
		{
			// create an empty record
			$result = array();
			$result['bookmark_id'] = -1;
			$result['bookmark_title'] = '';
			$result['bookmark_url'] = '';
			$result['bookmark_summary'] = '';
			$result['bookmark_created_by'] = -1;
			$result['bookmark_approved'] = $default_to_approved;
			$result['bookmark_create_date'] = NULL;
			$result['bookmark_modified_date'] = NULL;
			$result['categories'] = array();
		}
	
		return $result;
	}

		
	function GetCategories($order_by='category_order, category_name')
	{
		$db = $this->cms->db; /* @var $db ADOConnection */
		$categories_table_name = $this->categories_table_name;
		$sql = "SELECT * FROM $categories_table_name";
		if($order_by != '')
	    {
	        $sql .= " ORDER BY $order_by";
	    }
	
	    $result = array();
	    $rs = $db->Execute($sql);
	    if($rs && $rs->RowCount() > 0)
	        $result = $rs->GetArray();
	    return $result;
	}	
	
	function AllowAccess($permission='Modify Bookmarks') {
		$access = $this->CheckPermission($permission);
		if (!$access)  {
			DisplayErrorPage($id, $params, $returnid, $this->Lang('needpermission'));
			
			return false;
		}
		else return true;
	
	}
	
	function DoAction($name, $id, $parameters, $returnid='')
	{
		switch($name)
		{
			case 'default':  // default user front end page
				$this->UserDisplay($id, $parameters, $returnid);
				break;
				
			case 'bookmarks_useradd':
				$this->UserAddBookmark($id, $parameters, $returnid);
				break;
				
			case 'defaultadmin': // default admin page
				if($this->AllowAccess()) {
					
					$this->AdminDisplayDefaultAdminPage($id, $parameters, $returnid);
				}
				break;
				
			case 'admin_categories_update':
				if($this->AllowAccess())
				{
					$this->AdminUpdateCategories($id, $parameters, $returnid);
				}
				break;
				
			case 'admin_add_bookmark':
			case 'admin_edit_bookmark':
				if($this->AllowAccess())
				{
					$this->AdminDisplayAddBookmark($id, $parameters, $returnid);
				}
				break;
				
			case 'admin_manage_categories':
				if($this->AllowAccess())
				{
					$this->AdminDisplayCategories($id, $parameters, $returnid);
				}
				break;
				
			case 'admin_bookmarks_update':
				if($this->AllowAccess())
				{
					$this->AdminUpdateBookmarks($id, $parameters, $returnid);
					$url = $this->CreateLink($id, 'defaultadmin', $returnid, $contents='', $params=array('result'=>'1'), '', true);
					$url = str_replace('&amp;', '&', $url);
					
					redirect($url);
				}
				break;
				
			case 'admin_delete_bookmark':
				if($this->AllowAccess())
				{
					$this->AdminDeleteBookmark($id, $parameters, $returnid);
					$url = $this->CreateLink($id, 'defaultadmin', $returnid, $contents='', $params=array('result'=>'1'), '', true);
					$url = str_replace('&amp;', '&', $url);
					
					redirect($url);
				}
				break;
				
			case 'set_approve_false':
            	if ($this->AllowAccess()) {
					$this->SetStatus($id, $parameters, $return_id, 0);
				}
				break;

			case 'set_approve_true':
            	if ($this->AllowAccess()) {
					$this->SetStatus($id, $parameters, $return_id, 1);
				}
				break;

			case 'editcat':
            	if ($this->AllowAccess()) {
            		$this->EditCat($id, $parameters, $return_id);
            	}
				break;
				
			case 'savecat':
				if ($this->AllowAccess()) {
            		$this->SaveCat($id, $parameters, $return_id);
            	}
				break;
			case 'delcat':
            	if ($this->AllowAccess()) {
            		$this->DeleteCat($id, $parameters, $return_id);
            	}
				break;


			default:
				break;
		}
	}

	/*---------------------------------------------------------
	   DisplayErrorPage($id, $params, $return_id, $message)
	   NOT PART OF THE MODULE API

	   This is an example of a simple method to display
	   error information on the admin side.
	---------------------------------------------------------*/
    function DisplayErrorPage($id, &$params, $returnid, $message='') {
		$this->smarty->assign('title_error', $this->Lang('error'));
		if ($message != '')
			{
			$this->smarty->assign_by_ref('message', $message);
			}

        // Display the populated template
        echo $this->ProcessTemplate('error.tpl');
    }
	
	/*************************************************************
		DisplayAdminNav:
			Show the top nav with the links for managing cats and 
			links.  Changed this to tabs.
	**************************************************************/
	
	function DisplayAdminNav ($id, &$params, $returnid, $catcount, $showtab) {
        global $gCms;

        $this->smarty->assign('tab_headers',$this->StartTabHeaders().
            $this->SetTabHeader('links',$this->Lang('title_links_tab')).
            $this->SetTabHeader('categories',$this->Lang('title_categories_tab'),($catcount=0||$showtab=='cat'?true:false)).
            $this->EndTabHeaders().$this->StartTabContent());
        $this->smarty->assign('end_tab',$this->EndTab());
        $this->smarty->assign('tab_footers',$this->EndTabContent());
        $this->smarty->assign('start_links_tab',$this->StartTab('links'));
        $this->smarty->assign('start_cats_tab',$this->StartTab('categories'));

	} // End DisplayAdminNav
	
	/*************************************************************
		DisplayLinksFilter:
			Show the filter for the manage links.  Ripped from the 
			code to display links. 
			Put into a template.
	**************************************************************/
	function DisplayLinksFilter ($id, &$params, $returnid) {
        global $gCms;
		
		$not_approved_only = get_parameter_value($params, 'not_approved_only', 0, 'Bookmarks-not_approved_only');

		$this->smarty->assign('start_links_filter_form', $this->CreateFormStart($id, 'defaultadmin', $returnid, $method='get', $enctype=''));
		$this->smarty->assign('end_links_filter_form', $this->CreateFormEnd());
		
		$this->smarty->assign('title_filter_by', $this->Lang('filterby'));
		$this->smarty->assign('title_app_status', $this->Lang('approvalstatus'));
		
		$this->smarty->assign('input_filter_by', $this->CreateInputText($id, 'keyword', $keyword, 25));
		$this->smarty->assign('input_app_status', $this->CreateInputDropdown($id, 'not_approved_only', array($this->Lang('anystatus')=>0,$this->Lang('not_approved_only')=>1), $not_approved_only));
		$this->smarty->assign('input_filter_button', $this->CreateInputSubmit($id, 'submit', $this->Lang('sortgo')));
		
	} // End DisplayLinksFilter
	
	
	function AdminDisplayDefaultAdminPage($id, $parameters, $returnid, $message='', $showtab='') {
		
		// We need to display the nav, and pass in whether or not categories is selected.

		$db = &$this->cms->db; /* @var $db ADOConnection */
		$categories = $this->GetCategories();
					
		$this->DisplayAdminNav($id, $parameters, $returnid, count($categories), $showtab);

		// Build up the category tab.
		$this->AdminDisplayCategories($id, $parameters, $returnid);
					
		// then the boomkark tab.
		$this->AdminDisplayManageBookmarks($id, $parameters, $returnid);

        if ($message != "") {
            $this->smarty->assign_by_ref('message', $message);
		}
				
		// render the template.  Doing it here to make sure everything is handled before passing it off
		echo $this->ProcessTemplate('adminpanel.tpl');
	
	}

	function EditCat($id, &$params, $returnid, $message="") {

		// Init the DB
        $db = &$this->cms->db;

		if (isset($params['category_id'])) {
			$query = "SELECT * FROM " . cms_db_prefix() . "module_bookmarks_categories WHERE category_id=?";
			$dbresult = $db->Execute($query,array($params['category_id']));
			if ($dbresult !== false && $row = $dbresult->FetchRow()) {
				$title = $row['category_name'];
				$sort = $row['category_order'];
			}
		} else {
			$title='';

			// To make things easier on the user, we are going to get the max sort order from the db and add 10... this way they dont have to.
			$query = "SELECT max(category_order) max_sort FROM " . cms_db_prefix() . "module_bookmarks_categories";
			$dbresult = $db->Execute($query);
			if ($dbresult !== false && $row = $dbresult->FetchRow()) {
				$sort = $row['max_sort'] + 10;
			} else {
				$sort=10;
			}
		}
		
		$this->smarty->assign('startform', $this->CreateFormStart($id, 'savecat', $returnid));
		$this->smarty->assign('endform', $this->CreateFormEnd());
		$this->smarty->assign('submit', $this->CreateInputSubmit($id, 'submit', 'Submit'));

        $this->smarty->assign('title_name',$this->Lang('form_title'));
        $this->smarty->assign('title_sort',$this->Lang('form_sort'));
		
        $this->smarty->assign('input_name',$this->CreateInputText($id, 'title', $title, 50));
        $this->smarty->assign('input_sort',$this->CreateInputText($id, 'sort', $sort, 10));

		$this->smarty->assign ('cancel',$this->CreateInputSubmit ($id, 'cancel','Cancel'));
		
        if (isset($params['category_id']))
            {
            $this->smarty->assign('hidden',$this->CreateInputHidden($id,'category_id',$params['category_id']).
                    $this->CreateInputHidden($id,'start',$params['start']));
            }
        if($message != '') {
            $this->smarty->assign('message',$message);
		}

        echo $this->ProcessTemplate('editcat.tpl');

	}
	
	function SaveCat($id, &$params, $returnid) {
		$db = &$this->cms->db;

    	if (isset($params['category_id'])) {
    	   	// updating
            $query = 'UPDATE '. cms_db_prefix().
				'module_bookmarks_categories set category_name=?, category_oder=? WHERE category_id=?';
            $dbresult = $db->Execute($query,array(
                    $params['title'],$params['sort'],$params['category_id']));
            if ($dbresult === false) {
                return $this->DisplayErrorPage($id, $params, $returnid, $db->ErrorMsg());
			}
            $term_id = $params['category_id'];
			$message = $this->Lang('catedited');
		} elseif (isset ($params['cancel'])) {
			// This is a cancel handling... Go back to menu.
			return $this->AdminDisplayDefaultAdminPage($id, $params, $returnid, $message, 'cat');
		} else {
    	   	// new record
    	   	$term_id = $db->GenID(cms_db_prefix().'module_bookmarks_categories_seq');
			// Status and Sort order will be added at a later date.  As the program progresses.
			$query = 'INSERT INTO '. cms_db_prefix(). 'module_bookmarks_categories (category_id, category_name, category_order) VALUES (?,?,?)';

			$dbresult = $db->Execute($query,array($term_id,$params['title'],$params['sort']));
			$message = $this->Lang('catadded');

            if ($dbresult === false) {
                return $this->DisplayErrorPage($id, $params, $returnid, $db->ErrorMsg());
			}
        }
		
      
        return $this->AdminDisplayDefaultAdminPage($id, $params, $returnid, $message, 'cat');
	}
	
	function DeleteCat($id, &$params, $returnid) {
		$db = &$this->cms->db;
    	if (isset($params['category_id'])) {
			// Delete the category
            $query = 'DELETE FROM '. cms_db_prefix().'module_bookmarks_categories WHERE category_id=?';
            $dbresult = $db->Execute($query,array($params['category_id']));
            if ($dbresult === false) {
                return $this->DisplayErrorPage($id, $params, $returnid, $db->ErrorMsg());
            }
			
			// Delete all the links to the category
            $query = 'DELETE FROM '. cms_db_prefix().'module_bookmarks_to_categories WHERE category_id=?';
            $dbresult = $db->Execute($query,array($params['category_id']));
            if ($dbresult === false) {
                return $this->DisplayErrorPage($id, $params, $returnid, $db->ErrorMsg());
            }
    	}
        return $this->AdminDisplayDefaultAdminPage($id, $params, $returnid, $this->Lang('catdeleted'), 'cat');
    }
		
	function AdminDisplayAddBookmark($id, $params, $returnid) {
			
		$db = &$this->cms->db;

		$cats = array();
		$membercats=array();
		$status = array();

		$status[$this->Lang('status_notapproved')] = 0;
		$status[$this->Lang('status_approved')] = 1;
		
		$query = "SELECT category_id, category_name FROM " . cms_db_prefix() . "module_bookmarks_categories";
		$dbresult = $db->Execute($query);
		
		while ($dbresult !== false && $row = $dbresult->FetchRow()) {
			$cats[$row['category_name']] = $row['category_id'];
		}
		
		if (isset($params['bookmark_id'])) {
			$query = "SELECT * FROM " . cms_db_prefix() . "module_bookmarks WHERE bookmark_id=?";
			$dbresult = $db->Execute($query,array($params['bookmark_id']));
			
			if ($dbresult !== false && $row = $dbresult->FetchRow()) {
				$title = $row['bookmark_title'];
				$descript = $row['bookmark_summary'];
				$url = $row['bookmark_url'];
				if ($row['bookmark_approved'] == 1) {
					$rowstatus = 1;
				} else {
					$rowstatus = 0;
				}
			}
			$query = "SELECT category_id FROM " . cms_db_prefix() . "module_bookmarks_to_categories WHERE bookmark_id=?";
			$dbresult = $db->Execute($query,array($params['bookmark_id']));
			while ($dbresult !== false && $row = $dbresult->FetchRow()) {
				array_push($membercats, $row['category_id']);
			}

		} else {
			$title='';
			$descript='';
			$url='';
			$rowstatus = 1;
		}
	
		$this->smarty->assign('startform', $this->CreateFormStart($id, 'admin_bookmarks_update', $returnid));
		$this->smarty->assign('endform', $this->CreateFormEnd());
		$this->smarty->assign('submit', $this->CreateInputSubmit($id, 'submit', 'Submit'));

        $this->smarty->assign('title_field_title',$this->Lang('form_title'));
        $this->smarty->assign('title_field_descript',$this->Lang('form_descript'));
        $this->smarty->assign('title_field_cats',$this->Lang('form_cats'));
        $this->smarty->assign('title_field_url',$this->Lang('form_url'));
        $this->smarty->assign('title_field_status',$this->Lang('form_status'));

        $this->smarty->assign('input_field_title',$this->CreateInputText($id, 'title', $title, 50));
        $this->smarty->assign('input_field_url',$this->CreateInputText($id, 'url', $url, 50));
                
		$this->smarty->assign('input_field_descript',$this->CreateTextArea(false, $id, $descript, 'descript'));
        $this->smarty->assign('input_field_cats',$this->CreateInputSelectList($id, 'cats[]', $cats, $membercats, 5));
        $this->smarty->assign('input_field_status',$this->CreateInputDropdown($id, 'status', $status, $rowstatus));

		$this->smarty->assign ('cancel',$this->CreateInputSubmit ($id, 'cancel','Cancel'));
		
        if (isset($params['bookmark_id'])) {
            $this->smarty->assign('hidden',$this->CreateInputHidden($id,'bookmark_id',$params['bookmark_id']));
		}
        if($message != '') {
            $this->smarty->assign('message',$message);
		}

        echo $this->ProcessTemplate('editlinks.tpl');
	}
	

	function AdminDisplayCategories($id, $parameters, $returnid)
	{
    	global $gCms;
	
        $entryarray = array();

		// Init the DB
        $db = &$this->cms->db;

		// pull the initial query for the cats
        $query = "SELECT category_id, category_name FROM ".cms_db_prefix().'module_bookmarks_categories'.
            " ORDER by category_order, category_name";
        $dbresult =  $db->Execute($query);

        $rowclass = 'row1';

        while ($dbresult !== false && $row = $dbresult->FetchRow()) {
			
			$onerow = new stdClass();

			$onerow->id = $row['category_id'];
			$onerow->name = $this->CreateLink($id, 'editcat', $returnid, $row['category_name'], array('category_id'=>$row['category_id']));
			$onerow->rowclass = $rowclass;
		   
			$onerow->editlink = $this->CreateLink($id, 'editcat', $returnid,
				$gCms->variables['admintheme']->DisplayImage('icons/system/edit.gif',$this->Lang('edit'),'','','systemicon'),
				array('category_id'=>$row['category_id']),'','',true);
	       $onerow->dellink = $this->CreateLink($id, 'delcat', $returnid,
	       		$gCms->variables['admintheme']->DisplayImage('icons/system/delete.gif',$this->Lang('deletecat'),'','','systemicon'),
	       		array('category_id'=>$row['category_id']),$this->Lang('suredeletecat',$row['category_name']));

	       array_push($entryarray, $onerow);

	       ($rowclass=="row1"?$rowclass="row2":$rowclass="row1");
		}
		
        $this->smarty->assign('column_category', $this->Lang('column_category'));
        $this->smarty->assign('column_action', $this->Lang('column_action'));

    	$this->smarty->assign('title_section', $this->Lang('title_cat_list'));

        $this->smarty->assign('addcat', $this->CreateLink($id, 'editcat', $returnid,
            $gCms->variables['admintheme']->DisplayImage('icons/system/newobject.gif',
                $this->Lang('addcat'),'','','systemicon'),
                array('start'=>$startPoint),
                '', false, false, '') .' '.
            $this->CreateLink($id, 'editcat', $returnid, $this->Lang('addcat'),
                array('start'=>$startPoint),
                '',false, false, 'class="pageoptions"'));

        $this->smarty->assign_by_ref('cats', $entryarray);
        $this->smarty->assign('catcount', count($entryarray));

        if ($message != "")
            {
            $this->smarty->assign_by_ref('message', $message);
            }
				
		// lastly assign the no cats message if its needed.
		$this->smarty->assign('nocats', $this->Lang('nocats'));
		
	}
	
	
	function AdminUpdateBookmarks($id, $params, $returnid) {
		$db = &$this->cms->db;

		$user_id = $this->cms->variables['user_id'];		

    	if (isset($params['bookmark_id'])) {
    	   	// updating
            $query = 'UPDATE '. cms_db_prefix().'module_bookmarks SET 
						bookmark_title=?,
						bookmark_summary=?,
						bookmark_url=?,
						bookmark_approved=?,
						bookmark_modified_date=NOW() 
					WHERE bookmark_id=?';
					
            $dbresult = $db->Execute($query,array($params['title'],$params['descript'],$params['url'],$params['status'],$params['bookmark_id']));
			
            if ($dbresult === false) {
                return $this->DisplayErrorPage($id, $params, $returnid, $db->ErrorMsg());
			}
            $link_id = $params['bookmark_id'];
			$message = $this->Lang('bookmarkedited');

		} elseif (isset($params['cancel'])) {
			// This is a cancel handling... Go back to menu.
			return $this->AdminDisplayDefaultAdminPage($id, $params, $returnid, $message, '');
			
		} else {
			echo "here";
    	   	// new record
    	   	$link_id = $db->GenID(cms_db_prefix().'module_bookmarks_seq');
			$query = 'INSERT INTO '. cms_db_prefix().'module_bookmarks 
						(bookmark_id, bookmark_title, bookmark_summary, bookmark_url, bookmark_approved, bookmark_create_date, bookmark_modified_date, bookmark_created_by) 
					VALUES 
						(?,?,?,?,?,NOW(),NOW(),?)';

			$dbresult = $db->Execute($query,array($link_id,$params['title'],$params['descript'],$params['url'],$params['status'],$user_id));
			
            if ($dbresult === false) {
                return $this->DisplayErrorPage($id, $params, $returnid, $db->ErrorMsg());
			}

			$message = $this->Lang('bookmarkadded');
			
		}
        
		if (isset($params['cats'])) {
            if (! is_array($params['cats'])) {
                $params['cats'] = array($params['cats']);
			}
			$query = 'DELETE FROM '. cms_db_prefix().'module_bookmarks_to_categories where bookmark_id=?';
			$dbresult = $db->Execute($query,array($link_id));
			
            if ($dbresult === false) {
                return $this->DisplayErrorPage($id, $params, $returnid,$db->ErrorMsg());
			}
            
			foreach ($params['cats'] as $thisCat) {
                $query = 'INSERT INTO '. cms_db_prefix().'module_bookmarks_to_categories (bookmark_id, category_id) VALUES (?,?)';
                $dbresult = $db->Execute($query,array($link_id, $thisCat));
                if ($dbresult === false) {
                    return $this->DisplayErrorPage($id, $params, $returnid,$db->ErrorMsg());
				}
			}
		}	
		return $this->AdminDisplayDefaultAdminPage($id, $params, $returnid, $message, 'bookmark');
	}
	
	function AdminDeleteBookmark($id, $parameters, $returnid)
	{
		$db = $this->cms->db; /* @var $db ADOConnection */
		$bookmarks_table_name = $this->bookmarks_table_name;
		$bookmarks_to_categories_table_name = $this->bookmarks_to_categories_table_name;
	
		$bookmark_id = get_parameter_value($parameters, 'bookmark_id', -1);
	
		// delete current bookmarks_to_categories records for this bookmark
		$sql = "DELETE FROM $bookmarks_to_categories_table_name WHERE bookmark_id = $bookmark_id";
		$db->Execute($sql);
	
		// delete this bookmark
		$sql = "DELETE FROM $bookmarks_table_name WHERE bookmark_id = $bookmark_id";
		$db->Execute($sql);
	}

	function SetStatus($id, $params, $returnid, $newstatus) {
		$db = &$this->cms->db;

		$query = 'UPDATE '. cms_db_prefix(). 
			'module_bookmarks set bookmark_approved=? WHERE bookmark_id=?';
		
		$dbresult = $db->Execute($query,array($newstatus,$params['bookmark_id']));
		if ($dbresult === false) {
		echo "here";
			return $this->DisplayErrorPage($id, $params, $returnid, $db->ErrorMsg());
		}
		
		if ($newstatus == 0) {
			$status = $this->Lang('set_status_notapproved');
		} else {
			$status = $this->Lang('set_status_approved');
		}
		return $this->AdminDisplayDefaultAdminPage($id, $params, $returnid );
		
	}
	
	function AdminDisplayManageBookmarks($id, $parameters, $returnid, $message='') {
		global $gCms;

		// Array Inits for the display... Please if theres a better way to do this... someone tell me... 
		
		$categories = array();  // For the categories
		$entryarray = array();
		$catentry = array($entryarray);  // To attach the rows of the links to something we can parse in the template... Like i said... if you know an easier way show me please :)
		$itemcount = array();  // this is for error checking in the template,  so we can show a "no links for cat" error when there are none.

		
		$keyword = get_parameter_value($parameters, 'keyword', '', 'Bookmarks-keyword');
		$not_approved_only = get_parameter_value($parameters, 'not_approved_only', 0, 'Bookmarks-not_approved_only');
	
		// Removed the Links filter to its own function to clean things up a bit.
		$this->DisplayLinksFilter($id, $parameters, $returnid);
		
		$db = &$this->cms->db;

		$db->debug = false;
	
		// Set up the categories pull.  We need to maintain an array to hold the data we are pusing to the template.  This makes things easier in the long run.
		$catsql = "SELECT category_name, category_id FROM ". $this->categories_table_name . " ";
		
		$order_by = get_parameter_value($parameters, 'order_by', '');
		
		if(!empty($order_by))
		{
			$catsql .= "ORDER BY ". $this->bookmarks_table_name . ".modified_date";
		}
		else
		{
			$catsql .= "ORDER BY ". $this->categories_table_name . ".category_order
	    				,". $this->categories_table_name . ".category_name";
		}
		
        $catrs =  $db->Execute($catsql);
		$i = 0;
		while ($catrs !== false && $catrow = $catrs->FetchRow()) {
			
			// now that we have the cats starting, we need to pull in the bookmarks for that cat.  This is rather than the large join and the parse to write html.
			$sql = "SELECT ". $this->bookmarks_table_name . ".* 
			FROM ". $this->bookmarks_table_name ."
			LEFT JOIN ". $this->bookmarks_to_categories_table_name . "
			   ON ". $this->bookmarks_table_name . ".bookmark_id = ". $this->bookmarks_to_categories_table_name . ".bookmark_id ";
			
			$sql .= "WHERE ". $this->bookmarks_table_name . ".bookmark_id=" . $this->bookmarks_to_categories_table_name . ".bookmark_id 
			AND " . $this->bookmarks_to_categories_table_name . ".category_id=" . $catrow['category_id'] . " ";

			if($not_approved_only) {
				$sql .= "AND ". $this->bookmarks_table_name . ".bookmark_approved = 0 ";
			}
			
			if(!empty($keyword)) {
				$sql .= "AND (". $this->bookmarks_table_name . ".bookmark_title LIKE '%$keyword%'
					OR ". $this->bookmarks_table_name . ".bookmark_url LIKE '%$keyword%'
					OR ". $this->bookmarks_table_name . ".bookmark_summary LIKE '%$keyword%') ";
			}
		
			if(!empty($order_by)) {
				$sql .= "ORDER BY ". $this->bookmarks_table_name . ".modified_date";
			} else {
				$sql .= "ORDER BY ". $this->bookmarks_table_name . ".bookmark_title ";
			}
			
			$linkrs = $db->Execute($sql);

			$j = 0;
			$entryarray = array();
			if ($linkrs && $linkrs->RowCount() > 0) {

				while($linkrow = $linkrs->FetchRow()) {
					
					$onerow = new stdClass();

					$onerow->id = $linkrow['bookmark_id'];
					$onerow->namelink = $this->CreateLink($id, 'admin_edit_bookmark', $returnid,$linkrow['bookmark_title'],array('bookmark_id'=>$linkrow['bookmark_id'], '','',true));

					$onerow->name = $linkrow['bookmark_title'];
					$onerow->url = "<a href=\"" . $linkrow['bookmark_url'] . "\" target=\"_blank\">" . $linkrow['bookmark_url'] . "</a>";
					$onerow->rowclass = $rowclass;
					$onerow->definition = strip_tags($linkrow['bookmark_summary']);
					if ($linkrow["bookmark_approved"] == 1) {
						$onerow->active = $this->CreateLink($id, 'set_approve_false', $returnid,
							$gCms->variables['admintheme']->DisplayImage('icons/system/true.gif',$this->Lang('set_approve_false'),'','','systemicon'), 
							array('bookmark_id'=>$linkrow['bookmark_id']));
					} else {
						$onerow->active = $this->CreateLink($id, 'set_approve_true', $returnid,
							$gCms->variables['admintheme']->DisplayImage('icons/system/false.gif',$this->Lang('set_approve_true'),'','','systemicon'), 
							array('bookmark_id'=>$linkrow['bookmark_id']));
					}

					// Edit Link
					$onerow->editlink = $this->CreateLink($id, 'admin_edit_bookmark', $returnid,
						$gCms->variables['admintheme']->DisplayImage('icons/system/edit.gif',$contents=$this->Lang('managebookmarks'),'','','systemicon'),
						array('bookmark_id'=>$linkrow['bookmark_id']));
				
					// Delete Link
					$onerow->dellink = $this->CreateLink($id, 'admin_delete_bookmark', $returnid,
						$gCms->variables['admintheme']->DisplayImage('icons/system/delete.gif',$this->Lang('delete'),'','','systemicon'),
						array('bookmark_id'=>$linkrow['bookmark_id']),$this->Lang('suredelete',$linkrow['bookmark_title']));

					// Push it out to the array for displaying later.
					array_push($entryarray, $onerow);

					// Set the row class so it alternates correctly
					($rowclass=="row1"?$rowclass="row2":$rowclass="row1");
					$j++;
				}
			}
			
			$categories[$i] = $catrow['category_name'];
			$itemcount[$i] = $j;
			$catentry[$i] = $entryarray;
			
			$i++;
			
		}
		
		// Now we have to check to see if there are any links not assigned to a category.
		// When we delete a cat we do not delete the links.  it is possible that we have links without cats.
		$sql = "SELECT ". $this->bookmarks_table_name . ".* FROM ". $this->bookmarks_table_name . " ";
						
		$sql .= "WHERE ". $this->bookmarks_table_name . ".bookmark_id NOT IN (SELECT DISTINCT bookmark_id FROM " . $this->bookmarks_to_categories_table_name . ") ";

		if($not_approved_only) {
			$sql .= "AND ". $this->bookmarks_table_name . ".bookmark_approved = 0 ";
		}
			
		if(!empty($keyword)) {
			$sql .= "AND (". $this->bookmarks_table_name . ".bookmark_title LIKE '%$keyword%'
				OR ". $this->bookmarks_table_name . ".bookmark_url LIKE '%$keyword%'
				OR ". $this->bookmarks_table_name . ".bookmark_summary LIKE '%$keyword%') ";
		}
		
		if(!empty($order_by)) {
			$sql .= "ORDER BY ". $this->bookmarks_table_name . ".modified_date";
		} else {
			$sql .= "ORDER BY ". $this->bookmarks_table_name . ".bookmark_title ";
		}

		$linkrs = $db->Execute($sql);

		$j = 0;
		$entryarray = array();
		if ($linkrs && $linkrs->RowCount() > 0) {

			while($linkrow = $linkrs->FetchRow()) {
				$onerow = new stdClass();

				$onerow->id = $linkrow['bookmark_id'];
				$onerow->namelink = $this->CreateLink($id, 'admin_edit_bookmark', $returnid,$linkrow['bookmark_title'],array('bookmark_id'=>$linkrow['bookmark_id'], '','',true));

				$onerow->name = $linkrow['bookmark_title'];
				$onerow->url = "<a href=\"" . $linkrow['bookmark_url'] . "\" target=\"_blank\">" . $linkrow['bookmark_url'] . "</a>";
				$onerow->rowclass = $rowclass;
				$onerow->definition = strip_tags($linkrow['bookmark_summary']);
				if ($linkrow["bookmark_approved"] == 1) {
					$onerow->active = $this->CreateLink($id, 'set_approve_false', $returnid,
						$gCms->variables['admintheme']->DisplayImage('icons/system/true.gif',$this->Lang('set_approve_false'),'','','systemicon'), 
						array('bookmark_id'=>$linkrow['bookmark_id']));
				} else {
					$onerow->active = $this->CreateLink($id, 'set_approve_true', $returnid,
						$gCms->variables['admintheme']->DisplayImage('icons/system/false.gif',$this->Lang('set_approve_true'),'','','systemicon'), 
						array('bookmark_id'=>$linkrow['bookmark_id']));
				}

				// Edit Link
				$onerow->editlink = $this->CreateLink($id, 'admin_edit_bookmark', $returnid,
					$gCms->variables['admintheme']->DisplayImage('icons/system/edit.gif',$contents=$this->Lang('managebookmarks'),'','','systemicon'),
					array('bookmark_id'=>$linkrow['bookmark_id']));
			
				// Delete Link
				$onerow->dellink = $this->CreateLink($id, 'admin_delete_bookmark', $returnid,
					$gCms->variables['admintheme']->DisplayImage('icons/system/delete.gif',$this->Lang('delete'),'','','systemicon'),
					array('bookmark_id'=>$linkrow['bookmark_id']),$this->Lang('suredelete',$linkrow['bookmark_title']));

				// Push it out to the array for displaying later.
				array_push($entryarray, $onerow);

				// Set the row class so it alternates correctly
				($rowclass=="row1"?$rowclass="row2":$rowclass="row1");
				$j++;
			}
			
			// now push this data out to the variables so we can use it :)
			$categories[$i] = $this->Lang('no_category');
			$itemcount[$i] = $j;
			$catentry[$i] = $entryarray;
		}

			
		$this->smarty->assign('categories',$categories);
		$this->smarty->assign('linklist',$catentry);
		$this->smarty->assign('itemcount',$itemcount);
		
		// Assign the included templates, so it knows where to get them... I hope
		$this->smarty->assign('listlinkstpl','module_file_tpl:'.$this->GetName().';listlinks.tpl');
		$this->smarty->assign('listcatstpl','module_file_tpl:'.$this->GetName().';listcats.tpl');
		$this->smarty->assign('linksfilter','module_file_tpl:'.$this->GetName().';linksfilter.tpl');
		
		// Assign other stuff
		$this->smarty->assign('category_head_text',$this->Lang('category_head_text'));
        $this->smarty->assign('column_title', $this->Lang('column_title'));
        $this->smarty->assign('column_url', $this->Lang('column_url'));
        $this->smarty->assign('column_summary', $this->Lang('column_summary'));
        $this->smarty->assign('column_action', $this->Lang('column_action'));
        $this->smarty->assign('column_approve', $this->Lang('column_approve'));

        $this->smarty->assign('addlink', $this->CreateLink($id, 'admin_edit_bookmark', $returnid,
            $gCms->variables['admintheme']->DisplayImage('icons/system/newobject.gif',
                $this->Lang('addlink'),'','','systemicon'),
                array('start'=>$startPoint),
                '', false, false, '') .' '.
            $this->CreateLink($id, 'admin_edit_bookmark', $returnid, $this->Lang('addlink'),
                array('start'=>$startPoint),
                '',false, false, 'class="pageoptions"'));		
				
	}
	
	function UserDisplay($id, $parameters, $returnid)
	{
		// handling of display of {cms_module module=Bookmarks}
		if(isset($parameters['makerssbutton']))
		{
			echo $this->CreateLink($id, $returnid, 'default', '<img border="0" src="images/cms/xml_rss.gif" alt="'.$this->Lang("rssfeed").'" />',
					array('showtemplate'=>'false','type'=>'rss'));
			return;
		}
	
		if(isset($parameters['auto_detect_link']))
		{
			$title = empty($parameters['title']) ? $this->Lang("bookmarks_rss") : $parameters['title'];
			$url = $this->CreateLink($id, $returnid, $this->Lang("default"), '',
					array('showtemplate'=>'false','type'=>'rss'), '', true);
		
			echo "<link rel='alternate' type='application/rss+xml' title='$title' href='$url' />";
			return;
		}
	
		if(isset($parameters["addform"]))
		{
			$this->DisplayAddBookmarkForm($id, $parameters, $returnid);
			return;
		}
	
		$type = get_parameter_value($parameters, 'type', 'text');
		$display_unapproved = get_parameter_value($parameters, display_unapproved, false);
		$show_category_with_title = get_parameter_value($parameters, 'show_category_with_title', true);
		$category = get_parameter_value($parameters, 'category', '');
		$number = get_parameter_value($parameters, 'number', '');
		$order_by_date = get_parameter_value($parameters, 'order_by_date', false);
		if($order_by_date)
		{
			$show_category_with_title = false;
		}
			
		$categories_table_name = $this->categories_table_name;
		$bookmarks_to_categories_table_name = $this->bookmarks_to_categories_table_name;
		$bookmarks_table_name = $this->bookmarks_table_name;
	
		$db = $this->cms->db;
		if($type == "text")
		{
			$sql = "SELECT $bookmarks_to_categories_table_name.category_id
						,$categories_table_name.category_name
						,$bookmarks_table_name.*
				FROM $bookmarks_table_name
				INNER JOIN $bookmarks_to_categories_table_name
				   ON $bookmarks_table_name.bookmark_id = $bookmarks_to_categories_table_name.bookmark_id
				INNER JOIN $categories_table_name
				   ON $bookmarks_to_categories_table_name.category_id = $categories_table_name.category_id
			";
	
			$where = 'WHERE';
			if($display_unapproved == false)
			{
				$sql .= "$where $bookmarks_table_name.bookmark_approved <> 0 ";
				$where = 'AND';
			}
			if($category)
			{
				$cats = explode(',', $category);
				$sql .= $where . ' (';
				$count = 0;
				foreach($cats as $cat)
				{
					$cat = trim($cat);
					if($count != 0)
					{
						$sql .= ' OR ';
					}
					$count++;
					$sql .= "$categories_table_name.category_name LIKE '$cat' ";
				}
				$sql .=	') ';
			}
	
			if($order_by_date)
			{
				$sql .= "ORDER BY $bookmarks_table_name.bookmark_modified_date";
			}
			else
			{
				$sql .= "ORDER BY $categories_table_name.category_order
		    				,$categories_table_name.category_name
		        	        ,$bookmarks_table_name.bookmark_title ";
			}
		}
		else
		{
			$sql = "SELECT DISTINCT $bookmarks_table_name.*
				FROM $bookmarks_table_name
				INNER JOIN $bookmarks_to_categories_table_name
				   ON $bookmarks_table_name.bookmark_id = $bookmarks_to_categories_table_name.bookmark_id
				INNER JOIN $categories_table_name
				   ON $bookmarks_to_categories_table_name.category_id = $categories_table_name.category_id
				";
	
			$where = 'WHERE';
			if($display_unapproved == false)
			{
				$sql .= "$where $bookmarks_table_name.bookmark_approved <> 0 ";
				$where = 'AND';
			}
	
			if($category)
			{
				$cats = explode(',', $category);
				$sql .= $where . ' (';
				$count = 0;
				foreach($cats as $cat)
				{
					$cat = trim($cat);
					if($count != 0)
					{
						$sql .= ' OR ';
					}
					$count++;
					$sql .= "$categories_table_name.category_name LIKE '$cat' ";
				}
				$sql .=	') ';
			}
	
			$sql .= "ORDER BY $bookmarks_table_name.bookmark_modified_date DESC";
		}

		if($number)
		{
			$dbresult = $db->SelectLimit($sql,$number); /* @var $dbresult ADORecordSet */
		}
		else
		{
			$dbresult = $db->Execute( $sql ); /* @var $dbresult ADORecordSet */
		}
	
		if(isset($parameters['assign']))
		{
			// if assigning, just return array of data
			return $dbresult->GetArray();
		}
	
		if ($dbresult && $dbresult->RowCount() > 0)
		{
			if ($type == "rss")
			{
				// find most recent date - first record as we are date_ordered
				$date = $dbresult->fields['bookmark_modified_date'];
				$date = date('r', strtotime($date));
				$title = $this->Lang("friendlyname");
	
				if (isset($parameters[$id."title"]))
					$title = $parameters[$id."title"];
				while(@ob_end_clean());
				header('Content-type: text/xml');
	
				echo "<?xml version='1.0'?>\n";
				echo "<rss version='2.0'>\n";
				echo "   <channel>\n";
				echo "   <title>$title</title>\n";
				echo "   <link>{$cms->config['root_url']}</link>\n";
				echo "   <pubDate>$date</pubDate>\n";
				echo "   <lastBuildDate>$date</lastBuildDate>\n";
				if(isset($parameters["category"]))
					echo "   <description>{$parameters["category"]} ".$this->Lang("friendlyname")."</description>\n";
				else
					echo "   <description>".$this->Lang("friendlyname")."</description>\n";
	
			}
			else
			{
				$number_of_columns = get_parameter_value($parameters, 'columns', 3);
	
				$num_rows = $dbresult->RecordCount();
				$rows_per_column = intval($num_rows / $number_of_columns) + 10; /* 10 is a fudge factor to make it look better! */
	
				$current_category_name = '';
				$columns = array();
				$col_number = 0;
				ob_start();
	
			}
			
			$target = '';
			if(isset($parameters['target']))
			{
				$target = ' target = "' . $parameters["target"] . '" ';
			}
	
			$row_count = 0;
			while( ($row = $dbresult->FetchRow()) )
			{
				$row_count++;
				$category_name = $row['category_name'];
				$bookmark_title = $row['bookmark_title'];
				$bookmark_url = $row['bookmark_url'];
	
				if(strstr($bookmark_url, '//') === false)
				{
					$bookmark_url = 'http://' . $bookmark_url;
				}
	
				if ($type == "rss")
				{
					$bookmark_modified_date = $row['bookmark_modified_date'];
					$bookmark_modified_date_rfc822 = date('r', strtotime($bookmark_modified_date));
					$bookmark_summary = $row['bookmark_summary'];
	
					echo "   <item>\n";
					echo "      <title>$bookmark_title</title>\n";
					echo "      <link>$bookmark_url</link>\n";
					if($bookmark_summary)
						echo "      <description>$bookmark_summary</description>\n";
					echo "      <pubDate>$bookmark_modified_date_rfc822</pubDate>\n";
					echo "   </item>\n";
				}
				else
				{
					$bookmark_summary = empty($parameters["summaries"]) ? '' : "<div class='cms-module-bookmarks-summary'>" . $row['bookmark_summary'] . "</div>";
	
					if($current_category_name != $category_name)
					{
						// new category - can we start a new column?
						$current_category_name = $category_name;
	
						// close the list
						if($row_count != 1)
						{
							echo "\t\t</ul>\n";
							$string = ob_get_contents();
							ob_end_clean();
							$columns[$col_number] .= $string;
							$col_number++;
							if($col_number >= $number_of_columns)
								$col_number = 0;
						ob_start();
						}
	
						// start new column
						if($show_category_with_title)
							echo "\t\t<div class='cms-module-bookmarks-header'>$current_category_name</div>\n";
						echo "\t\t<ul class='cms-module-bookmarks-list'>\n";
					}
	
					echo "\t\t\t<li><a class='bookmark-link' $target href='$bookmark_url'>$bookmark_title</a>$bookmark_summary</li>\n";
				}
			}
	
			if ($type == 'rss')
			{
				echo "   </channel>\n";
				echo "</rss>\n";
				exit;
			}
			else
			{
				// close off final column
				echo "\t\t</ul>\n";
				$string = ob_get_contents();
				ob_end_clean();
				$columns[$col_number] .= $string;
	
				// display in a table
				echo "<!-- Bookmarks Module: number of bookmarks: $num_rows -->\n";
				echo "<table id='cms-module-bookmarks' border='0' cellpadding='0' cellspacing='0'>\n<tr>\n";
	
				foreach($columns as $col)
				{
					echo "<td valign='top'>\n";
					echo $col;
					echo "</td>\n";
				}
				echo "</tr>\n</table>\n";
				echo "<!-- End of Bookmarks Module -->\n";
			}
		}		
	}
	
	function DisplayAddBookmarkForm($id, $parameters, $returnid) {
		$db = &$this->cms->db;
		
		// set up the cats array for use.
		$cats = array();
		$membercats = array();
		$query = "SELECT category_id, category_name FROM " . cms_db_prefix() . "module_bookmarks_categories";
		$dbresult = $db->Execute($query);
		
		while ($dbresult !== false && $row = $dbresult->FetchRow()) {
			$cats[$row['category_name']] = $row['category_id'];
		}

		// Set the default vaules of the forms... always nothing here
		$title='';
		$descript='';
		$url='';
		$hidden = '';
		
		// pull the parameters
		$include_back_button = get_parameter_value($parameters, 'include_back_button', false);
		$email_to = get_parameter_value($parameters, 'email_to', 0);
		$email_from = get_parameter_value($parameters, 'email_from', 0);
	
		if($email_to > 0) {
			$hidden .= $this->CreateInputHidden($id, 'email_to', $email_to);
		}
		if($email_from > 0) {
			$hidden .= $this->CreateInputHidden($id, 'email_from', $email_from);
		}

		// Assign some hidden fields
		$hidden .= $this->CreateInputHidden($id, 'bookmark_id', $bookmark_id);
		$hidden .= $this->CreateInputHidden($id, 'referer', $_SERVER['SCRIPT_NAME']);
		
		$this->smarty->assign('startform', $this->CreateFormStart($id, 'bookmarks_useradd', $returnid));
		$this->smarty->assign('endform', $this->CreateFormEnd());
		$this->smarty->assign('submit', $this->CreateInputSubmit($id, 'submit', $this->Lang("add")));

		$this->smarty->assign('hidden',$hidden);
		
        $this->smarty->assign('title_field_title',$this->Lang('form_title'));
        $this->smarty->assign('title_field_descript',$this->Lang('form_descript'));
        $this->smarty->assign('title_field_cats',$this->Lang('form_cats'));
        $this->smarty->assign('title_field_url',$this->Lang('form_url'));

        $this->smarty->assign('input_field_title',$this->CreateInputText($id, 'bookmark_title', $title, 50));
        $this->smarty->assign('input_field_url',$this->CreateInputText($id, 'bookmark_url', $url, 50));
                
		$this->smarty->assign('input_field_descript',$this->CreateTextArea(false, $id, $descript, 'bookmark_summary','','','','',50,5,''));
        $this->smarty->assign('input_field_cats',$this->CreateInputSelectList($id, 'cats[]', $cats, $membercats, 5));
		
		echo $this->ProcessTemplate('addbookmark.tpl');

	}
	
	function UserAddBookmark($id, $parameters, $returnid)
	{
		$parameters['bookmark_approved']=0;
		$this->UserUpdateBookmarks($id, $parameters, $returnid);
		$email_to = get_parameter_value($parameters, 'email_to', 0);
		if($email_to > 0)
		{
			// get email address to send email to:
			$sql = 'SELECT email FROM ' .cms_db_prefix().'users WHERE user_id=' . $email_to;
			$rs = $this->cms->db->SelectLimit($sql, 1);
			$email_to = '';
			if($rs)
			{
				$email_to = $rs->fields['email'];
			}
			if($email_to != '')
			{
				// get from email address.
				$email_from = get_parameter_value($parameters, 'email_from', 0);
				if($email_from <= 0)
				{
					// none set - pick up first email we can find.
					$email_from = 'bookmarksmodule@example.com';
					$sql = 'SELECT email FROM ' .cms_db_prefix().'users WHERE email IS NOT NULL ORDER BY user_id ASC';
					$rs = $this->cms->db->SelectLimit($sql, 1);
					if($rs)
						$email_from = $rs->fields['email'];
				}
		
				$subject = $this->Lang("new_bookmark_submitted_subject");
				$body = $this->Lang("new_bookmark_submitted_body")."\n\n";
				$body .= $this->Lang("title"). ": {$parameters[$module_id.'title']}\n";
				$body .= $this->Lang("url").": {$parameters[$module_id.'url']}\n";
				$body .= $this->Lang("summary").": {$parameters[$module_id.'summary']}\n";
		
				$headers = $this->Lang("mailheader_from").": ".$this->Lang("mailheader_sender")." <$email_from>\r\n"
		    	    ."Return-Path: ".$this->Lang("mailheader_sender")." <$email_from>\r\n"
			        ."X-Mailer: PHP/" . phpversion();
			        
			    $result = @mail($email_to, $subject, $body, $headers);
			}
		}

		$link = $this->CreateReturnLink($id, $returnid, $this->Lang("return"));
		echo "<p class='cms-module-bookmarks-submitted'>".$this->Lang("thank_you_for_submission")." . $link.</p>";
		
		
	}
	
	
	function UserUpdateBookmarks($id, $params, $returnid) {
		$db = &$this->cms->db;

		$user_id = 1;		

   	   	// new record
		$link_id = $db->GenID(cms_db_prefix().'module_bookmarks_seq');
		$query = 'INSERT INTO '. cms_db_prefix().'module_bookmarks 
					(bookmark_id, bookmark_title, bookmark_summary, bookmark_url, bookmark_approved, bookmark_create_date, bookmark_modified_date, bookmark_created_by) 
				VALUES 
					(?,?,?,?,?,NOW(),NOW(),?)';

		$dbresult = $db->Execute($query,array($link_id,$params['bookmark_title'],$params['bookmark_summary'],$params['bookmark_url'],0,$user_id));
			
		if ($dbresult === false) {
			return $this->DisplayErrorPage($id, $params, $returnid, $db->ErrorMsg());
		}

		if (isset($params['cats'])) {
            if (! is_array($params['cats'])) {
                $params['cats'] = array($params['cats']);
			}
			$query = 'DELETE FROM '. cms_db_prefix().'module_bookmarks_to_categories where bookmark_id=?';
			$dbresult = $db->Execute($query,array($link_id));
			
            if ($dbresult === false) {
                return $this->DisplayErrorPage($id, $params, $returnid,$db->ErrorMsg());
			}
            
			foreach ($params['cats'] as $thisCat) {
                $query = 'INSERT INTO '. cms_db_prefix().'module_bookmarks_to_categories (bookmark_id, category_id) VALUES (?,?)';
                $dbresult = $db->Execute($query,array($link_id, $thisCat));
                if ($dbresult === false) {
                    return $this->DisplayErrorPage($id, $params, $returnid,$db->ErrorMsg());
				}
			}
		}	
		
		//redirect('index.php');
	}	
	
};
?>