<?php
/**
 * class.CSK_CSSKit.php
 *
 * @author David Broz <db-cmsms@elunia.ch>
 * @copyright (c) 2009 by David Broz - {@link http://www.elunia.ch elunia visual media} Zurich, Switzerland
 * @license http://opensource.org/licenses/gpl-2.0.php GNU General Public License Version 2
 * @since 30/04/2009 4:58:44 PM
 * @package CSSKit
 */

if($_SERVER['SCRIPT_FILENAME'] == __FILE__ )
{
	exit();
}


/**
 * CSK_CSSKit
 *
 * This is the main wrapper class for the CSS Kit.
 *
 * @author David Broz <db-cmsms@elunia.ch>
 * @copyright (c) 2009 by David Broz - {@link http://www.elunia.ch elunia visual media} Zurich, Switzerland
 * @license http://opensource.org/licenses/gpl-2.0.php GNU General Public License Version 2
 * @package CSSKit
 */
class CSK_CSSKit
{
	/**
	 * tableName
	 *
	 * @var string
	 */
	const tableName = "css";

	/**
	 * $_storage
	 *
	 * @var MDK_StorageCache
	 */
	protected $_storage;

	/**
	 * $_db
	 *
	 * @var ADOConnection
	 */
	protected $_db;

	/**
	 * $_cskInstance
	 *
	 * @var CSSKit
	 */
	protected $_cskInstance;


	/**
	 * CSK_CSSKit (constructor)
	 *
	 * @param CSSKit $i_cskInstance
	 * @return CSK_CSSKit
	 */
	function CSK_CSSKit($i_cskInstance=null)
	{
		if (!is_null($i_cskInstance))
		{
			$this->setModuleInstance($i_cskInstance);
		}
		$this->setADOConnection();
	}

	/**
	 * setADOConnection
	 *
	 * @param ADOConnection $i_db
	 */
	private function setADOConnection()
	{
		if (!($this->_db instanceof ADOConnection))
		{
			$this->_db= &$this->_cskInstance->cms->db;
		}
	}

	/**
	 * getSQLTableStruct
	 *
	 * Database table structure in ADOdb Dataformat
	 *
	 * @link http://phplens.com/lens/adodb/docs-datadict.htm
	 * @return array
	 */
	static function getSQLTableStruct()
	{
		$table = MDK_StorageCache::getSQLTableStruct(self::tableName);
		return $table;
	}


	/**
	 * setModuleInstance
	 *
	 */
	private function setModuleInstance(&$i_module=null)
	{
		if ($i_module instanceof CSSKit)
		{
			$this->_cskInstance = &$i_module;
		}
		$mod = &$this->_cskInstance;
		if (!($this->_storage instanceof MDK_StorageCache ))
		{
			$this->_storage = $mod->MDK->MDK_GetStorageCache($mod->cms->db,($mod->MDK->MDK_GetModuleTablePrefix($mod)).self::tableName,$mod->GetVersion());
			$this->_storage->checkVersion($mod->GetVersion());
		}
	}

	/**
	 * getStyleSheetLinks
	 *
	 * @param string $name
	 * @param string $media
	 * @return string
	 */
	public function getStyleSheetLinks($name='',$media='')
	{
		$this->setADOConnection();
		if (!empty($name))
		{
			return $this->getSingleStyleSheetLink($name,$media);
		}
		return $this->getMultipleStyleSheetLinks();
	}

	/**
	 * getMultipleStyleSheetLinks
	 *
	 * @return string
	 */
	private function getMultipleStyleSheetLinks()
	{
		$mod = &$this->_cskInstance;
		$pageinfo = $mod->cms->variables['pageinfo'];
		$query = 'SELECT DISTINCT A.css_id,A.media_type,A.css_text,B.assoc_order
                      FROM '.cms_db_prefix().'css A, '.cms_db_prefix().'css_assoc B
                     WHERE A.css_id = B.assoc_css_id
                       AND B.assoc_type = ?
                       AND B.assoc_to_id = ?
                     ORDER BY B.assoc_order';

		$stylesheets = $this->_db->GetArray($query,array('template',$pageinfo->template_id));
		if ($stylesheets)
		{
			$types = array();
			foreach( $stylesheets as $record )
			{
				$mTypes = explode(",",$record['media_type']);
				foreach ($mTypes as $key=>$val)
				{
					if (!isset($types[$val]))
					{
						$types[$val] = "";
					}
					$types[$val] .= $record['css_text'];
				}
			}
			$links = "";
			foreach($types as $key=>$val)
			{
				require_once(CSK_3RDPARTY_LIB."bsd/Minify/CSS/Compressor.php");
				$css_text = Minify_CSS_Compressor::process($val);
				$media_type = $key;
				$hash = $this->getStyleSheetHash($css_text);
				if (!$this->_storage->hashExists($hash,true)){
					$this->_storage->setData($hash,array($css_text),true);
				}
				$links .= $this->getStyleSheetTag($hash,$media_type);
			}
			return $links;
		}
		return '';

	}

	/**
	 * getSingleStyleSheetLink
	 *
	 * @param string $name
	 * @param string $media_type
	 * @return string
	 */
	private function getSingleStyleSheetLink($name,$media_type)
	{
		$query = 'SELECT css_text FROM '.cms_db_prefix().'css WHERE css_name=?;';
		$css = $this->_db->GetOne( $query, array($name));
		$link='';
		if( $css )
		{
			require_once(CSK_3RDPARTY_LIB."bsd/Minify/CSS/Compressor.php");
			$css_text = Minify_CSS_Compressor::process($css['css_text']);
			$hash = $this->getStyleSheetHash($css_text);
			if (!$this->_storage->hashExists($hash,true)){
				$this->_storage->setData($hash,array($css_text),true);
			}
			$link = $this->getStyleSheetTag($hash,$media_type);
		}
		return $link;
	}


	/**
	 * getCSS
	 *
	 * @param string $hash
	 * @return string
	 */
	public function getCSS($hash='')
	{
		if (empty($hash))
		{
			return '';
		}
		$css = $this->_storage->getData($hash,true);
		return isset($css[0]) ? $css[0] : '';
	}


	/**
	 * touchCSS
	 *
	 * @param string $hash
	 */
	public function touchCSS($hash='')
	{
		if(!empty($hash))
		{
			$this->_storage->touchData($hash);
		}
	}

	/**
	 * getExpiryTime
	 * 
	 * returns the expiry time of a combined stylesheet identified by hash.
	 *
	 * @param string $hash
	 * @return int (seconds)
	 */
	public function getExpiryTime($hash='')
	{
		return $this->_storage->getExpiryTime($hash);
	}
	
	/**
	 * getCacheAge
	 *
	 * returns the default storage cache age in seconds.
	 * 
	 * @return int (seconds)
	 */
	public function getCacheAge()
	{
		return $this->_storage->getCacheAge();
	}


	/**
	 * getStyleSheetHash
	 *
	 * @param string $css_text
	 * @return string
	 */
	private function getStyleSheetHash($css_text)
	{
		return $this->_storage->getHash($css_text);
	}


	/**
	 * getStyleSheetTag
	 *
	 * @param string $hash
	 * @param string $media_type
	 * @return string
	 */
	private function getStyleSheetTag($hash,$media_type='')
	{
		$mod =& $this->_cskInstance;
		$url = $mod->cms->config['root_url']."/modules/".$mod->GetName()."/css.php?css=".$hash;
		$media='';
		if (!empty($media_type))
		{
			$media=' media="'.$media_type.'"';
		}
		return '<link rel="stylesheet" type="text/css"'.$media.' href="'.$url.'" />'."\n";
	}
}
?>