<?php
#BEGIN_LICENSE
#-------------------------------------------------------------------------
# Module: SelfRegistration (c) 2008 by Robert Campbell
#         (calguy1000@cmsmadesimple.org)
#  An addon module for CMS Made Simple to allow users to register themselves
#  with a website.
#
# Version: 1.1.5
#
#-------------------------------------------------------------------------
# CMS - CMS Made Simple is (c) 2005 by Ted Kulp (wishy@cmsmadesimple.org)
# This projects homepage is: http://www.cmsmadesimple.org
#
#-------------------------------------------------------------------------
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# However, as a special exception to the GPL, this software is distributed
# as an addon module to CMS Made Simple.  You may not use this software
# in any Non GPL version of CMS Made simple, or in any version of CMS
# Made simple that does not indicate clearly and obviously in its admin
# section that the site was built with CMS Made simple.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
# Or read it online: http://www.gnu.org/licenses/licenses.html#GPL
#
#-------------------------------------------------------------------------
#END_LICENSE
use \MAMS\RRUtils as RRUtils;

if(!isset($gCms))
{
  exit;
}

//
// this file displays the form for registration and handles validating the data
// but does not do registration itself.
//
$error                = $message = NULL;
$eparams              = reg_utils::clean_params($params);
$mams                 = \xt_utils::get_module(MOD_MAMS);
$username_is_email    = $mams->GetPreference('username_is_email');
$nocaptcha            = \xt_param::get_bool($eparams, 'nocaptcha', 0);
$allow_overwrite      = \xt_param::get_bool($eparams, 'allowoverwrite');
$reg_additionalgroups = (int)$this->GetPreference('reg_additionalgroups', 0);

unset($eparams['action'], $eparams['module'], $eparams['returnid']);

// get data passed into the session by the reguser action
// this only occurs if there is an error in that action.
$tmp = $this->session_get('reguser_error');
$this->session_clear('reguser_error');
if($tmp)
{
  $t_params = $this->session_get('signup');
  $this->session_clear('signup');
  if($t_params)
  {
    $eparams = array_merge($eparams, $t_params);
  }
  unset($eparams['submit']);
  $message = $tmp;
  $error   = 1;
}

//if( !isset( $eparams['group'] ) )
//{
//    // this is ugly for the user to see
//    // but at least the admin will be able to figure it out
//    audit('',$this->GetName(),'no group parameter specified for the signup action');
//    $this->_DisplayErrorPage( $id, $eparams, $returnid, $this->Lang('error_insufficientparams'));
//    return;
//}

// yep, all the modules are here, now we convert a single name to an integer id
// or an array of integer ids... then we validate them.
$grpids = reg_utils::expand_group($eparams['group']);
//if( !count($grpids) )
//{
//    $this->_DisplayErrorPage( $id, $eparams, $returnid, $this->Lang('error_novalidgroups'));
//    return;
//}

// now we have group ids... have to get a list of the properties, and property definitions.
$tmp   = $properties = [];
$defns = $mams->GetPropertyDefns();

foreach($grpids as $gid)
{
  $grelns = $mams->GetGroupPropertyRelations($gid);
  if(count($grelns) == 2 && $grelns[0] === FALSE)
  {
    continue;
  }
  $tmp = RRUtils::array_merge_by_name_required($tmp, $grelns);
  uasort($tmp, 'xt_array::compare_elements_by_sortorder_key');
}

foreach($tmp as $one)
{
  $name = $one['name'];
  if(!isset($defns[$name]))
  {
    throw new \LogicException($this->Lang('error_dberror'));
  }
  $one                      = array_merge($one, $defns[$name]);
  $properties[$one['name']] = $one;
}

unset($tmp);


//
// handle submit
//
class MAMSREGValidationError extends \RuntimeException
{
}

if(isset($eparams['submit']))
{
  try
  {
    if(!\xt_utils::valid_form_csrf())
    {
      throw new \RuntimeException($this->Lang('error_security'));
    }
    
    // for field types that have to be 'built' we do that first, so we don't lose their data
    // if another error occurs.
    foreach($properties as $propname => $prop)
    {
      $defn     = $defns[$propname];
      $type     = $defn['type'];
      $required = ($prop['required'] == 2);
      $val      = \xt_utils::get_param($eparams, 'input_' . $propname);
      
      switch($type)
      {
        case 5: // multiselect
          if($required && !$val)
          {
            throw new \RuntimeException($this->Lang('error_requiredfield', $defn['prompt']));
          }
          if($val)
          {
            $val = $eparams['input_' . $propname] = implode(',', $val);
          }
        break;
        
        case 8: // date
          if($required && !isset($eparams['input_' . $propname . 'Month']))
          {
            throw new \RuntimeException($this->Lang('error_requiredfield', $defn['prompt']));
          }
          if(isset($eparams['input_' . $propname . 'Month']))
          {
            $val = $eparams['input_' . $propname] =
              mktime(
                0, 0, 0,
                \xt_param::get_int($eparams, 'input_' . $propname . 'Month'),
                \xt_param::get_int($eparams, 'input_' . $propname . 'Day'),
                \xt_param::get_int($eparams, 'input_' . $propname . 'Year')
              );
            unset($eparams['input_' . $propname . 'Month']);
            unset($eparams['input_' . $propname . 'Day']);
            unset($eparams['input_' . $propname . 'Year']);
          }
        break;
      }
    }
    
    // attempt to register to an absolutely verboten group
    $tmp = $this->GetPreference('noregister_groups');
    if($tmp)
    {
      $verboten = explode(',', $tmp);
      $tmp      = array_intersect($grpids, $verboten);
      
      if(count($tmp))
      {
        throw new \MAMSREGValidationError($this->Lang('error_noregister'));
      }
    }
    
    // get the username and password
    $username_1 = cms_html_entity_decode(\xt_param::get_string($eparams, 'input_username'));
    $username_2 = cms_html_entity_decode(\xt_param::get_string($eparams, 'input_username_again'));
    $password_1 = cms_html_entity_decode(\xt_param::get_string($eparams, 'input_password'));
    $password_2 = cms_html_entity_decode(\xt_param::get_string($eparams, 'input_repeatpassword'));
    
    if(!$username_1)
    {
      throw new \RuntimeException(
        ($username_is_email) ? $this->Lang('error_emptyemail') : $this->Lang('error_emptyusername')
      );
    }
    
    // check the passwords
    if(!$password_1)
    {
      throw new \MAMSREGValidationError($this->Lang('error_invalidpassword'));
    }
    if(!$mams->IsValidPassword($password_1))
    {
      throw new \MAMSREGValidationError($this->Lang('error_invalidpassword'));
    }
    if($password_1 != $password_2)
    {
      throw new \RuntimeException($this->Lang('error_passwordsdontmatch'));
    }
    
    // get an email field, so that we can do some emailing
    $email_field = NULL;
    
    if($username_is_email)
    {
      $email_field = 'input_username';
    }
    else
    {
      foreach($properties as $propname => $defn)
      {
        if($defn['type'] == 2)
        {
          $email_field = 'input_' . $propname;
          break;
        }
      }
    }
    
    if(!$email_field)
    {
      throw new \LogicException($this->Lang('error_noemailaddress'));
    }
    $email = \xt_param::get_string($eparams, $email_field);
    $res   = $mams->IsValidEmailAddress($email, -1, FALSE);
    if(!is_array($res) || $res[0] == FALSE)
    {
      throw new \MAMSREGValidationError($this->Lang('error_invalidemail'));
    }
    $eparams['use_email'] = $email;  // this is used by the reguser action
    
    // check the repeated email field.
    if($this->GetPreference('mamsreg_force_email_twice'))
    {
      if(!isset($eparams[$email_field . '_again']))
      {
        throw new \MAMSREGValidationError($this->Lang('error_nosecondemailaddress'));
      }
      if($eparams[$email_field] != $eparams[$email_field . '_again'])
      {
        throw new \RuntimeException($this->Lang('error_emaildoesnotmatch'));
      }
    }
    
    // check the captcha
    if(!$nocaptcha)
    {
      $captcha = \cms_utils::get_module('Captcha');
      if(is_object($captcha) && !$captcha->CheckCaptcha(\xt_param::get_string($eparams, 'input_captcha')))
      {
        throw new \MAMSREGValidationError($this->Lang('error_captchamismatch'));
      }
    }
    
    $overwrite_uid = NULL;
    if($reg_additionalgroups && $allow_overwrite)
    {
      // allow registration to additional groups
      // make sure that the user is currently logged in.
      $mams_uid = $mams->LoggedInId();
      if($mams_uid < 1)
      {
        throw new \MAMSREGValidationError($this->Lang('error_additionalgroups_notloggedin'));
      }
      
      // make sure that the username he entered matches the one on file.
      $query = new feu_user_query();
      $query->set_pagelimit(1);
      $query->add_and_opt(feu_user_query_opt::MATCH_USERLIST, $mams_uid);
      $rs  = $query->execute();
      $cnt = $rs->get_found_rows();
      if($cnt == 1)
      {
        // only one match found.
        $data = $rs->fields;
        // make sure that the username we are logged in as, matches this form submission.
        if($data['username'] != $username_1)
        {
          throw new \MAMSREGValidationError($this->Lang('error_usernamemismatch'));
        }
        $overwrite_uid = $mams_uid;
      }
    }
    
    
    // check if the username is taken
    if(!$overwrite_uid)
    {
      // make sure that this username is not already used in this module
      if($this->GetTempUserID($username_1))
      {
        throw new \RuntimeException($this->Lang('error_usernametaken_needvalidation'));
      }
      
      // make sure that this username is not already used in FEU
      if($mams->GetUserID($username_1))
      {
        throw new \RuntimeException($this->Lang('error_usernametaken'));
      }
    }
    
    // make sure it's a valid username...
    if(!$mams->IsValidUsername($username_1, ($overwrite_uid < 1) ? TRUE : FALSE))
    {
      throw new \MAMSREGValidationError(
        $username_is_email ? $this->Lang('error_invalidemail') : $this->Lang('error_invalidusername')
      );
    }
    
    // validate the individual fields.
    $matchfields_str = $this->GetPreference('additionalgroups_matchfields', '');
    
    
    $matchfields = explode('::', $matchfields_str);
    foreach($properties as $propname => $prop)
    {
      $defn         = $defns[$propname];
      $type         = $defn['type'];
      $required     = ($prop['required'] == 2);
      $force_unique = $defn['force_unique'];
      $val          = \xt_utils::get_param($eparams, 'input_' . $propname);
      
      if($required && !$val)
      {
        throw new \RuntimeException($this->Lang('error_requiredfield', $defn['prompt']));
      }
      
      switch($type)
      {
        case 2: // email
          if($val)
          {
            $tmp = $mams->IsValidEmailAddress($val);
            if(!$tmp[0])
            {
              throw new \RuntimeException($tmp[1]);
            }
          }
        break;
      }
      
      if($force_unique && (!$allow_overwrite || !in_array($propname, $matchfields)))
      {
        if($val && !$mams->IsUserPropertyValueUnique(-1, $propname, $val))
        {
          throw new \RuntimeException($this->Lang('error_uniquefield', $defn['prompt']));
        }
      }
    }
    
    if(($flag = $this->GetPreference('enable_whitelist')))
    {
      // this could go into a hook handler
      $matched = FALSE;
      $list    = preg_split("/((\r(?!\n))|((?<!\r)\n)|(\r\n))/", $this->GetPreference('whitelist', ''));
      if(count($list))
      {
        foreach($list as $one_line)
        {
          $regex = '/^' . str_replace("@", "\@", str_replace("\*", ".*", preg_quote($one_line))) . '$/';
          if(preg_match($regex, $username_1) > 0)
          {
            $matched = TRUE;
            break;
          }
        }
      }
      
      if(($flag == 'exclude' && $matched) || ($flag == 'include' && !$matched))
      {
        // @todo: use lang string and get rid of this preference.
        throw new \MAMSREGValidationError($this->GetPreference('whitelist_trigger_message', 'Whitelist Matched'));
      }
    }
    
    // one further event
    $pass    = TRUE;
    $message = NULL;
    \CMSMS\HookManager::do_hook(
      'MAMSRegistration::beforeNewUser',
      [
        'username' => $username_1, 'email' => $email, 'password' => $password_1, 'pass' => &$pass,
        'message'  => &$message
      ]
    );
    
    $pass = \cms_to_bool($pass);
    
    if(!$pass)
    {
      if(!$message)
      {
        $message = $this->Lang('error_validation_other');
      }
      throw new MAMSREGValidationError($message);
    }
    
    //
    // done all validation
    //
    $eparams['overwrite_uid'] = $overwrite_uid;
    $this->session_put('signup', $eparams); // save the data incase of error.
    $this->redirect('cntnt01', 'reguser', $returnid);
  }
  catch(\MAMSREGValidationError $e)
  {
    audit('', $this->GetName(), 'Signup Validation Error: ' . $e->GetMessage());
    $error   = 1;
    $message = $e->GetMessage();
  }
  catch(\Exception $e)
  {
    $error   = 1;
    $message = $e->GetMessage();
  }
}

//
// show something to the user
//

// now we're ready to populate the template
// first we put in stuff that is required (username, password, etc, etc)
$rowarray = [];

// make sure username is in there
$onerow                  = new StdClass();
$onerow->color           = $mams->GetPreference('required_field_color', 'blue');
$onerow->marker          = $mams->GetPreference('required_field_marker', '*');
$onerow->required        = 1;
$val                     = (isset($eparams['input_username'])) ? $eparams['input_username'] : '';
$onerow->prompt          = $mams->GetUsernamePrompt();
$onerow->name            = 'username';
$onerow->input_name      = $id . 'mr_input_username';
$onerow->type            = ($username_is_email) ? 'email' : 'text';
$onerow->labelfor        = $id . 'username';
$onerow->size            = $mams->GetPreference('usernamefldlength');
$onerow->maxlength       = $mams->GetPreference('max_usernamelength');
$onerow->readonly        = FALSE;
$onerow->val             = $val;
$onerow->control         = $this->CreateInputText(
  $id, 'mr_input_username', $val,
  $mams->GetPreference('usernamefldlength'),
  $mams->GetPreference('max_usernamelength')
);
$rowarray[$onerow->name] = $onerow;

if($this->GetPreference('mamsreg_force_email_twice') && $username_is_email)
{
  // add it again
  $onerow           = new StdClass();
  $onerow->color    = $mams->GetPreference('required_field_color', 'blue');
  $onerow->marker   = $mams->GetPreference('required_field_marker', '*');
  $onerow->required = 1;
  $val              = (isset($eparams['input_username_again'])) ? $eparams['input_username_again'] : '';
  
  if($mams->GetPreference('username_is_email'))
  {
    $onerow->prompt = $this->Lang('email');
  }
  else
  {
    $onerow->prompt = $this->Lang('username');
  }
  $onerow->prompt          .= ' (' . $this->Lang('again') . ')';
  $onerow->name            = 'username_again';
  $onerow->labelfor        = $id . 'username_again';
  $onerow->input_name      = $id . 'mr_input_username_again';
  $onerow->type            = ($username_is_email) ? 'email' : 'text';
  $onerow->size            = $mams->GetPreference('usernamefldlength');
  $onerow->maxlength       = $mams->GetPreference('max_usernamelength');
  $onerow->readonly        = FALSE;
  $onerow->val             = $val;
  $onerow->control         = $this->CreateInputText(
    $id, 'mr_input_username_again', $val,
    $mams->GetPreference('usernamefldlength'),
    $mams->GetPreference('max_usernamelength')
  );
  $rowarray[$onerow->name] = $onerow;
}


// and password
$onerow                  = new StdClass();
$onerow->color           = $mams->GetPreference('required_field_color', 'blue');
$onerow->marker          = $mams->GetPreference('required_field_marker', '*');
$onerow->required        = 1;
$val                     = (isset($eparams['input_password'])) ? $eparams['input_password'] : '';
$onerow->prompt          = $this->Lang('password');
$onerow->name            = 'password';
$onerow->labelfor        = $id . 'password';
$onerow->input_name      = $id . 'mr_input_password';
$onerow->type            = 'password';
$onerow->size            = $mams->GetPreference('passwordfldlength');
$onerow->maxlength       = $mams->GetPreference('max_passwordlength');
$onerow->readonly        = FALSE;
$onerow->val             = $val;
$onerow->control         = $this->CreateInputPassword(
  $id, 'mr_input_password', $val,
  $mams->GetPreference('passwordfldlength'),
  $mams->GetPreference('max_passwordlength')
);
$rowarray[$onerow->name] = $onerow;

// and make him repeat the password
$onerow                  = new StdClass();
$onerow->color           = $mams->GetPreference('required_field_color', 'blue');
$onerow->marker          = $mams->GetPreference('required_field_marker', '*');
$onerow->required        = 1;
$val                     = (isset($eparams['input_repeatpassword'])) ? $eparams['input_repeatpassword'] : '';
$onerow->prompt          = $this->Lang('repeatpassword');
$onerow->name            = 'repeatpassword';
$onerow->labelfor        = $id . 'repeatpassword';
$onerow->input_name      = $id . 'mr_input_repeatpassword';
$onerow->type            = 'password';
$onerow->size            = $mams->GetPreference('passwordfldlength');
$onerow->maxlength       = $mams->GetPreference('max_passwordlength');
$onerow->readonly        = FALSE;
$onerow->val             = $val;
$onerow->control         = $this->CreateInputPassword(
  $id, 'mr_input_repeatpassword', $val,
  $mams->GetPreference('passwordfldlength'),
  $mams->GetPreference('max_passwordlength')
);
$rowarray[$onerow->name] = $onerow;

$relations2      = [];
$done_email_reqd = FALSE;
foreach($properties as $propname => $reln)
{
  if($reln['type'] == 6 || $reln['type'] == 9)
  {
    // images and data fields are ignored. user can fill them in after registration
    continue;
  }
  if($reln['required'] == 3)
  {
    continue;
  } // ignore hidden fields.
  
  if($reln['type'] == 2 && !$done_email_reqd)
  {
    // email
    $done_email_reqd = TRUE;
    if(!$username_is_email)
    {
      // gotta have an email address, force it to be required.
      $reln['required'] = 2;
    }
    $relations2[] = $reln;
    if($this->GetPreference('mamsreg_force_email_twice'))
    {
      $reln['mapto'] = $reln['name'];
      $reln['name']  = $reln['name'] . '_again';
      $relations2[]  = $reln;
    }
    continue;
  }
  else
  {
    $relations2[] = $reln;
  }
}

// convert relations to properties.
foreach($relations2 as $reln)
{
  // get the property definition
  $defn = NULL;
  if(isset($reln['mapto']))
  {
    $defn = $mams->GetPropertyDefn($reln['mapto']);
  }
  else
  {
    $defn = $mams->GetPropertyDefn($reln['name']);
  }
  $attribs = NULL;
  if(isset($defn['attribs']) && $defn['attribs'])
  {
    $attribs = unserialize($defn['attribs']);
  }
  
  $color  = '';
  $marker = '';
  if($defn['encrypt'])
  {
    $color  = $this->GetPreference('secure_field_color', 'yellow');
    $marker = $this->GetPreference('secure_field_marker', '^^');
  }
  if($reln['required'] == 2)
  {
    $color  = $mams->GetPreference('required_field_color', 'blue');
    $marker = $mams->GetPreference('required_field_marker', '*');
  }
  $onerow             = new StdClass();
  $onerow->required   = ($reln['required'] == 2);
  $onerow->color      = $color;
  $onerow->marker     = $marker;
  $onerow->prompt     = $reln['prompt'];
  $onerow->name       = $reln['name'];
  $onerow->labelfor   = $id . $reln['name'];
  $onerow->readonly   = ($reln['required'] == 4);
  $onerow->input_name = $id . 'mr_input_' . $reln['name'];
  
  $val = NULL;
  // default values for fields...
  switch($defn['type'])
  {
    case 1:
      if(isset($attribs['checked']))
      {
        $val = $attribs['checked'];
      }
    break;
  }
  $val         = isset($eparams['input_' . $reln['name']]) ? $eparams['input_' . $reln['name']] : $val;
  $onerow->val = $val;
  
  switch($defn['type'])
  {
    case 0: // text
      $onerow->type      = 'text';
      $onerow->size      = $defn['length'];
      $onerow->maxlength = $defn['maxlength'];
      $onerow->control   = $this->CreateInputText(
        $id, 'mr_input_' . $reln['name'],
        $val, $defn['length'], $defn['maxlength']
      );
    break;
    
    case 1: // checkbox
      $onerow->type    = 'checkbox';
      $onerow->control = MAMSUtils::myCreateInputCheckbox($id, 'mr_input_' . $reln['name'], 1, $val);
    break;
    
    case 2: // email
      $onerow->type      = 'email';
      $onerow->size      = $defn['length'];
      $onerow->maxlength = $defn['maxlength'];
      $onerow->control   = $this->CreateInputText(
        $id, 'mr_input_' . $reln['name'],
        $val, $defn['length'], ($defn['maxlength'])
      );
    break;
    
    case 3: // textarea
      $onerow->type      = 'textarea';
      $onerow->maxlength = isset($defn['maxlength']) ? $defn['maxlength'] : NULL;
      $onerow->control   = $this->CreateTextArea(FALSE, $id, $val, 'mr_input_' . $reln['name']);
    break;
    
    case 4: // dropdown
      $onerow->type    = 'dropdown';
      $onerow->options = array_flip($mams->GetSelectOptions($defn['name'], 1));
      $onerow->control = $this->CreateInputDropdown(
        $id, 'mr_input_' . $reln['name'],
        $mams->GetSelectOptions($defn['name'], 1),
        -1, $val
      );
    break;
    
    case 5: // multiselect
      $onerow->type = 'multiselect';
      $tmp          = $mams->GetSelectOptions($defn['name'], 1);
      if(!is_array($val))
      {
        $val = explode(',', $val);
      }
      $onerow->options    = array_flip($tmp);
      $onerow->input_name = $onerow->input_name . '[]';
      $onerow->val        = $val;
      $onerow->control    = $this->CreateInputSelectList(
        $id, 'mr_input_' . $defn['name'] . '[]', $tmp, $val, min(count($tmp), 5)
      );
    break;
    
    case 7: // radio buttons
      $onerow->type    = 'radiogroup';
      $onerow->options = array_flip($mams->GetSelectOptions($defn['name'], 1));
      $onerow->control = $this->CreateInputRadioGroup(
        $id, 'mr_input_' . $defn['name'],
        $mams->GetSelectOptions($defn['name'], 1),
        $val, '', '<br/>'
      );
    break;
    
    case 8: // date field
      $onerow->type    = 'date';
      $attribs         = unserialize($defn['attribs']);
      $parms           = [];
      $parms['prefix'] = $id . 'mr_input_' . $defn['name'];
      if($val)
      {
        $parms['time'] = $val;
      }
      $parms['start_year'] = (isset($attribs['startyear'])) ? $attribs['startyear'] : "-5";
      $parms['end_year']   = (isset($attribs['endyear'])) ? $attribs['endyear'] : "+10";
      $str                 = '{html_select_date ';
      foreach($parms as $key => $value)
      {
        $str .= $key . '="' . $value . '" ';
      }
      $str                .= '}';
      $onerow->start_year = isset($attribs['startyear']) ? $attribs['startyear'] : '-5';
      $onerow->end_year   = isset($attribs['endyear']) ? $attribs['endyear'] : '+10';
      $onerow->control    = $this->ProcessTemplateFromData($str);
    break;
  }
  
  $rowarray[$onerow->name] = $onerow;
}

$inline = $this->GetPreference('inline_forms', TRUE);
if(isset($eparams['noinline']))
{
  $inline = FALSE;
}

$tplname = \xt_param::get_string($params, 'regtemplate');
if($tplname)
{
  $tplname = '::' . $tplname;
}
else
{
  $tplname = 'mamsreg_reg1template';
}
$tpl = $this->CreateSmartyTemplate($tplname);

// and the rest of the stuff
$eparams['mr_orig_url']    = (isset($eparams['mr_orig_url'])) ? $eparams['mr_orig_url'] : xt_url::current_url();
$eparams['allowoverwrite'] = $allow_overwrite;
$tmp                       = ['mr_data' => base64_encode(serialize($eparams))];
$tpl->assign('startform', $this->XTCreateFormStart($id, 'signup', $returnid, $tmp, $inline));
$tpl->assign('endform', $this->CreateFormEnd());
$tpl->assign('controls', $rowarray);
$tpl->assign('controlcount', count($rowarray));
$tpl->assign('error', $error);
$tpl->assign('message', $message);

$captcha = $this->GetModuleInstance('Captcha');
if(is_object($captcha) && !$nocaptcha)
{
  $test = method_exists($captcha, 'NeedsInputField') ? $captcha->NeedsInputField() : TRUE;
  $tpl->assign('captcha_title', $this->Lang('captcha_title'));
  $tpl->assign('captcha', $captcha->getCaptcha());
  $tpl->assign('input_captcha', $this->CreateInputText($id, 'mr_input_captcha', '', 10));
  $tpl->assign('need_captcha_input', $test);
}

// todo, put this into the database and let the admin play with it.
$tpl->display();

#
# EOF
#
