<?php
#BEGIN_LICENSE
#-------------------------------------------------------------------------
# Module: CGNotifier (c) 2012 by Robert Campbell 
#         (calguy1000@cmsmadesimple.org)
# An addon module for CMSMS to allow sending notifications in various forms.
# 
#-------------------------------------------------------------------------
# CMS - CMS Made Simple is (c) 2005 by Ted Kulp (wishy@cmsmadesimple.org)
# This project's homepage is: http://www.cmsmadesimple.org
#
#-------------------------------------------------------------------------
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# However, as a special exception to the GPL, this software is distributed
# as an addon module to CMS Made Simple.  You may not use this software
# in any Non GPL version of CMS Made simple, or in any version of CMS
# Made simple that does not indicate clearly and obviously in its admin 
# section that the site was built with CMS Made simple.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
# Or read it online: http://www.gnu.org/licenses/licenses.html#GPL
#
#-------------------------------------------------------------------------
#END_LICENSE

class CGNotifier extends CGExtensions
{
  const CURDFLT_EDITFORM_NAME = 'curdflt_editform_name';
  const SYSDFLT_EDITFORM_TPL  = 'sysdflt_editform_tpl';
  const FEU_PROPNAME = '__CGNotifier__';

  public function __construct()
  {
    parent::__construct();
    $this->AddImageDir('images');
  }

  public function GetName() { return 'CGNotifier'; }
  public function GetVersion() { return '1.0.4'; }
  public function GetAuthor() { return 'calguy1000'; }
  public function GetAuthorEmail() { return 'calguy1000@cmsmadesimple.org'; }
  public function IsPluginModule() { return TRUE; }
  public function HasAdmin() { return true; }
  public function GetAdminSection() { return 'siteadmin'; }
  public function GetAdminDescription() { return $this->Lang('moddescription'); }
  public function VisibleToAdminUser() {
    return $this->CheckPermission('Modify Site Preferences') ||
      $this->CheckPermission('Modify Templates') ||
      $this->CheckPermission('Manage My Notifications');
  }
  public function MinimumCMSVersion() { return "1.10.2"; }
  public function GetDependencies() {
    return array('CGExtensions'=>'1.32.6',
		 'FrontEndUsers'=>'1.21.14',
		 'JQueryTools'=>'1.2.5',
		 'CGSimpleSmarty'=>'1.6.2'); 
  }
  public function LazyLoadAdmin() { return TRUE; }
  public function LazyLoadFrontend() { return TRUE; }
  public function HandlesEvents() { return TRUE; }

  public function GetFriendlyName() {
    $tmp = $this->GetPreference('friendlyname');
    if( !$tmp ) $tmp = $this->Lang('friendlyname');
    return $tmp;
  }

  public function GetHelp()
  {
    return @file_get_contents(dirname(__FILE__).'/help.inc');
  }

  public function GetChangeLog()
  {
    return @file_get_contents(dirname(__FILE__).'/changelog.inc');
  }

  public function InitializeFrontend()
  {
    $this->RestrictUnknownParams();
    $this->RegisterModulePlugin();
    $this->SetParameterType(CLEAN_REGEXP.'/cgn_.*/',CLEAN_STRING);
    $this->SetParameterType('template',CLEAN_STRING);
  }

  public function InstallPostMessage() { return $this->Lang('postinstall'); }
  public function UninstallPostMessage() { return $this->Lang('postuninstall'); }
  public function UninstallPreMessage() { return $this->Lang('really_uninstall'); }

  public function HasCapability($capability, $params = array())
  {
    if( $capability == 'notifications' ) return TRUE;
    if( $capability == 'cg_messaging_client' ) {
      if( $this->GetPreference('handle_emailnotifications') ) return TRUE;
    }
    return FALSE;
  }

  function get_transport_name() { return $this->Lang('email'); }

  /*---------------------------------------------------------
   GetHeaderHTML()
   ---------------------------------------------------------*/
  function GetHeaderHTML()
  {
    $txt = parent::GetHeaderHTML();
    $obj = $this->GetModuleInstance('JQueryTools');
    if( is_object($obj) ) {
      $tmpl = <<<EOT
{JQueryTools action='incjs' exclude='form'}
{JQueryTools action='ready'}
EOT;
      $txt .= $this->ProcessTemplateFromData($tmpl);
    }
    return $txt;
  }

  public function notify(notification_message $msg,notification_profile &$profile = null)
  {
    // get the priority
    $priority = $msg->priority;

    // get the recipient profile
    if( !is_object($profile) ) {
      if( $msg->to <= 0 ) {
	// see if we can get an admin user profile
	if( $msg->to < 0 ) {
	  $profile = get_preference($msg->to*-1,$this->GetName().'_profile');
	  if( $profile ) {
	    $profile = unserialize($profile);
	    if( !is_object($profile) ) $profile = null;
	  }
	}
	if( !$profile ) {
	  // admin user profile specified, but couldn't find one, use default
	  $profile = $this->GetPreference('dfltprofile');
	  if( $profile ) {
	    $profile = unserialize($profile);
	    if( !is_object($profile) ) $profile = null;
	  }
	}
      }
      else if( $msg->to > 0 ) {
	// see if we can get an feu user profile.
	$feu = cms_utils::get_module('FrontEndUsers');
	if( $feu ) {
	  $profile = $feu->GetUserPropertyFull(self::FEU_PROPNAME,$msg->to);
	  if( $profile ) {
	    $profile = unserialize($profile);
	    if( !is_object($profile) ) $profile = null;
	  }
	}
      }
      else {
	// no to specified... grab default profile.
	$profile = $this->GetPreference('dfltprofile');
	if( $profile ) {
	  $profile = unserialize($profile);
	  if( !is_object($profile) ) $profile = null;
	}
      }
    }

    if( !is_object($profile) ) return FALSE;

    // if we have targets at this priority
    $items = $profile->find_item($msg);
    if( !is_array($items) || count($items) == 0 ) {
      audit('',$this->GetName(),'Notification message to '.$msg->to.': no handlers found');
      return FALSE;
    }

    $nsent = 0;
    foreach( $items as $one_item ) {
      $mod = cms_utils::get_module($one_item->send_method);
      if( !is_object($mod) ) continue;

      // double check that this module has the capability. (maybe it got shut down)
      if( !$mod->HasCapability('cg_messaging_client') ) {
	audit('',$this->GetName(),'Notification message to '.$msg->to.' failed: module '.$mod->GetName().' does not have the capability.');
	return FALSE;
      }

      // get the target
      $res = $mod->send_message($msg);
      if( $res ) $nsent++;
    }

    if( $nsent > 0 ) return TRUE;
    return FALSE;
  }

  function send_message(notification_message $msg)
  {
    if( !$this->GetPreference('handle_emailnotifications') ) return FALSE;

    if( !$msg->body ) return FALSE;

    // see if we can get an email address for this user
    $email_addresses = null;
    if( $msg->to ) {
      if( $msg->to < 0 ) {
	// negative UID indicates an admin user.
	// get email address for single admin account
	$user = UserOperations::get_instance()->LoadUserByID($msg->to * -1);
	if( !is_object($user) ) {
	  audit('',$this->GetName(),'Attempt to send message to uid '.$msg->to.' but user not found');
	  return FALSE;
	}
	if( $user->email == '' ) {
	  audit('',$this->GetName(),'Attempt to send message to uid '.$msg->to.' but no email address found');
	  return FALSE;
	}
	$email_addresses = array();
	$email_addresses[] = $feu->GetEmail($user->email);
      }
      else {
	// positive values indicate an FEU user.
	$feu = cms_utils::get_module('FrontEndUsers');
	if( $feu ) {
	  $email_addresses = array();
	  $email_addresses[] = $feu->GetEmail($msg->to);
	}
      }
    }
    else {
      // to address not specified. ... use default notification group.
      $gid = $this->GetPreference('dflt_notificationsgroup',-1);
      if( $gid <= 0 ) return FALSE;

      // use default group name... get all matching email addresses.
      $email_addresses = cge_userops::expand_group_emails($gid);
    }

    if( !is_array($email_addresses) || count($email_addresses) == 0 ) {
      audit('',$this->GetName(),'Attempt to send message to: '.$msg->to.' but no email addresses found');
      return FALSE;
    }

    $mailer = cms_utils::get_module('CMSMailer');
    foreach( $email_addresses as $email ) {
      $mailer->AddAddress($email);
    }
    if( $msg->subject ) $mailer->SetSubject($msg->subject);
    $mailer->IsHTML($msg->html);
    $mailer->SetBody($msg->body);
    $res = $mailer->Send();
    if( count($email_addresses) == 1 ) {
      audit('',$this->GetName(),'Notification message sent with subject: '.$msg->subject.' to '.$email_addresses[0]);
    }
    else {
      audit('',$this->GetName(),'Notification message sent with subject: '.$msg->subject.' to '.count($email_addresses).' addresses');
    }
    $mailer->reset();

    return $res;
  }
} // class

?>