<?php

class cggm2_map_generator
{
  private $_map;
  private $_defer;
  private $_api_ver = '3.s';
  private $_all_icons = FALSE;
  
  private $_markers = array();        // a list of all displayable marker objects (after combining points)
  private $_icon_names = array();     // a list of all used icons
  private $_icons = array();          // an array of use icon objects
  private $_meta = array();           // calculated metadata
  private $_errors = array();         // errors
  private static $_instance = 0;      // instance generator.

  public function __construct(cggm2_map& $map,$defer = false)
  {
    self::$_instance++;
    $this->_map = $map;
    $this->_defer = $defer;
  }

  public function get_errors()
  {
    if( !count($this->_errors) ) return;
    return $this->_errors;
  }

  private function _calc_coords_meta()
  {
    if( count($this->_markers) ==  0 ) return;
    if( isset($this->_meta['center_lat']) ) return;

    // calculate minimum/maximum lat/long
    $min_lat = 9999999.9;
    $min_lon = 9999999.9;
    $max_lat = -9999999.9;
    $max_lon = -9999999.9;
    foreach( $this->_markers as $marker_name => $marker ) {
      $min_lat = (float)min($marker->get_latitude(),$min_lat);
      $min_lon = (float)min($marker->get_longitude(),$min_lon);
      $max_lat = (float)max($marker->get_latitude(),$max_lat);
      $max_lon = (float)max($marker->get_longitude(),$max_lon);
    }
    // adjust min/max lat/long by bounds fudge
    $diff_lat = $max_lat - $min_lat;
    $diff_lon = $max_lon - $min_lon;
    $min_lat -= $diff_lat * (float)$this->_map->bounds_fudge;
    $min_lon -= $diff_lon * (float)$this->_map->bounds_fudge;
    $max_lat += $diff_lat * (float)$this->_map->bounds_fudge;
    $max_lon += $diff_lon * (float)$this->_map->bounds_fudge;

    // calculate the center.
    $center_lat = $min_lat + ($max_lat - $min_lat) / 2.0;
    $center_lon = $min_lon + ($max_lon - $min_lon) / 2.0;

    $this->_meta['min_lat'] = (float)$min_lat;
    $this->_meta['min_lon'] = (float)$min_lon;
    $this->_meta['max_lat'] = (float)$max_lat;
    $this->_meta['max_lon'] = (float)$max_lon;
    $this->_meta['center_lat'] = (float)$center_lat;
    $this->_meta['center_lon'] = (float)$center_lon;
  }

  public function set_all_icons($flag = TRUE)
  {
    $this->_all_icons = $flag;
  }

  protected function prepare_markers()
  {
    $map = $this->_map;
    $mod = cge_utils::get_module('CGGoogleMaps2');
    $this->_markers = array();
    $input_names = $map->get_marker_names();
    if( !$input_names ) return;
    foreach( $input_names as $marker_name ) {
      // get the marker.
      $marker = $map->get_marker_by_name($marker_name);

      // make sure it has an icon
      if( $marker->get_icon() == '' || $marker->get_icon() == -1 ) $marker->set_icon($map->default_icon);
      if( !in_array($marker->get_icon(),$this->_icon_names) ) $this->_icon_names[] = $marker->get_icon();

      // grab a lat/long for this marker if necessary.
      if( $marker->get_latitude() == '' || $marker->get_longitude() == '' ) {
	$coords = cggm2_address_lookup::lookup($marker->get_address());
	if( !$coords ) {
	  // this point has to be skipped
	  $this->_errors[] = $mod->Lang('marker_skipped',$marker->get_title());
	  continue;
	}
	$marker->set_coords($coords[0],$coords[1]);
      }

      // handle combined points
      if( $map->combine_points && $map->point_combine_fudge > 0.000 ) {
	$merged = false;
	foreach( $this->_markers as $m_name => &$m_info ) {
	  $box_toplat = $m_info->get_latitude() - $map->point_combine_fudge;
	  $box_toplon = $m_info->get_longitude() - $map->point_combine_fudge;
	  $box_botlat = $m_info->get_latitude() + $map->point_combine_fudge;
	  $box_botlon = $m_info->get_longitude() + $map->point_combine_fudge;
	  if( $marker->get_latitude() >= $box_toplat && $marker->get_latitude() <= $box_botlat &&
	      $marker->get_longitude() >= $box_toplon && $marker->get_longitude() <= $box_botlon ) {
	    // gotta combine this marker with an existing one.
	    if( !$m_info instanceof cggm2_meta_marker ) {
	      $m_info = new cggm2_meta_marker($m_info,$map->combined_icon);
	      $m_info->set_title($mod->Lang('title_combined_marker'));
	    }
	    $m_info->add_marker($marker);
	    // done merging;
	    $merged = true;
	    break;
	  }
	}
	if( $merged ) {
	  // done merging;
	  continue;
	}
      }

      $this->_markers[$marker_name] = $marker;
    }
  }

  public function get_marker_data()
  {
    $map = $this->_map;
    if( count($this->_markers) == 0 ) return;
    $mod = cge_utils::get_module('CGGoogleMaps2');

    $txt = '';
    $txt .= '<div id="cggm2_map_data_'.self::$_instance.'" style="display: none;">'."\n";
    $txt .= '<div class="cggm2_map_markers" style="display: none;">'."\n";
    foreach( $this->_markers as $marker_name => $marker ) {
      $tmp = '<div class="cggm2_marker" id="map_'.self::$_instance.'_marker_'.$marker_name.'">'."\n";
      $tmp .= '<input type="hidden" name="name" value="'.$marker_name.'"/>'."\n";
      $tmp .= '<input type="hidden" name="title" value="'.cms_htmlentities($marker->get_title()).'"/>'."\n";
      $tmp .= '<input type="hidden" name="latitude" value="'.number_format($marker->get_latitude(),6,'.','').'"/>'."\n";
      $tmp .= '<input type="hidden" name="longitude" value="'.number_format($marker->get_longitude(),6,'.','').'"/>'."\n";
      $tmp .= '<input type="hidden" name="icon" value="'.$marker->get_icon().'"/>'."\n";
      if( ($tooltip = $marker->get_tooltip()) ) $tmp .= '<input type="hidden" name="tooltip" value="'.$tooltip.'"/>'."\n";

      $description = '';
      if( $marker instanceof cggm2_meta_marker ) {
	$tmp .= '<input type="hidden" name="meta" value="1"/>'."\n";
	for( $i = 0; $i < $marker->count_markers(); $i++ ) {
	  $tmpm = $marker->get_marker($i);
	  $description .= '<div class="cggm2_infowindow_item">'.$tmpm->get_description();
	  $description .= '</div>'."\n";
	}
      }
      else {
	$tmp .= '<input type="hidden" name="meta" value="0"/>'."\n";
	$description .= '<div class="cggm2_infowindow_item">'.$marker->get_description().'</div>'."\n";
      }

      if( $map->directions ) {}
      if( $map->sv_controls ) { }

      $tmp .= '<div class="cggm2_infowindow_contents">'."\n";
      $tmp .= '<div class="cggm2_infowindow" id="cggm2_map_infowindow_'.self::$_instance.'_'.$marker_name.'">'."\n".$description.'</div>'."\n";
      $tmp .= "</div><!-- cggm2_infowindow_contents -->\n";
      $tmp .= "</div><!-- cggm2_marker -->\n"; // close marker tag.
      $txt .= $tmp;
    }

    $txt .= "</div><!-- cggm2_map_markers -->\n"; // close markers tag
    $txt .= "</div><!-- cggm2_map_data -->\n"; // close data tag
    $txt .= '<div id="cggm2_infowindow_'.self::$_instance.'" style="position: absolute; left: -1000px;"></div>'."\n";
    return $txt;
  }

//   protected function generate_icons()
//   {
//     $db = cmsms()->GetDb();
//     $query = 'SELECT * FROM '.cms_db_prefix().'module_cggooglemaps2_icons';
//     $tmp = $db->GetArray($query);
//     if( is_array($tmp) ) $tmp = cge_array::to_hash($tmp,'name');
//     return $tmp;
//   }

  protected function generate_map_data()
  {
    $defertxt = '';
    $mod = cge_utils::get_module('CGGoogleMaps2');
    if( $this->_defer ) $defertxt = ' defer="defer"';
    $smarty = cmsms()->GetSmarty();
    $smarty->assign('map_defertxt',$defertxt);
    //$smarty->assign('msg_browser_incompatible',$mod->Lang('error_browser_incompatible'));
    //$smarty->assign('msg_mapelem_notfound',$mod->Lang('error_mapelem_notfound'));
    //$smarty->assign('map_elem','map_'.self::$_instance);
    $kml_files = $this->_map->get_kml_files();
    if( is_array($kml_files) && count($kml_files) ) {
      $smarty->assign('kml_files',$kml_files);
    } else {
      $smarty->assign('kml_files','');
    }
  }

  protected function generate_map()
  {
    $template = $this->_map->get_map_template_name();
    if( $template ) {
      $mod = cge_utils::get_module('CGGoogleMaps2');
      $output = $mod->ProcessTemplateFromDatabase($template);
      return $output;
    }
  }

  public function generate()
  {
    $map = $this->_map;
    $smarty = cmsms()->GetSmarty();
    $smarty->assign('map',$this->_map);
    $smarty->assign('mapinstance',self::$_instance);
    $smarty->assign('generator',$this);

    // parse through the markers, cleaning up icons if necessary.
    $this->prepare_markers();

    $mod = cge_utils::get_module('CGGoogleMaps2');
    $smarty->assign('icon_base_url',$mod->GetModuleURLPath().'/icons');

    // and put it all together
    $output = $this->generate_map();
    return $output;
  }

  public function get_marker_count()
  {
    return count($this->_markers);
  }

  public function get_center_lat()
  {
    if( $this->_map->center_lat != '' ) return $this->_map->center_lat;

    $this->_calc_coords_meta();
    $val = 0;
    if( isset($this->_meta['center_lat']) ) $val = $this->_meta['center_lat'];
    return number_format($val,6,'.',',');
  }

  public function get_center_lon()
  {
    if( $this->_map->center_lon != '' ) return $this->_map->center_lon;

    $this->_calc_coords_meta();
    $val = 0;
    if( isset($this->_meta['center_lon']) ) $val = $this->_meta['center_lon'];
    return number_format($val,6,'.',',');
  }

  public function get_min_lat()
  {
    $this->_calc_coords_meta();
    return number_format($this->_meta['min_lat'],6,'.',',');
  }

  public function get_min_lon()
  {
    $this->_calc_coords_meta();
    return number_format($this->_meta['min_lon'],6,'.',',');
  }

  public function get_max_lat()
  {
    $this->_calc_coords_meta();
    return number_format($this->_meta['max_lat'],6,'.',',');
  }

  public function get_max_lon()
  {
    $this->_calc_coords_meta();
    return number_format($this->_meta['max_lon'],6,'.',',');
  }

  public function get_google_maptype()
  {
    switch($this->_map->type) {
    case 'map':
      return 'ROADMAP';
      break;
    case 'satellite':
    case 'terrain':
    case 'hybrid':
      return strtolower($this->_map->type);
    }
  }

  public function _array_to_jsarray($in)
  {
    $data = array();
    foreach( $in as $val ) {
      if( $val == '' ) continue;

      $datatype = gettype($val);
      switch( $datatype ) {
      case 'boolean':
	$data[] = (bool)$val;
	continue;

      case 'integer':
	$data[] = (int)$val;
	continue;

      case 'double':
	$data[] = (double)$val;
	continue;

      case 'string':
	if( ($val == 'true' || $val == 'TRUE' || $val == 'false' || $val == 'FALSE') ) {
	  $data[] = (bool)$val;
	}
	else {
	  if( !startswith($val,"'") && !startswith($val,'"') ) {
	    $data[] = "'".$val."'";
	  }
	  else {
	    $data[] = $val;
	  }
	}
	continue;

      case 'array':
	if( cge_array::is_hash($val) ) {
	  $data[] = $this->_hash_to_jsobj($val);
	}
	else {
	  $data[] = $this->_array_to_jsarray($val);
	}
	continue;
	
      case 'object':
      case 'NULL':
      default:
	// do nothing
	continue;
      }
    }

    return '[ '.implode(',',$data).' ]';
  }

  public function _hash_to_jsobj($in)
  {
    $data = array();
    foreach( $in as $fld => $val ) {
      if( $val == '' ) continue;

      $datatype = gettype($val);
      switch( $datatype ) {
      case 'boolean':
	$data["'".$fld."'"] = (bool)$val;
	continue;

      case 'integer':
	$data["'".$fld."'"] = (int)$val;
	continue;

      case 'double':
	$data["'".$fld."'"] = (double)$val;
	continue;

      case 'string':
	if( ($val == 'true' || $val == 'TRUE' || $val == 'false' || $val == 'FALSE') ) {
	  $data["'".$fld."'"] = (bool)$val;
	}
	elseif( (int)$val != 0 || $val == '0' ) {
	  $data["'".$fld."'"] = (int)$val;
	}
	else {
	  if( !startswith($val,"'") && !startswith($val,'"') ) {
	    $data["'".$fld."'"] = "'".$val."'";
	  }
	  else {
	    $data["'".$fld."'"] = $val;
	  }
	}
	continue;

      case 'array':
	if( cge_array::is_hash($val) ) {
	  $data["'".$fld."'"] = $this->_hash_to_jsobj($val);
	}
	else {
	  $data[] = $this->_array_to_jsarray($val);
	}
	continue;
	
      case 'object':
      case 'NULL':
      default:
	// do nothing
	continue;
      }
    }

    $out = '{ '.cge_array::implode_with_key($data,': ',',').' }';
    return $out;
  }

  public function get_map_options_js()
  {
    $data = array();
    $fields = $this->_map->get_fields();
    foreach( array_keys($fields) as $fld ) {
      $val = $this->_map->$fld;
      if( $val == '' ) continue;
      $data[$fld] = $val;
    }

    $mod = cge_utils::get_module('CGGoogleMaps2');
    $data['youareherestr'] = $mod->Lang('youarehere');
    $data["mapinstance"] = self::$_instance;
    $data['center'] = "'".$this->get_center_lat().', '.$this->get_center_lon()."'";
    if( !in_array($this->_map->default_icon,$this->_icon_names) ) $this->_icon_names[] = $this->_map->default_icon;
    if( !in_array($this->_map->sensor_icon,$this->_icon_names) ) $this->_icon_names[] = $this->_map->sensor_icon;
    if( !in_array($this->_map->combined_icon,$this->_icon_names) ) $this->_icon_names[] = $this->_map->combined_icon;
    if( (is_array($this->_icon_names) && count($this->_icon_names)) || $this->_all_icons ) {
      // get the info for these icons by name
      $db = cmsms()->GetDb();
      $query = '';
      if( $this->_all_icons ) {
	$query = 'SELECT name,url,anchor_x,anchor_y FROM '.cms_db_prefix().'module_cggooglemaps2_icons';
      }
      else {
	$query = 'SELECT name,url,anchor_x,anchor_y FROM '.cms_db_prefix().'module_cggooglemaps2_icons WHERE name IN (';
	$query .= cge_array::implode_quoted($this->_icon_names,',',"'");
	$query .= ')';
      }
      $dbr = $db->GetArray($query);
      $data['icons'] = cge_array::to_hash($dbr,'name');
    }
    $out = $this->_hash_to_jsobj($data);
    return $out;
  }

  public function get_directions_form()
  {
    if( !$this->_map->directions ) return;
    $mod = cms_utils::get_module(MOD_CGGOOGLEMAPS2);
    return $mod->ProcessTemplateFromDatabase($this->_map->get_directions_template_name());
  }
} // end of class
?>