<?php
#BEGIN_LICENSE
#-------------------------------------------------------------------------
# Module: CGJobMgr (c) 2011 by Robert Campbell 
#         (calguy1000@cmsmadesimple.org)
#  An addon module for CMS Made Simple to manage jobs that take considerable
#  amounts of time to process.
# 
#-------------------------------------------------------------------------
# CMS - CMS Made Simple is (c) 2005 by Ted Kulp (wishy@cmsmadesimple.org)
# This project's homepage is: http://www.cmsmadesimple.org
#
#-------------------------------------------------------------------------
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# However, as a special exception to the GPL, this software is distributed
# as an addon module to CMS Made Simple.  You may not use this software
# in any Non GPL version of CMS Made simple, or in any version of CMS
# Made simple that does not indicate clearly and obviously in its admin 
# section that the site was built with CMS Made simple.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
# Or read it online: http://www.gnu.org/licenses/licenses.html#GPL
#
#-------------------------------------------------------------------------
#END_LICENSE

abstract class cgjobmgr_task
{
  const STATUS_NONE = 'none';
  const STATUS_ERROR = 'error';
  const STATUS_INCOMPLETE = 'incomplete';
  const STATUS_COMPLETE = 'complete';
  
  private $_name;
  private $_persistent;
  private $_function;

  /**
   * Constructor
   *
   * @param string The task name
   */
  public function __construct($name = '')
  {
    $this->set_status(self::STATUS_NONE);
    $this->_name = $name;
  }


  /**
   * Get the task name
   *
   * @returns string
   */
  public function get_name()
  {
    return $this->_name;
  }


  /**
   * Set the function to be called by the task to actually do the work
   *
   * i.e:  $task->set_function('some_function');
   * i.e:  $task->set_function('some_class::some_function'); -- call a static method
   * i.e:  $task->set_function(array('some_class','some_function')); -- call a static method
   *
   * @param mixed fn... may be a string containing a function name (or static method name)
   */
  public function set_function($fn)
  {
    if( is_string($fn) )
      {
	$this->_function = $fn;
      }
    else if( is_array($fn) && count($fn) == 2 && is_string($fn[0]) && is_string($fn[1]) )
      {
	$this->_function = $fn;
      }
    else
      {
	throw new Exception('Invalid value passed to set_function');
      }
  }


  /**
   * Retrieve data stored with the task
   *
   * @param string the data key
   * @param mixed the default value for the key if it cannot be found
   * @returns mixed
   */
  public function get_persistent_data($key,$dflt = '')
  {
    if( !is_array($this->_persistent) ) return $dflt;
    if( !isset($this->_persistent[$key]) ) return $dflt;
    return $this->_persistent[$key];
  }


  /**
   * Associate data with the task.
   * The data is serialized before being saved.
   *
   * @param string The data key
   * @param mixed  The data to store
   */
  public function set_persistent_data($key,$value)
  {
    if( !is_array($this->_persistent) ) $this->_persistent = array();
    $this->_persistent[$key] = $value;
  }


  /**
   * Retrieves the function that will do the actual work for the task
   * 
   * @returns mixed
   * @internal
   */
  protected function get_function()
  {
    if( !$this->_function )
      throw new CGJobMgrException('No function defined for task');

    return $this->_function;
  }


  /**
   * Protected method that actually calls the processing function
   * 
   * @internal
   * @returns the status of the task
   */
  abstract protected function _run_task();


  /**
   * Set the task status
   */
  public function set_status($status)
  {
    switch( $status )
      {
      case self::STATUS_NONE:
      case self::STATUS_INCOMPLETE:
      case self::STATUS_ERROR:
      case self::STATUS_COMPLETE:
	$this->set_persistent_data('__status__',$status);
	break;
      default:
	throw new CGJobMgrException('Cannot set task status to: '.$status);
      }
  }


  /**
   * Retrieve the task status
   */
  public function get_status()
  {
    return $this->get_persistent_data('__status__',self::STATUS_NONE);
  }


  /**
   * Execute the task
   *
   * @returns the status of the task
   */
  public function execute()
  {
    $res = $this->_run_task();
    return $res;
  }

  /**
   * Get CGJobMgr run interval.
   *
   * @return integer the run interval (in minutes), or null.
   */
  public function get_run_interval()
  {
    $mod = cms_utils::get_module('CGJobMgr');
    $stats = $mod->GetStats();
    if( !is_array($stats) ) return;
    if( !isset($stats['interval']) ) return;

    return (int)$stats['interval'];
  }
} // end of class

#
# EOF
#
?>