﻿using UnityEngine;
using System.Collections;
using System.Collections.Generic;
using System;

public class MovingObject : MonoBehaviour 
{
	/// <summary>
	/// The previous position.
	/// </summary>
	public Vector2 mOldPosition;
	/// <summary>
	/// The current position.
	/// </summary>
	public Vector2 mPosition;
    public Vector2 mScale;

	/// <summary>
	/// The current speed in pixels/second.
	/// </summary>
	public Vector2 mSpeed;
	
	/// <summary>
	/// The previous speed in pixels/second.
	/// </summary>
	public Vector2 mOldSpeed;

    public Vector2 mAABBOffset;

	/// <summary>
	/// The AABB for collision queries.
	/// </summary>
	public AABB mAABB;

    public Game mGame;

	/// <summary>
	/// True if the instance is right beside the right wall.
	/// </summary>
	//[HideInInspector]
	public bool mPushesRightWall = false;
	/// <summary>
	/// True if the instance is right beside the left wall.
	/// </summary>
	//[HideInInspector]
	public bool mPushesLeftWall = false;
	/// <summary>
	/// True if the instance is on the ground.
	/// </summary>
	//[HideInInspector]
	public bool mOnGround = false;
	/// <summary>
	/// True if the instance hits the ceiling.
	/// </summary>
	//[HideInInspector]
	public bool mAtCeiling = false;
	/// <summary>
	/// The previous state of atCeiling.
	/// </summary>
	//[HideInInspector]
	public bool mWasAtCeiling = false;
	/// <summary>
	/// The previous state of onGround.
	/// </summary>
	//[HideInInspector]
	public bool mWasOnGround = false;
	/// <summary>
	/// The previous state of pushesRightWall.
	/// </summary>
	//[HideInInspector]
	public bool mPushedRightWall = false;
	/// <summary>
	/// The previous state of pushesLeftWall.
	/// </summary>
	//[HideInInspector]
	public bool mPushedLeftWall = false;
	
	public bool mOnOneWayPlatform = false;
	
	/// <summary>
	/// Depth for z-ordering the sprites.
	/// </summary>
	public float mSpriteDepth = -1.0f;

    public Transform mTransform;

	/// <summary>
	/// If the object is colliding with one way platform tile and the distance to the tile's top is less
	/// than this threshold, then the object will be aligned to the one way platform.
	/// </summary>
	public float cOneWayPlatformThreshold = 2.0f;

    public bool mIgnoresOneWayPlatforms = false;

	void OnDrawGizmos()
	{
		DrawMovingObjectGizmos ();
	}

	/// <summary>
	/// Draws the aabb and ceiling, ground and wall sensors .
	/// </summary>
	protected void DrawMovingObjectGizmos()
	{
		//calculate the position of the aabb's center
		var aabbPos = transform.position + (Vector3)mAABBOffset;
		
		//draw the aabb rectangle
		Gizmos.color = Color.yellow;
   		Gizmos.DrawWireCube(aabbPos, mAABB.halfSize*2.0f);
		
		//draw the ground checking sensor
		Vector2 bottomLeft = aabbPos - new Vector3(mAABB.halfSize.x, mAABB.halfSize.y, 0.0f) - Vector3.up + Vector3.right;
		var bottomRight = new Vector2(bottomLeft.x + mAABB.halfSize.x*2.0f - 2.0f, bottomLeft.y);
		
		Gizmos.color = Color.red;
		Gizmos.DrawLine(bottomLeft, bottomRight);
		
		//draw the ceiling checking sensor
		Vector2 topRight = aabbPos + new Vector3(mAABB.halfSize.x, mAABB.halfSize.y, 0.0f) + Vector3.up - Vector3.right;
		var topLeft = new Vector2(topRight.x - mAABB.halfSize.x*2.0f + 2.0f, topRight.y);
		
		Gizmos.color = Color.red;
		Gizmos.DrawLine(topLeft, topRight);
		
		//draw left wall checking sensor
		bottomLeft = aabbPos - new Vector3(mAABB.halfSize.x, mAABB.halfSize.y, 0.0f) - Vector3.right;
		topLeft = bottomLeft;
		topLeft.y += mAABB.halfSize.y * 2.0f;
		
		Gizmos.DrawLine(topLeft, bottomLeft);
		
		//draw right wall checking sensor
		
		bottomRight = aabbPos + new Vector3(mAABB.halfSize.x, -mAABB.halfSize.y, 0.0f) + Vector3.right;
		topRight = bottomRight;
		topRight.y += mAABB.halfSize.y * 2.0f;
		
		Gizmos.DrawLine(topRight, bottomRight);
	}

	/// <summary>
	/// Updates the moving object's physics, integrates the movement, updates sensors for terrain collisions.
	/// </summary>
	public void UpdatePhysics()
	{	
		//assign the previous state of onGround, atCeiling, pushesRightWall, pushesLeftWall
		//before those get recalculated for this frame
		mWasOnGround = mOnGround;
		mPushedRightWall = mPushesRightWall;
		mPushedLeftWall = mPushesLeftWall;
		mWasAtCeiling = mAtCeiling;
		
		mOnOneWayPlatform = false;
		
		//save the speed to oldSpeed vector
		mOldSpeed = mSpeed;
		
		//save the position to the oldPosition vector
		mOldPosition = mPosition;
		
		//integrate the movement only if we're not tweening
		mPosition += mSpeed*Time.deltaTime;

        if (mPosition.y <= 900.0f)
        {
            mPosition.y = 900.0f;
            mOnGround = true;
        }
        else
            mOnGround = false;
		
		//update the aabb
        mAABB.center = mPosition + mAABBOffset;

        //apply the changes to the transform
        mTransform.localScale = new Vector3(mScale.x, mScale.y, 1.0f);
		mTransform.position = new Vector3(Mathf.Round(mPosition.x), Mathf.Round(mPosition.y), -1.0f);
	}
}