/*
 * Decompiled with CFR 0.152.
 */
package org.bouncycastle.tls.crypto.impl;

import java.io.IOException;
import java.security.SecureRandom;
import org.bouncycastle.tls.ProtocolVersion;
import org.bouncycastle.tls.TlsFatalAlert;
import org.bouncycastle.tls.crypto.TlsCipher;
import org.bouncycastle.tls.crypto.TlsCrypto;
import org.bouncycastle.tls.crypto.TlsCryptoParameters;
import org.bouncycastle.tls.crypto.TlsHMAC;
import org.bouncycastle.tls.crypto.impl.TlsBlockCipherImpl;
import org.bouncycastle.tls.crypto.impl.TlsImplUtils;
import org.bouncycastle.tls.crypto.impl.TlsSuiteHMac;
import org.bouncycastle.tls.crypto.impl.TlsSuiteMac;
import org.bouncycastle.util.Arrays;

public class TlsBlockCipher
implements TlsCipher {
    protected TlsCryptoParameters cryptoParams;
    private final TlsCrypto crypto;
    protected byte[] randomData;
    protected boolean useExplicitIV;
    protected boolean encryptThenMAC;
    protected TlsBlockCipherImpl encryptCipher;
    protected TlsBlockCipherImpl decryptCipher;
    protected TlsSuiteMac writeMac;
    protected TlsSuiteMac readMac;

    public TlsBlockCipher(TlsCrypto crypto, TlsCryptoParameters cryptoParams, TlsBlockCipherImpl encryptCipher, TlsBlockCipherImpl decryptCipher, TlsHMAC writeMac, TlsHMAC readMac, int cipherKeySize) throws IOException {
        byte[] server_IV;
        byte[] client_IV;
        this.cryptoParams = cryptoParams;
        this.crypto = crypto;
        this.randomData = crypto.createNonce(256);
        this.useExplicitIV = TlsImplUtils.isTLSv11(cryptoParams);
        this.encryptThenMAC = cryptoParams.getSecurityParameters().isEncryptThenMAC();
        int key_block_size = 2 * cipherKeySize + writeMac.getMacLength() + readMac.getMacLength();
        if (!this.useExplicitIV) {
            key_block_size += encryptCipher.getBlockSize() + decryptCipher.getBlockSize();
        }
        byte[] key_block = TlsImplUtils.calculateKeyBlock(cryptoParams, key_block_size);
        int offset = 0;
        byte[] clientMacKey = Arrays.copyOfRange((byte[])key_block, (int)offset, (int)(offset + writeMac.getMacLength()));
        byte[] serverMacKey = Arrays.copyOfRange((byte[])key_block, (int)(offset += clientMacKey.length), (int)(offset + writeMac.getMacLength()));
        byte[] client_write_key = Arrays.copyOfRange((byte[])key_block, (int)(offset += serverMacKey.length), (int)(offset + cipherKeySize));
        byte[] server_write_key = Arrays.copyOfRange((byte[])key_block, (int)(offset += cipherKeySize), (int)(offset + cipherKeySize));
        offset += cipherKeySize;
        if (this.useExplicitIV) {
            client_IV = new byte[encryptCipher.getBlockSize()];
            server_IV = new byte[encryptCipher.getBlockSize()];
        } else {
            client_IV = Arrays.copyOfRange((byte[])key_block, (int)offset, (int)(offset + encryptCipher.getBlockSize()));
            server_IV = Arrays.copyOfRange((byte[])key_block, (int)(offset += encryptCipher.getBlockSize()), (int)(offset + encryptCipher.getBlockSize()));
            offset += encryptCipher.getBlockSize();
        }
        if (offset != key_block_size) {
            throw new TlsFatalAlert(80);
        }
        this.readMac = new TlsSuiteHMac(cryptoParams, readMac);
        this.writeMac = new TlsSuiteHMac(cryptoParams, writeMac);
        this.encryptCipher = encryptCipher;
        this.decryptCipher = decryptCipher;
        if (cryptoParams.isServer()) {
            this.writeMac.setKey(serverMacKey);
            this.readMac.setKey(clientMacKey);
            this.encryptCipher.setKey(server_write_key);
            this.decryptCipher.setKey(client_write_key);
            this.encryptCipher.init(server_IV);
            this.decryptCipher.init(client_IV);
        } else {
            this.writeMac.setKey(clientMacKey);
            this.readMac.setKey(serverMacKey);
            this.encryptCipher.setKey(client_write_key);
            this.decryptCipher.setKey(server_write_key);
            this.encryptCipher.init(client_IV);
            this.decryptCipher.init(server_IV);
        }
    }

    @Override
    public int getPlaintextLimit(int ciphertextLimit) {
        int blockSize = this.encryptCipher.getBlockSize();
        int macSize = this.writeMac.getSize();
        int plaintextLimit = ciphertextLimit;
        if (this.useExplicitIV) {
            plaintextLimit -= blockSize;
        }
        if (this.encryptThenMAC) {
            plaintextLimit -= macSize;
            plaintextLimit -= plaintextLimit % blockSize;
        } else {
            plaintextLimit -= plaintextLimit % blockSize;
            plaintextLimit -= macSize;
        }
        return --plaintextLimit;
    }

    @Override
    public byte[] encodePlaintext(long seqNo, short type, byte[] plaintext, int offset, int len) throws IOException {
        int blockSize = this.encryptCipher.getBlockSize();
        int macSize = this.writeMac.getSize();
        ProtocolVersion version = this.cryptoParams.getServerVersion();
        int enc_input_length = len;
        if (!this.encryptThenMAC) {
            enc_input_length += macSize;
        }
        int padding_length = blockSize - 1 - enc_input_length % blockSize;
        if (!version.isDTLS() && !version.isSSL()) {
            int maxExtraPadBlocks = (255 - padding_length) / blockSize;
            int actualExtraPadBlocks = this.chooseExtraPadBlocks(this.crypto.getSecureRandom(), maxExtraPadBlocks);
            padding_length += actualExtraPadBlocks * blockSize;
        }
        int totalSize = len + macSize + padding_length + 1;
        if (this.useExplicitIV) {
            totalSize += blockSize;
        }
        byte[] outBuf = new byte[totalSize];
        int outOff = 0;
        if (this.useExplicitIV) {
            byte[] explicitIV = this.crypto.createNonce(blockSize);
            this.encryptCipher.init(explicitIV);
            System.arraycopy(explicitIV, 0, outBuf, outOff, blockSize);
            outOff += blockSize;
        }
        int blocks_start = outOff;
        System.arraycopy(plaintext, offset, outBuf, outOff, len);
        outOff += len;
        if (!this.encryptThenMAC) {
            byte[] mac = this.writeMac.calculateMac(seqNo, type, plaintext, offset, len);
            System.arraycopy(mac, 0, outBuf, outOff, mac.length);
            outOff += mac.length;
        }
        for (int i = 0; i <= padding_length; ++i) {
            outBuf[outOff++] = (byte)padding_length;
        }
        this.encryptCipher.doFinal(outBuf, blocks_start, outOff - blocks_start, outBuf, blocks_start);
        if (this.encryptThenMAC) {
            byte[] mac = this.writeMac.calculateMac(seqNo, type, outBuf, 0, outOff);
            System.arraycopy(mac, 0, outBuf, outOff, mac.length);
            outOff += mac.length;
        }
        return outBuf;
    }

    @Override
    public byte[] decodeCiphertext(long seqNo, short type, byte[] ciphertext, int offset, int len) throws IOException {
        int blockSize = this.decryptCipher.getBlockSize();
        int macSize = this.readMac.getSize();
        int minLen = blockSize;
        minLen = this.encryptThenMAC ? (minLen += macSize) : Math.max(minLen, macSize + 1);
        if (this.useExplicitIV) {
            minLen += blockSize;
        }
        if (len < minLen) {
            throw new TlsFatalAlert(50);
        }
        int blocks_length = len;
        if (this.encryptThenMAC) {
            blocks_length -= macSize;
        }
        if (blocks_length % blockSize != 0) {
            throw new TlsFatalAlert(21);
        }
        if (this.encryptThenMAC) {
            boolean badMac;
            int end = offset + len;
            byte[] receivedMac = Arrays.copyOfRange((byte[])ciphertext, (int)(end - macSize), (int)end);
            byte[] calculatedMac = this.readMac.calculateMac(seqNo, type, ciphertext, offset, len - macSize);
            boolean bl = badMac = !Arrays.constantTimeAreEqual((byte[])calculatedMac, (byte[])receivedMac);
            if (badMac) {
                throw new TlsFatalAlert(20);
            }
        }
        if (this.useExplicitIV) {
            this.decryptCipher.init(Arrays.copyOfRange((byte[])ciphertext, (int)offset, (int)(offset + blockSize)));
            offset += blockSize;
            blocks_length -= blockSize;
        }
        this.decryptCipher.doFinal(ciphertext, offset, blocks_length, ciphertext, offset);
        int totalPad = this.checkPaddingConstantTime(ciphertext, offset, blocks_length, blockSize, this.encryptThenMAC ? 0 : macSize);
        boolean badMac = totalPad == 0;
        int dec_output_length = blocks_length - totalPad;
        if (!this.encryptThenMAC) {
            int macInputLen = dec_output_length -= macSize;
            int macOff = offset + macInputLen;
            byte[] receivedMac = Arrays.copyOfRange((byte[])ciphertext, (int)macOff, (int)(macOff + macSize));
            byte[] calculatedMac = this.readMac.calculateMacConstantTime(seqNo, type, ciphertext, offset, macInputLen, blocks_length - macSize, this.randomData);
            badMac |= !Arrays.constantTimeAreEqual((byte[])calculatedMac, (byte[])receivedMac);
        }
        if (badMac) {
            throw new TlsFatalAlert(20);
        }
        return Arrays.copyOfRange((byte[])ciphertext, (int)offset, (int)(offset + dec_output_length));
    }

    protected int checkPaddingConstantTime(byte[] buf, int off, int len, int blockSize, int macSize) {
        int end = off + len;
        byte lastByte = buf[end - 1];
        int padlen = lastByte & 0xFF;
        int totalPad = padlen + 1;
        int dummyIndex = 0;
        int padDiff = 0;
        if (TlsImplUtils.isSSL(this.cryptoParams) && totalPad > blockSize || macSize + totalPad > len) {
            totalPad = 0;
        } else {
            int padPos = end - totalPad;
            do {
                padDiff = (byte)(padDiff | buf[padPos++] ^ lastByte);
            } while (padPos < end);
            dummyIndex = totalPad;
            if (padDiff != 0) {
                totalPad = 0;
            }
        }
        byte[] dummyPad = this.randomData;
        while (dummyIndex < 256) {
            padDiff = (byte)(padDiff | dummyPad[dummyIndex++] ^ lastByte);
        }
        dummyPad[0] = (byte)(dummyPad[0] ^ padDiff);
        return totalPad;
    }

    protected int chooseExtraPadBlocks(SecureRandom r, int max) {
        int x = r.nextInt();
        int n = this.lowestBitSet(x);
        return Math.min(n, max);
    }

    protected int lowestBitSet(int x) {
        if (x == 0) {
            return 32;
        }
        int n = 0;
        while ((x & 1) == 0) {
            ++n;
            x >>= 1;
        }
        return n;
    }
}

