# -*- coding: utf-8 -*-
import argparse

import lief

KEY_SCHED_FUNC_START = 0x1a10
KEY_SCHED_REV_FUNC_START = 0x1bd6
ENCRYPT_BLOCK_FUNC_START = 0x22c9
DECRYPT_BLOCK_FUNC_START = 0x230a

# this is the constructor function that implements self-ptracing
# it can be found in the INIT array, or e.g. by searching for `syscall` instruction
SELF_PTRACE_CTOR_FUNC_START = 0x137b

if __name__ == '__main__':
    # parse cmd args
    parser = argparse.ArgumentParser()
    parser.add_argument('binary_path', type=str, help='Path to tridea binary')
    parser.add_argument('-o', '--output', type=str,
                        default='libtridea.so', help='Path to output library file, default is "./libtridea.so"')
    args = parser.parse_args()

    # 1. open the binary
    tridea = lief.parse(args.binary_path)

    # 2. export IDEA cipher core functions: `key_sched`, `key_sched_rev`, `encrypt_block`, `decrypt_block`
    tridea.add_exported_function(KEY_SCHED_FUNC_START, 'key_sched')
    tridea.add_exported_function(KEY_SCHED_REV_FUNC_START, 'key_sched_rev')
    tridea.add_exported_function(ENCRYPT_BLOCK_FUNC_START, 'encrypt_block')
    tridea.add_exported_function(DECRYPT_BLOCK_FUNC_START, 'decrypt_block')

    # 3. remove duplicate symbols
    for dynsym_name in ['stdin', 'stdout', 'stderr']:
        dynsym = tridea.get_dynamic_symbol(dynsym_name)
        if dynsym:
            tridea.remove_dynamic_symbol(dynsym)

    # 4. remove the constructor function that calls ptrace via direct syscall
    # N.B. this is only necessary if we want to debug a binary that links this .so we're making
    tridea[lief.ELF.DYNAMIC_TAGS.INIT_ARRAY].remove(SELF_PTRACE_CTOR_FUNC_START)

    # 5. remove PIE flag so that the output .so is treated an LSB shared object, not an LSB pie executable
    # see also https://lief.re//doc/latest/tutorials/08_elf_bin2lib.html#warning-for-glibc-2-29-users
    tridea[lief.ELF.DYNAMIC_TAGS.FLAGS_1].remove(lief.ELF.DYNAMIC_FLAGS_1.PIE)

    # 6. save the patched binary as a shared library
    tridea.write(args.output)

    pass
